clear;close all; clc;
addpath('lib');

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This program provides the following three results
%  1. impulse response and its confidence intervals
%  2. test for identification condition
%  3. overidentifying restrictions test
% under the setup of FAVAR identified through heteroskedasticity
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 0. Setting
% 0-1. Parameters
p                    = 1; 
r1                   = 3;  
r2                   = 1;  
r                    = r1+r2;

shock                = [zeros(1,r1) 1];
uhet                 = 1;
hmax                 = 4;
nrep                 = 499;
c                    = 0.1;

shocksize = 0.25; % 25 bps

% random number generation
rng('default'); rng(2376421);

% 0-2. Data
datfile = 'data/DataUSFin';
outfile = 'data/DataUS';
labels = {'MPEvents','EuroDollar','StockPrices','TreasuryYields','CBYields','Forex','ComPrices'};
transf = [0 1 2 1 1 2 2];

% full sample
sdate = [1994  1  3];
edate = [2019 12 11];

YVars  = 'ED03'; 
XVars  = {'SP500','NASDAQ','DJIA','WILS5000','RUS2000','TR6M','TR01','TR02','TR05','TR10','TR30','CBAAA','CBBAA','USDEUR','USDJPY','USDGBP','USDCAD','USDAUD','WTI','GSCIOIL','GSCIGOLD'};

% -------------
% naming variables
ZVars  = {'TR02','SP500','USDEUR'}; 

AllVars = [XVars,YVars];

% named factor nomalization (choose r variables) Assumption D
[~,rindY] = ismember(YVars,AllVars);
[~,rindZ] = ismember(ZVars,AllVars);

% index of naming variables including the observed factor
rind     = [rindZ,rindY]'; 
% -------------

% ================================================================
% results will be saved at dir_results
dir_results = 'results/';
baseline =[dir_results,'results.mat']; 
logfile = [dir_results,'log.txt'];

disp([' * save results to: ',dir_results]);
mkdir(dir_results);
delete(logfile); diary(logfile);

disp([' * Observed factor: ',YVars]);
disp(' ');
disp(' * Naming variables');
disp(char(ZVars));
disp(' ');
disp(' * Variables');
disp(char(XVars));
disp(' ');

% ================================================================
DATA = crdataset(datfile,labels,transf,sdate,edate);
save(outfile, 'DATA');

% ================================================================
Vars  = DATA.Names;
Data  = DATA.Data;
Dates = DATA.Dates;
DatesAD = DATA.DatesAD; % announcement dates

DatesND_RS = DATA.DatesND_RS; % 1 day before the announcement dates
DatesND = DATA.DatesND; % non-announcement dates

[~,locX]             = ismember(XVars,Vars);
[~,locY]             = ismember(YVars,Vars);
X                    = Data(:,locX); 
Y                    = Data(:,locY); 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%  'regime' is a T by 1 vector; whose elements take
%%%   1 if in the largest variance regime (AD)
%%%   2 if in the smallest variance regime (ND)
%%%   3 if in the other regime (You can also use 4,5,...)
regimeAD             = ismember(datetime(Dates),datetime(DatesAD));  
regimeND_RS          = ismember(datetime(Dates),datetime(DatesND_RS));  
regimeND             = ismember(datetime(Dates),datetime(DatesND));  
regime0              = [regimeAD,regimeND_RS,regimeND];
regime               = zeros(size(regime0,1),1);

for t=1:size(regime,1)
   if regimeND_RS(t,1)==1 % regime 1: one day prior to the event
      regime(t,1)   = 1; 
   elseif regimeAD(t,1)==1 % regime 2: event-day
      regime(t,1)   = 2; 
   else
      regime(t,1)   = 3;
   end
end

[T,N]                = size(X);

stdX = std(X);
stdY = std(Y);
adj = shocksize/stdY;

% standardization
X = (X-mean(X))./stdX; 
Y = (Y-mean(Y))./stdY;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 1. Impulse responses
% 1-1. Estimation

disp(' # step 1');
[F,L1,u,V,L2,Z]      = pcestim(X,Y,r2,r1);
H                    = [F Y];

L12                  = [L1 L2];

Ly = [zeros(1,r1) 1]; % loading associated with observed factor
L  = [L12;Ly];
    
scale                = ones(r,1);
for k=1:r
  scale(k,1)         = L(rind(k,1),k);
end
Cr                   = L(rind,:); 

disp(' # step 2');
tic;
[A,e]                = varestim_gls2(H,p,regime);
e1                   = [];
e2                   = [];
e3                   = [];

regime1 = regime(1+p:end,:);
for t=1:T-p
   if regime1(t,1)==1
      e1             = [e1;e(t,:)];
   elseif regime1(t,1)==2
      e2             = [e2;e(t,:)];
   else
      e3             = [e3;e(t,:)];
   end
end

T1                   = size(e1,1);
T2                   = size(e2,1);
T3                   = size(e3,1);
Sig1                 = e1'*e1/T1;
Sig2                 = e2'*e2/T2;
Sig3                 = e3'*e3/T3;
toc;

disp(' # step 3');
S                    = Cr*Sig1/Sig2/Cr;

[Evec0, Eval0]       = eig(S);
[Eval,ord]           = sort(diag(Eval0),'descend');
Evec                 = Evec0(:,ord);
D                    = zeros(r,r);
for i=1:r
   D(:,i)            = Evec(:,i)./Evec(i,i)*scale(i,1);
end

disp(' # step 4');
B                   = Cr\D;

irfh                = zeros(hmax,N);
for xind=1:N
    [irfh(:,xind),irfh_Y] = irfgen(shock,A,B,L1,L2,xind,hmax,p,r1,r2);
end

a_irfh   = irfh.*adj.*stdX;
a_irfh_Y = irfh_Y(1:hmax,end)*adj*stdY;

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Test for identification condition
[Q,pvQ] = test_ident(e1,e2,r);
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Structural shocks on monetary policy event days
epsAD = (B\e2')';
epsADcov  = cov(epsAD);

% Structural shocks one day prior to MP events
epsND = (B\e1')';
epsNDcov  = cov(epsND);

% 1-2. Confidence interval
ucla                 = zeros(hmax,N);
lcla                 = zeros(hmax,N);
uclb                 = zeros(hmax,N);
lclb                 = zeros(hmax,N);
pivot                = zeros(hmax,nrep,N);

% 1-2-1. Asymptotic inference
DL                   = duplication(r);
PD                   = DL*inv(DL'*DL)*DL';

disp('  ## Lemma A2');

u4 = [u,zeros(T,1)]; % residual for YVAR equation (= identity equation)
if uhet==0
  VC                 = kron(inv(H'*H/T),(u4(:,rind)'*u4(:,rind)/T));
elseif uhet==1
  vtemp              = zeros(T,r^2);
  for t=1:T
    vtemp(t,:)       = kron(H(t,:)',u4(t,rind)');
  end
  VC                 = kron(inv(H'*H/T),eye(r))*(vtemp'*vtemp/T)*kron(inv(H'*H/T),eye(r));
end

disp('  ## Lemma A3');
VSig1                = 2*PD*kron(Sig1,Sig1)*PD;
VSig2                = 2*PD*kron(Sig2,Sig2)*PD; 
Ds1                  = kron(inv(Sig2'),eye(r));
Ds2                  = kron(inv(Sig2'),Sig1*inv(Sig2));
VSig                 = Ds1*VSig1*Ds1'+Ds2*VSig2*Ds2'; 
 
disp('  ## Lemma A4');
Ds                   = kron(inv(Cr'),Cr); 
Dc1                  = kron(inv(Cr'),Cr*S*inv(Cr));
Dc2                  = kron(inv(Cr')*S',eye(r));
VS                   = Ds*VSig*Ds'+Dc1*VC*Dc1'+Dc2*VC*Dc2';
 
disp('  ## Lemma A5');
VD                   = zeros(r^2,r^2);
Cstar                = Cr*B;
for k1=1:r
 for k2=1:r
   if k1==k2 
      Vd             = zeros(r,r);
      for i=1:r
       if i==k1
          Vd         = Vd+zeros(r,r);
       else
          cstar      = Cstar(:,i);
          vuii       = Eval(k1,1)*Eval(i,1); 
          Vd         = Vd+vuii/((Eval(k1,1)-Eval(i,1))^2)*(cstar*cstar');
       end
      end
    else 
     cstar1          = Cstar(:,k1); 
     cstar2          = Cstar(:,k2); 
     vuij            = Eval(k1,1)*Eval(k2,1); 
     Vd              = -vuij/((Eval(k1,1)-Eval(k2,1))^2)*(cstar1*cstar2'); 
    end
  VD(r*(k1-1)+1:r*(k1-1)+r,r*(k2-1)+1:r*(k2-1)+r) = Vd; 
 end
end
   
disp('  ## Lemma 1');
tic;
Dd                    = kron(eye(r),inv(Cr));
Dc                    = kron(D'*inv(Cr'),inv(Cr));
VB                    = Dd*VD*Dd'+Dc*VC*Dc'; 

irfhf = irfh;

toc;   

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Overidentifying restrictions test
[J,pvJ] = test_overident(B,VB,T1+T2,r1,r2);
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% 1-2-2. Bootstrap inference
tic;

for rep=1:nrep

    tS = tic;
    disp(rep);
    H0b               = bootvar2(T,A,p,H,e,regime(p+1:end,1));
    F0b               = H0b(:,1:r1);          
    Xb                = bootfm(L12',H0b,u,uhet);           
    Gb                = H0b(:,r1+1:end);

    [Fb,L1b,ub,Vb,L2b]= pcestim(Xb,Gb,r2,r1);

    Lb12              = [L1b L2b];
    Lb                = [Lb12;Ly];
    
    Hb                = [Fb Gb];

    % re-scaling
    scaleb            = ones(r,1);
    for k=1:r
      scaleb(k,1)      = Lb(rind(k,1),k);
    end

    Cb                = Lb(rind,:); 

    [Ab,eb]           = varestim_gls2(Hb,p,regime);   
    eb1               = [];
    eb2               = [];

    for t=1:T-p
     if regime1(t,1)==1
        eb1           = [eb1;eb(t,:)];
     elseif regime1(t,1)==2
        eb2           = [eb2;eb(t,:)];
     end
    end

    Sigb1             = eb1'*eb1/T1;
    Sigb2             = eb2'*eb2/T2; 
    Sb                = Cb*Sigb1/Sigb2/Cb;
 
    [Evecb0, Evalb0]  = eig(Sb);
    [Evalb,ordb]      = sort(diag(Evalb0),'descend');
    Evecb             = Evecb0(:,ordb);
  
    Db                = zeros(r,r);
    
    % re-scaling
    for i=1:r
      Db(:,i)         = Evecb(:,i)./Evecb(i,i)*scaleb(i,1);
    end

    Bb                = Cb\Db;

    for xind=1:N
        irfb              = irfgen(shock,Ab,Bb,L1b,L2b,xind,hmax,p,r1,r2);  
        pivot(:,rep,xind)      = irfb -irfhf(:,xind);
    end
    
    tE = toc(tS);

end
toc;

for xind=1:N
    lclb(:,xind)          = irfhf(:,xind)-quantile(pivot(:,:,xind)',1-c/2)';
    uclb(:,xind)          = irfhf(:,xind)-quantile(pivot(:,:,xind)',c/2)';
end

a_lclb = lclb.*adj.*stdX;
a_uclb = uclb.*adj.*stdX;

save(baseline);

diary off;


