/* AUTOCOR.g computes specified autocorrelations for each column of a matrix.
              The data are assumed to have 0 mean. Thus, use x = x-meanc(x)'
              prior to the use of this function if mean is not 0.

   a = autocor(x,F,L);

INPUTS: x -- nxk matrix, for which autocorrelations are to be computed for
             each column separately. Assumed to have 0 mean.
        F -- 1x1 scalar in range [0,rows(x)-1], denoting the first auto-
             correlation to compute.
        L -- 1x1 scalar in range [0,rows(x)-1], denoting the last auto-
             correlation to compute. It must be that: F <= L; if L = 0
             L is set to rows(x)-1 and all autocorrelations from F to L
             are computed. If L = 0 and F < 0, then only the 0th order
             autocorrelation is computed (this equals x'x).
OUTPUT: c -- gxk matrix, where g = L - F + 1, containing the autocorrelations
             of order F, F+1, ..., L for each of the columns of x. If the
             variance of any variable is 0, missings will be returned for that
             variable.
Remarks:     The 0th autocorrelation will always be 1.
EXAMPLE:     autocor(x,2,8)  -- will compute the 2nd through 8th order
                                autocorrelations.
SEE ALSO:    autocor, conv,
----------------------------------------------------------------------------*/
proc autocor(x,F,L);
 local n, acov, var;
    n = rows(x);
    if L > (N - 1) or F > L;
"ERROR: Illegal values specified for starting and/or ending autocorrelations.";
     end;
    endif;

    if L == 0 and F == 0; L = n - 1;
    elseif L == 0 and F < 0; F = 0;
    endif;
  acov = rev( conv(x, rev(x), n - L, n - F ) );
  if F > 0;
   var = sumc( (x.*x) )';   /* compute variances, if necessary */
  else;
   var = acov[1,.];
  endif;
  if not(var /= 0);
      var = miss(var,0);  /* return all missings for variable if 0 variance */
  endif;

  retp( acov./var );
endp;
/*--------------------------------------------------------------------------*/
