/*
 * probit.cpp
 *
 *  Created on: Jun 5, 2013
 *      Author: Aico van Vuuren
 */

#ifndef PROBIT_INCLUDED
#define PROBIT_INCLUDED

using namespace std;
#include <iostream>
#include "maximize.cpp"
#include "NormalDistribution.cpp"
#include <mpi.h>
#include <stdlib.h>

class Probit : public Maximize {
	private : double total;
	private : double Lambda_H;
	double** H1_hulp;
	double** H2_hulp;
	double ** X;
	double* Beta;
	double* G1;
	int n, k;
	private : NormalDistribution* normal;

	int myid, numprocs, increment;
	MPI_Status status;
	int Min, Max;

	public : Probit(double** H2_totaal, double** H4_totaal, double** X, double* G1, double* Beta, double Lambda_H, int n, int k, int cp,
				int myid, int numprocs, MPI_Status status, int Min, int Max)
	{
		this->Lambda_H = Lambda_H;
		this->X = X;
		this->n = n;
		this->k = k;
		this->cp = cp;
		this->myid = myid;
		this->numprocs = numprocs;
		this->status = status;
		increment = n/numprocs;
		this->Min = Min;
		this->Max = Max;

		this->G1 = G1;
		total = 0;

		normal = new NormalDistribution();
		this->Beta = (double*)malloc(sizeof(double) * cp);
		memcpy(this->Beta,Beta, cp * sizeof(double)); 
		int i;
		H1_hulp = (double**)malloc(sizeof(double*) * n);
		H2_hulp = (double**)malloc(sizeof(double*) * n);

		for(i=Min;i<=Max;i++)
		{
			int j;
			double* h1_hulp = (double*)malloc(sizeof(double) * k);
			double* h2_hulp = (double*)malloc(sizeof(double) * k);
			for(j=0;j<k;j++)
			{
				if (!(H2_totaal[i][j] == -1))
				{
					double hulp = exp(H2_totaal[i][j] / Lambda_H);
					h1_hulp[j] = hulp / (1 + hulp);
				}
				else
					h1_hulp[j] = -1;
				if (!(H4_totaal[i][j] == -1))
				{
					double hulp = exp(H4_totaal[i][j] / Lambda_H);
					h2_hulp[j] = hulp / (1 + hulp);
				}
				else
					h2_hulp[j] = -1;
			}
			H1_hulp[i] = h1_hulp;
			H2_hulp[i] = h2_hulp;
			
			
		}
		
		normal = new NormalDistribution(0.,1.);
	}

	~Probit()
	{
		free(Beta);
		int i;
		for(i=Min;i<=Max;i++)
		{
			free(H1_hulp[i]);
			free(H2_hulp[i]);
		}
		free(H1_hulp);
		free(H2_hulp);
	}

	public: virtual double Function(double* variable_list) throw(RuntimeException)
	{
		double total = 0;
		int i;

		double adfunc0 = FunctionThread(variable_list, Min, Max);
		int tag = 7;

		if (myid == 0)
		{
		  	total = adfunc0;

		 	for(i=1;i<numprocs;i++)
		   	{
		 		double adfunc1;

		 		int rc = MPI_Recv(&adfunc1, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD,
		 								&status);
		   		total += adfunc1;
		   	}

	    	for(i=1;i<numprocs;i++)
	    	{
	    		double hulp = total;
	    		MPI_Send(&hulp, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
		    }
		 }
		 else
		 {
			 MPI_Send(&adfunc0, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD);

			 double hulp1;
			 int rc = MPI_Recv(&hulp1, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD,
			 		 								&status);
			 total = hulp1;
		}

		return total / n;
	}
	
	
	public: virtual double* dFunction(double* variable_list) throw(RuntimeException)
	{
		double* total;
		int i;

		double* avscore0 = dFunctionThread(variable_list, Min, Max);
		total = avscore0;
		int tag = 7;

		if (myid == 0)
		{
		 	for(i=1;i<numprocs;i++)
		   	{
		 		double* avscore1 = (double*)malloc(sizeof(double) * cp);

				int rc = MPI_Recv(avscore1, cp, MPI_DOUBLE, i, tag, MPI_COMM_WORLD,
			 								&status);
		   		add_to(total, avscore1);
		   		free(avscore1);
		   	}

		   	for(i=1;i<numprocs;i++)
		   	{
		   		MPI_Send(total, cp, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
		    }
		 }
		 else
		 {
			 MPI_Send(avscore0, cp, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD);

			 double* hulp1 = (double*)malloc(sizeof(double) * cp);
			 int rc = MPI_Recv(hulp1, cp, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD,
				 		 								&status);

			free(total);
			 total = hulp1;

		}


		multiply_to(total, 1. / (double)(n));
		return total;
	}

	public: double FunctionThread(double* variable_list, int min, int max) throw(RuntimeException) {
		Initialize(variable_list);
	  	double result = 0;
		int i;

		double* x;

	  	for(i=min;i<=max;i++)
		{
	  		x = X[i];
			
	  		double hulp = FLoglik10(H1_hulp[i], H2_hulp[i], x, G1[i]);
	  		result += (hulp);
	  	}

	    return result;
	}
	
	public: double* dFunctionThread(double* variable_list, int min, int max) throw(RuntimeException) {
		Initialize(variable_list);
	  	double* result = zeros(cp);
		int i;

		double* x;

	  	for(i=min;i<=max;i++)
		{
	  		x = X[i];
			
	  		double* hulp = dFLoglik10(H1_hulp[i], H2_hulp[i], x, G1[i]);
	  		add_to(result, hulp);
	  		free(hulp);
	  	}

	    return result;
	}

	private : double FLoglik10(double* H1, double* H2, double* x1, double g)
	{
		int i;
		double result = 1.;
		double hulp = internal_product(x1, Beta, cp);
		
	//	cout << x1[0] * Beta[0] << "\t" << x1[1] * Beta[1] << "\t" << hulp << endl;

		double hulp1 = normal->cumulativeProbability(hulp);
		for(i=0;i<k;i++)
		{
			if (!(H1[i] == -1))
			{
				double h2 = H1[i];
				//cout << hulp1 << endl;
				double expression1 = (h2 * log(hulp1));
				double expression2 = ((1-h2) * log(1-hulp1));

				result +=  expression1 + expression2;
			}
			if (!(H2[i] == -1))
			{
				double h2 = H2[i];
				double expression1 = (h2 * log(hulp1));
				double expression2 = ((1-h2) * log(1-hulp1));

				result +=  expression1 + expression2;
			}
		}
		
		return result;
	}

	private : double* dFLoglik10(double* H1, double* H2, double* x1, double g)
		{
			
			int i;
			double* result = zeros(cp);
			double hulp = internal_product(x1, Beta, cp);
			
		//	cout << x1[0] * Beta[0] << "\t" << x1[1] * Beta[1] << "\t" << hulp << endl;

			double hulp1 = normal->cumulativeProbability(hulp);
			double hulp2 = normal->Density(hulp);
			for(i=0;i<k;i++)
			{
				if (!(H1[i] == -1))
				{
					double h2 = H1[i];
					double expression1 = (h2 * hulp2 / (hulp1));
					double expression2 = -((1-h2) * hulp2 / (1-hulp1));

					double* hulp_x = (double*)malloc(sizeof(double) * cp);
					int j;
					for(j=0;j<cp;j++)
						hulp_x[j] =  (expression1 + expression2) * x1[j];
					add_to(result, hulp_x);
					free(hulp_x);
				}
			
				if (!(H2[i] == -1))
				{
					double h2 = H2[i];
					double expression1 = (h2 * hulp2 / (hulp1));
					double expression2 = -((1-h2) * hulp2 / (1-hulp1));

					double* hulp_x = (double*)malloc(sizeof(double) * cp);
					int j;
					for(j=0;j<cp;j++)
						hulp_x[j] =  (expression1 + expression2) * x1[j];
					add_to(result, hulp_x);
					free(hulp_x);
				}
			}
			return result;
		}

	private : void Initialize(double* variable_list)
	{
		memcpy(Beta, variable_list, sizeof(double) * cp);
	}
};
#endif






