/*
 * maximize.cpp
 *
 *  Created on: May 28, 2013
 *      Author: Aico van Vuuren
 * Implementation of Simulated Annealing -- Based on Original Ox code of Charles Bos.
 */

#ifndef MAXSAN_INCLUDED
#define MAXSAN_INCLUDED


#include <iostream>
#include <string>
#include <iomanip>
#include <stdlib.h>
#include <string.h>
#include <vector>
#include "RuntimeException.cpp"
#include "retval.cpp"
#include <cmath>
#include <random>

using namespace std;

class Maximize_SAN {	  
	protected :  int s_mxEval;
	protected :  double s_dEps;
	protected : int s_iPrint;      /* print results every s_iPrint'th iteration */
	protected : int s_iNEps;
	protected : int s_iNS;
	protected : double s_iNT;
	protected : int s_vC;
	protected : int s_vM;
	protected : double s_dRT;
	protected : bool s_bInit;
	protected : double* avP;
	protected : double* vM;
	protected : double adFunc, adStep;
	//protected : RetVal ret_val;
	protected : int cp_SAN;
	private : bool limits_hi_set, limits_lo_set, vm_set;

	public : double* vHi;
	public : double* vLo;
	random_device rd;
	mt19937* mt;
	uniform_real_distribution<double>* dist;

	protected : Maximize_SAN()
	{
		limits_hi_set = false;
		limits_lo_set = false;
		vm_set = false;
		adFunc = 0;
		this->cp_SAN = 0;
		avP = NULL;
		s_mxEval = 1e5;
		s_dEps = 1e-6;
		s_iNEps= 4;        /* convergence criteria */
		s_iPrint = 0;                    /* print results */
		s_iNS= 20;
		s_iNT= log(-1);
		s_vC = 2;
		s_vM= 1;
		s_dRT= 0.5;
		s_bInit= false;
		double s_Hi = std::numeric_limits<double>::infinity();
		double s_Lo = -std::numeric_limits<double>::infinity();
		mt = new mt19937(642012511);

	}


	public: virtual double Function(double* variable_list) throw(RuntimeException) = 0;

	public: virtual ~Maximize_SAN(){
	}
	
	public : void Set_Print_SAN(int d)
	{
		s_iPrint = d;
	}

	public : void Set_s_mxEval(int d)
	{
		s_mxEval = d;
	}

	public : void Set_s_dEps(double eps)
	{
		s_dEps = eps;
	}

	public : void Set_s_iNEps(int eps)
	{
		s_iNEps = eps;
	}

	public : void Set_s_dRT(double drt)
	{
		s_dRT = drt;
	}

	public : void Set_s_Hi(double Hi)
	{
		vHi = new double[cp_SAN];
		limits_hi_set = true;
		  fill(vHi,  vHi + cp_SAN, std::numeric_limits<double>::infinity());
	}

	public : void Set_s_Lo(double Lo)
	{
		vLo = new double[cp_SAN];
		limits_lo_set = true;
		  fill(vLo,  vLo + cp_SAN, -std::numeric_limits<double>::infinity());
	}

	public : void Set_s_Hi(double* Hi)
	{
		vHi = new double[cp_SAN];
		limits_hi_set = true;
		 memcpy(vHi,  Hi, cp_SAN * sizeof(double));
	}

	public : void Set_vM(double* vm)
	{
		vM = new double[cp_SAN];
		vm_set = true;

		memcpy(vM,  vm, cp_SAN * sizeof(double));
	}

	public : void Set_s_Lo(double* Lo)
	{
		vLo = new double[cp_SAN];
		limits_lo_set = true;
		 memcpy(vLo,  Lo, cp_SAN * sizeof(double));
	}
	public : double max(double x, double y)
	{
		return y>x?y:x;
	}

	public : double* MaxSA(double* avP, double* adT)
	{
//	  double ir, inAcc, inObds, inFnEv, j, h, m, dFOpt, vXOpt, vFStar,
//	       vnAcp, nUp, nRej, nNew, nDown, vX, vXP, dF, dFP, dU, dP, iN,
	//       vRatio, bRep, iNT, vM, vLo, vHi, va, nBounds;
	  	int iN= cp_SAN;

		double* vX = new double[cp_SAN];
		double* vXP = new double[cp_SAN];
		double* vXOpt = new double[cp_SAN];


		double* vFStar = new double[s_iNEps];

		if (!limits_lo_set)
		{
			vLo = new double[cp_SAN];
			fill(vLo,  vLo + cp_SAN, -std::numeric_limits<double>::infinity());
		}

		if (!limits_hi_set)
		{
			vHi = new double[cp_SAN];
			fill(vHi,  vHi + cp_SAN, std::numeric_limits<double>::infinity());
		}

		double dFOpt;
		double dFP;
		int inAcc = 0;
		int inObds= 0;
		int inFnEv= 0;

		memcpy(vX, avP, cp_SAN * sizeof(double));
		memcpy(vXOpt, avP, cp_SAN * sizeof(double));

	  int iNT= 5;
	  if (!vm_set)
	  {
		  vM = new double[cp_SAN];
		  fill(vM, vM+cp_SAN, s_vM);
	  }


	  fill(vFStar, vFStar + s_iNEps, -std::numeric_limits<double>::infinity());

	  double dF;
	  try {
		  dF= Function(vX);
	  }
	  catch (RuntimeException &e)
	  {
		  throw RuntimeException("Function evaluation failed");
	  }

	  ++inFnEv;

	  if (adT[0] <= 0)
	    throw RuntimeException("No convergence (initial temperature negative)");


	  if (!s_bInit)
	    cout << "Warning: Use MaxSAControlStep to init step-parameters;\n" <<
	             "         default values may be suboptimal.";
	  if (adT[0] < 1)
	    cout  << "Warning: Low initial temperature, may not leave sufficient room\n" <<
	             "         for exploring the full parameter space";
	  if (s_dRT > 1)
	    cout << "Warning: Temperature reduction factor >= 1, no temperature reduction";

	  if (s_iPrint > 0)
	    cout << "Initial result " << dF << "%r" << "at parameters" << vX;

	  dFOpt = dF;
	  vFStar[0] = dF;

	  double dU, dP;

	  bool bRep= true;
	  int nBounds= 0;
	  uniform_real_distribution<double> dist(0,1);

	  bool stoppen1 = false;

	  while (bRep)
	    {
	      // println ("r: nt= ", iNT, " ", inFnEv, " ", s_mxEval);
	      int nUp=0;
	      int nRej= 0;
	      int nNew= 0;
	      int nDown= 0;
	      for (int m= 0; m < iNT; ++m)
	      {
	          double* vnAcp= Maximize::zeros(cp_SAN);

	          for (int j= 0; j < s_iNS; ++j)
	            for (int h= 0; h < iN; ++h)
	            {
	            	memcpy(vXP, vX, cp_SAN * sizeof(double));

	            	bool stoppen = 0;
	            	while(stoppen == 0)
	            	{
	            		double u = dist(mt[0]);

	            		vXP[h]= vX[h] + (2*u-1)*vM[h];

	            		if ((vXP[h] < vLo[h]) || (vXP[h] > vHi[h]))
	            		{
	            			double u1 = dist(mt[0]);
	            			  vXP[h]= vLo[h] + (vHi[h] - vLo[h])*u1;
	            			   ++nBounds;
	            		}
	            		try {
	            			dFP = Function(vXP);
	            			stoppen = 1;
	            		}
	            		catch (RuntimeException &e)
	            		{

	            		}
	            		 ++inFnEv;

	                     if (inFnEv > s_mxEval)
	            	     {
	            	           cout << "Error: Too many function evaluations" << endl;
	            	           throw RuntimeException("Error: Too many function evaluations");
	            	     }
	            	}

	            	if (s_iPrint > 0)
	            		cout << m << "\t" << j << "\t"<< h << "\t" << iNT <<"\t"<< dFP << "\t" << dFOpt << endl;



	                if (dFP >= dF)
	                  {
	                    memcpy(vX, vXP, cp_SAN * sizeof(double));
	                    dF= dFP;
	                    ++inAcc;
	                    ++vnAcp[h];
	                    ++nUp;

	                    if (dFP > dFOpt)
	                      {
	                        memcpy(vXOpt, vXP, cp_SAN * sizeof(double));
	                        dFOpt= dFP;
	                        ++nNew;

	                        if (s_iPrint > 2)
	                          cout << "\nNew optimum of value " << dFP;
	                      }
	                  }
	                else
	                  {
	                    dP= exp((dFP-dF)/adT[0]);
	                    dU= dist(mt[0]);

	                    if (dU < dP)
	                      { // Accept
	                    	 memcpy(vX, vXP, cp_SAN * sizeof(double));
	                        dF= dFP;
	                        ++inAcc;
	                        ++vnAcp[h];
	                        ++nDown;
	                      }
	                    else
	                    {
	                      ++nRej;
	                    }
	                  }


	              }



	          double* vRatio = new double[cp_SAN];
	          for(int i=0;i<cp_SAN;i++)
	        	  vRatio[i] = vnAcp[i] / s_iNS;
	          for(int i=0;i<cp_SAN;i++)
	        	  vM[i] = vRatio[i] > .6 ? vM[i] *(1 + s_vC * (vRatio[i]-.6) / .4) :
	              vRatio[i] < .4 ? vM[i] /(1 + s_vC * (.4-vRatio[i]) / .4) :
	              vM[i];

	          for(int i=0;i<cp_SAN;i++)
	        	  vM[i] = vM[i] < vHi[i] - vLo[i]? vM[i]: vHi[i] - vLo[i];


	          if (s_iPrint >= 1)
	          {
	            cout << "\nIntermediate results after step length adjustment" << endl;
	            cout << "vM:\t" << endl;
	            Maximize::print_vector(vM, cp_SAN);
	            cout << "vX:\t" << endl;
	            Maximize::print_vector(vX, cp_SAN);
	            cout << "vXOpt:\t" << endl;
	            Maximize::print_vector(vXOpt, cp_SAN);
	          }
	      }

	      vFStar[0]= dF;
	      bRep= (dFOpt - vFStar[0] > s_dEps);

	      for(int i=0;i<s_iNEps;i++)
	    	  bRep = !(fabs(dF - vFStar[i]) <= s_dEps)? true : bRep;

	      memcpy(vX, vXOpt, cp_SAN * sizeof(double));
	      dF= dFOpt;


	      double* hulp = vFStar;
	      vFStar = new double[s_iNEps];
	      vFStar[0] = NAN;

	      for(int i=0;i<s_iNEps;i++)
	    	  vFStar[i+1] = hulp[i];

	      delete [] hulp;
	      // Adapt temperature
	      if (bRep)
	        {

	          if (s_iPrint > 0)
	          {
	            cout << "\nIntermediate results before next temperature reduction" << endl;

	                cout << "Current temperature:\t" << adT[0] << endl;
	                cout << "Max func:\t" << dFOpt << endl;
	                cout << "Total moves:\t" << (nUp+nDown+nRej) << endl;

	                 cout << "Improvements:\t" << nUp << endl;
	                 cout << "Accepted deteriorations:\t" << nDown << endl;
	                 cout << "Rejected deteriorations:\t" << nRej << endl;

	                 cout << "New maxima:\t" << nNew << endl;
	                 cout << "Out of bounds:\t" << nBounds << endl;
	          }
	          adT[0]*= s_dRT;
	        }
	    }

	  if (s_iPrint > 0)
	    cout << "Function evaluations: " << inFnEv;

	  memcpy(avP, vXOpt, cp_SAN * sizeof(double));

	  adFunc = dFOpt;

	  return avP;
	}

};


#endif




