/*
 * maximize.cpp
 *
 *  Created on: May 28, 2013
 *      Author: Aico van Vuuren
 */

#ifndef MAXIMIZE1_INCLUDED
#define MAXIMIZE1_INCLUDED


#include <iostream>
#include <string>
#include <iomanip>
#include <stdlib.h>
#include <string.h>
#include <vector>
#include "RuntimeException.cpp"
#include "retval.cpp"
#include <cmath>
#if _MSC_VER > 1500
#define isinf(x) (!_finite(x))
#define isnan(x) (_isnan(x))
#endif


using namespace std;

class Maximize1 {
	//public : enum RetVal {MAX_CONV, MAX_NOCONV, MAX_WEAK_CONV, MAX_MAXIT, MAX_LINE_FAIL, MAX_FUNC_FAIL, NOT_EVAL};
	protected :  int s_mxIter;
	protected :  double s_dEps1;
	protected : double s_dEps2;         /* convergence criteria */
	protected : int s_iPrint;      /* print results every s_iPrint'th iteration */
	protected  : bool s_bCompact;     /* TRUE: print iterations in compact format */
	protected : double s_dStepLinearGap;		/* smallest gap for line search */
	protected : double SQRT_EPS;        /* appr. square root of machine precision */
	protected : double DIFF_EPS; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/2 */
	public : double  DIFF_EPS1; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/3 */
	protected:  double  DIFF_EPS2; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/4 */
	protected : double* avP;
	protected : double adFunc, adStep;
	protected : RetVal ret_val;
	protected : int cp;
	protected : bool check_convergence;
	protected : bool minvhess_set;
	protected : double** minvhess;
	private : double** minvhess_1;
	unsigned int size;
	public : int N;

	protected : Maximize1()
	{
		adFunc = 0;
		this->cp = 0;
		avP = NULL;
		ret_val = NOT_EVAL;
		s_mxIter = 1000;
		s_dEps1 = 1e-4;
		s_dEps2 = 5e-3;         /* convergence criteria */
		s_iPrint = 1;      /* print results every s_iPrint'th iteration */
		s_bCompact = false;     /* TRUE: print iterations in compact format */
		s_dStepLinearGap = 1E-14;		/* smallest gap for line search */
		SQRT_EPS =1E-8;        /* appr. square root of machine precision */
		DIFF_EPS =1E-8; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/2 */
		DIFF_EPS1=5E-6; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/3 */
		DIFF_EPS2=1E-4; /* Rice's formula: log(DIFF_EPS)=log(MACH_EPS)/4 */
		check_convergence = true;
		minvhess_set = false;
	}


	public: virtual double Function(double* variable_list) throw(RuntimeException) = 0;
	public: virtual ~Maximize1(){
	}
	
	public : void Set_MinvHess(double** minvhess) {
		minvhess_1 = minvhess;
		minvhess_set = true;
	}

	public : void Set_Print(int d)
	{
		s_iPrint = d;
	}
	
	public : double* Get_dsteplen()
	{
		double* result = (double*)malloc(s_mxIter * sizeof(double));
		return result;
	}
	
	public : void Set_Linear_Step(double s_dStepLinearGap)
	{
		this->s_dStepLinearGap = s_dStepLinearGap;
	}

	public : void Set_s_mxIter(int s_mxIter)
	{
		this->s_mxIter = s_mxIter;
	}


	public : void Set_Check_Convergence(bool check_convergence)
	{
		this->check_convergence = check_convergence;
	}

	public : void Set_MaxControl(double s_dEps_1, double s_dEps_2)
	{
		this->s_dEps1 = s_dEps_1;
		this->s_dEps2 = s_dEps_2;
	}

	public : RetVal Get_ReturnValue()
	{
		return ret_val;
	}
	
	public : double** Get_MinvHess()
	{
		return minvhess_1;
	}

	public : double* GetParameters()
	{
		return avP;
	}

	public : double dFiniteDiff0(double x)
	{
	    return max( (fabs(x) + SQRT_EPS) * SQRT_EPS, DIFF_EPS);
	}

	public : double dFiniteDiff1(double x)
	{
	    return max( (fabs(x) + SQRT_EPS) * SQRT_EPS, DIFF_EPS1);
	}

	public : double dFiniteDiff2(double x)
	{
	    return max( (fabs(x) + SQRT_EPS) * SQRT_EPS, DIFF_EPS2);
	}



	public : double** Num2Derivative(double* vP)
	{
	    int i, j;
	    double pi, pj, hi, hj, f, fpp, fmm, fpm, fmp;
		double hia, hja, di, dj;

	    double** m = (double**)malloc(sizeof(double*) * cp);

	    f = Function(vP);// get function at vP, around which to differentiate


	    for (i = 0; i < cp; i++)// approximate 2nd deriv. by central difference
	    {
	    	double* m1 = (double*)malloc(sizeof(double) * cp);

	        for(j=0;j<i;j++)
	        {
	        	m1[j] = m[j][i];
	        }

	        pi = vP[i];
	        hi = dFiniteDiff2(pi);
	        di = pi + 2 * hi;
            // diagonal elements
	        vP[i] = di;           	                   // +1 +1
	        fpp = Function(vP);
	        hia = (di - pi) / 2;
	        di = pi - 2 * hi;
	        vP[i] = di;           	                   // -1 -1
	        fmm = Function(vP);
	        hia += (pi - di) / 2;



	        m1[i] = (((fpp - f) + (fmm - f)) / (hia * hia));
	        vP[i] = pi;

	        for (j = i+1; j < cp; j++)                    // off-diagonal elements
	        {
	            pj = vP[j];
	            hj = dFiniteDiff2(pj);

	            di = pi + hi;
	            dj = pj + hj;
	            vP[i] = di;
	            vP[j] = dj;     // +1 +1
				hia = di - pi;   hja = dj - pj;

				fpp = Function(vP);
				di = pi - hi;
				dj =  pj - hj;
				vP[i] = di;
				vP[j] = dj;      // -1 -1
				hia += pi - di;   hja += pj - dj;

				fmm = Function(vP);

	            vP[i] = pi+hi;
	            vP[j] = pj - hj;               // +1 -1
	            fpm = Function(vP);

	            vP[i] = pi - hi;
	            vP[j] = pj + hj;               // -1 +1
	            fmp = Function(vP);



	            m1[j] = (((fpp - fpm) + (fmm - fmp)) / (hia * hja));

	            vP[j] = pj;
	        }

	        vP[i] = pi;
	        m[i] = m1;
	    }
	    return m;
	}


	public : double* Num1Derivative(double* vP) throw(RuntimeException)
	{
	    int i;
	    double p, h, f, fm, fp, d, hr, right, left, hl;

	    
	    size = cp * sizeof(double);
	    double* v = (double*)malloc(size);

	    for (i = 0; i < cp; i++)    // get 1st derivative by central difference
	    {
	        p = (double)vP[i];

	        h = dFiniteDiff1(p);
	        d = p + h;

	        vP[i] = d;
			hr = h;
			hl = h;
			try {
				right = Function(vP);
				
				fp = right;
		        d = p - h;
				vP[i] = d;

				try {
					left = Function(vP);
					fm = left;
					vP[i] = p;                         // restore original parameter

					v[i] = (fp-fm) / (hr+hl);     // take central difference

				}
				catch (RuntimeException &e)
				{
					fp = right;
					vP[i] = p;                         // restore original parameter
					try
					{
						f = Function(vP);	      // not: try to get it
					    v[i] = (fp - f) / hr;  // take right difference
					}
					catch (RuntimeException &e1)
					{
						throw RuntimeException("Unable to evaluate numerical derivative");
					}
				}
		    }
			catch (RuntimeException &e)
			{
				try {
					left = Function(vP);
					fm = left;
					vP[i] = p;                         // restore original parameter
					try
					{
						f = Function(vP);	      // not: try to get it
						 v[i] = (f - fm) / hl; // take left difference
					}
					catch (RuntimeException &e1) {
						throw RuntimeException("Unable to evaluate numerical derivative");
					}

				}
				catch (RuntimeException &e2) {
					throw RuntimeException("Unable to evaluate numerical derivative");
				}
			}
	    } // for loop end

		return v;
	}

// End of Num1Derivative

	public: double* multiply(double* arraylist, double scalar)
	{
		int i;
	   	double* output = (double*)malloc(size);
		for(i=0;i<cp;i++)
		{
			output[i] = scalar * arraylist[i];
		}

		return output;
	}

	public: double* multiply(double* arraylist, double scalar, int k)
	{
		int i;
	   	double* output = (double*)malloc(sizeof(double) * k);
		for(i=0;i<k;i++)
		{
			output[i] = scalar * arraylist[i];
		}

		return output;
	}

	void multiply_to(double* arraylist, double scalar)
	{
		int i;
		for(i=0;i<cp;i++)
		{
			arraylist[i] = scalar * arraylist[i];
		}

	}

	void multiply_to(double* arraylist1, double* arraylist, double scalar)
	{
		int i;
		for(i=0;i<cp;i++)
		{
			arraylist1[i] = scalar * arraylist[i];
		}

	}

	void multiply_to(double* arraylist, double scalar, int k)
	{
		int i;
		for(i=0;i<k;i++)
		{
			arraylist[i] = scalar * arraylist[i];
		}

	}

/*
	vector< double > multiply2(vector< double > arraylist1, vector< double > arraylist2) throws RuntimeException
	{
		int cp = arraylist1.size();
		int cp1 = arraylist2.size();

		if (!(cp == cp1))
		{
			throw new RuntimeException("Trying to multiply element by element a vector of size " + cp + " with a vector of size " + cp1);
		}

		int i;
		vector< double > output = new vector< double >();
		for(i=0;i<cp;i++)
		{
			output.push_back(arraylist1[i] * arraylist2[i]);
		}

		return output;
	}
*/
	protected : double internal_product(double* arraylist, double* arraylist1)
	{
		int i;
		double output = 0;

		for(i=0;i<cp;i++)
		{
			output += arraylist[i] * arraylist1[i];
			//cout << output <<  endl;
		}

		return output;
	}

	public : static double internal_product(double* arraylist, double* arraylist1, int k)
	{
		int i;
		double output = 0;

		for(i=0;i<k;i++)
		{
			output += arraylist[i] * arraylist1[i];
			//cout << output <<  endl;
		}

		return output;
	}
	
	public : static double internal_product(double* arraylist, double* arraylist1, int min, int max)
	{
		int i;
		double output = 0;

		for(i=min;i<=max;i++)
		{
			output += arraylist[i] * arraylist1[i];
		}

		return output;
	}

	protected : double** multiply1(double* vector1, double* vector2)
	{
		int i;
		double** result = (double**)malloc(sizeof(double*) * cp);

		for(i=0;i< cp;i++)
		{
			double* result1 = (double*)malloc(size);
			int j;
			for(j=0;j<cp;j++)
			{
				result1[j] = vector1[i] * vector2[j];
			}
			result[i] = result1;
		}
		return result;
	}

	protected : double* multiply(double** matrix, double* vector1)
	{
	   	double* result = (double*)malloc(size);
		int i;
		int rows = cp;
		int n = cp;

		for(i=0;i<rows;i++)
		{
			int j;
			result[i] = 0;
			for(j=0;j<n;j++)
			{
				result[i] += matrix[i][j] * vector1[j];
			}
		}
		return result;
	}

	protected : void multiply_to(double* vector, double** matrix, double* vector1)
	{
		int i;
		int rows = cp;
		int n = cp;

		for(i=0;i<rows;i++)
		{
			int j;
			vector[i] = 0;
			for(j=0;j<n;j++)
			{
				vector[i] += matrix[i][j] * vector1[j];
			}
		}
	}

	protected : double** add1(double** matrix1, double** matrix2)
	{
		double** result = (double**)malloc(sizeof(double*) * cp);
		int i;
		for(i=0;i<cp;i++)
		{
			double* result1 = (double*)malloc(sizeof(double) * cp);
			int j;
			for(j=0;j<cp;j++)
			{
				result1[j] = matrix1[i][j]+matrix2[i][j];
			}
			result[i] = result1;
		}
		return result;
	}

	public : void add_to(double** matrix1, double** matrix2)
	{
		int i;
		for(i=0;i<cp;i++)
		{
			int j;
			for(j=0;j<cp;j++)
			{
				matrix1[i][j]+=matrix2[i][j];
			}
		}
	}
	
	public : void add_to(double** matrix1, double** matrix2, int k)
	{
		int i;
		for(i=0;i<k;i++)
		{
			int j;
			for(j=0;j<k;j++)
			{
				matrix1[i][j]+=matrix2[i][j];
			}
		}
	}

	public : double** subtract1(double** matrix1, double** matrix2)
	{
		double** result = (double**)malloc(sizeof(double*) * cp);
		int i;
		for(i=0;i<cp;i++)
		{
			double* result1 = (double*)malloc(sizeof(double) * cp);
			int j;
			for(j=0;j<cp;j++)
			{
				result1[j] = matrix1[i][j]-matrix2[i][j];
			}
			result[i] = result1;
		}
		return result;
	}

	protected : double* add(double* arraylist, double* arraylist1)
	{
	   	double* output = (double*)malloc(sizeof(double) * cp);
		int i;
		for(i=0;i<cp;i++)
		{
			output[i] = arraylist1[i] + arraylist[i];
		}

		return output;
	}

	protected : double* add(double* arraylist, double* arraylist1, int k)
	{
	   	double* output = (double*)malloc(sizeof(double) * k);
		int i;
		for(i=0;i<k;i++)
		{
			output[i] = arraylist1[i] + arraylist[i];
		}

		return output;
	}

	protected : void add_to(double* arraylist, double* arraylist1)
	{
		int i;
		for(i=0;i<cp;i++)
		{
			arraylist[i] = arraylist1[i] + arraylist[i];
		}
	}

	
	protected : void add_to(double* dest, double* arraylist, double* arraylist1)
	{
		int i;
		for(i=0;i<cp;i++)
		{
			dest[i] = arraylist1[i] + arraylist[i];
		}
	}

	public : void add_to(double* arraylist, double* arraylist1, int k)
	{
		int i;
		for(i=0;i<k;i++)
		{
			arraylist[i] = arraylist1[i] + arraylist[i];
		}
	}

	public : double* subtract(double* arraylist, double* arraylist1) throw(RuntimeException)
	{
		double* output = (double*)malloc(sizeof(double) * cp);


		int i;
		for(i=0;i<cp;i++)
		{
			output[i] = arraylist[i] - arraylist1[i];
		}

		return output;
	}

	void subtract_to(double* arraylist, double* arraylist1)
	{
		int i;
		for(i=0;i<cp;i++)
		{
			arraylist[i] = arraylist[i] - arraylist1[i];
		}

	}

	void subtract_to(double* dest, double* arraylist, double* arraylist1)
	{
		int i;
		for(i=0;i<cp;i++)
		{
			dest[i] = arraylist[i] - arraylist1[i];
		}

	}


	void subtract_to(double* arraylist, double* arraylist1, int k)
	{
		int i;
		for(i=0;i<k;i++)
		{
			arraylist[i] = arraylist[i] - arraylist1[i];
		}

	}
/*
	protected vector< double > copy(vector< double > arraylist)
	{
		int cp = arraylist.size();
		int i;
		vector< double > output = new vector< double >();
		for(i=0;i<cp;i++)
		{
			output.push_back(arraylist[i]);
		}

		return output;
	}
*/
	public : static double* zeros(int n) {
		int i;
		double* result = (double*)malloc(sizeof(double) * n);
		for(i=0;i<n;i++)
		{
			result[i]  = 0.;
		}
		return result;
	}

	public : static int* zeros_int(int n) {
		int i;
		int* result = (int*)malloc(sizeof(int) * n);
		for(i=0;i<n;i++)
		{
			result[i]  = 0.;
		}
		return result;
	}

	public : double* ones(int n) {
		int i;
		double* result = (double*)malloc(sizeof(double) * n);
		for(i=0;i<n;i++)
		{
			result[i]  = 1.;
		}
		return result;
	}

	protected :  double norm(double* arraylist)
	{
		double result = 0;
		int i;
		for(i=0;i<cp;i++)
		{
			if (abs(arraylist[i]) > result)
			{
				result = abs(arraylist[i]);
			}
		}
		return result;
	}

	protected : double* e(int c, int n)
	{
		double* result = (double*)malloc(sizeof(double) * n);
		int i;
		for(i=0;i<n;i++)
		{
			if (i==c)
				result[i] = 1.;
			else
				result[i] = 0.;
		}

		return result;
	}

	public : virtual double* dFunction(double* vP)  throw(RuntimeException)   {
		return Num1Derivative(vP);
	}
/*
	public double sum(vector< double > x)
	{
		double result = 0;
		int n = x.size(), i;
		for(i=0;i<n;i++)
		{
			result += x[i];
		}
		return result;
	}

	public vector< double > range(double start, double end, double increment)
	{
		double x = start;
		vector< double > result = new vector< double >();
		result.push_back(start);
		while ((x+1E-7) < end)
		{
			x += increment;
			result.push_back(x);
		}
		return result;
	}

	public vector< double > ones(int n)
	{
		int i;
		vector< double > result = new vector< double >();
		for(i=0;i<n;i++)
		{
			result.push_back(1.);
		}
		return result;
	}

	public vector< double > divide(vector< double > x, double y)
	{
		vector< double > result = new vector< double >();
		int n = x.size(), i;
		for(i=0;i<n;i++)
		{
			result.push_back(x[i] / y);
		}
		return result;
	}

	public ArrayList<vector< double >> divide1(ArrayList<vector< double >> x, double y)
	{
		ArrayList<vector< double >> result = new ArrayList<vector< double >>();
		int m = x.size(), i;
		for(i=0;i<m;i++)
		{
			vector< double > x1 = x[i];
			vector< double > result1 = new vector< double >();
			int n = x1.size();
			int j;
			for(j=0;j<n;j++)
				result1.push_back(x1.get(j) / y);
			result.push_back(result1);
		}
		return result;
	}
*/
	protected  : double** unit(int n)
	{
		int i;
		double** result = (double**)malloc(sizeof(double*) * n);

		for(i=0;i<n;i++)
		{
			double* row = e(i, n);
			result[i] = row;
		}
		return result;
	}

	protected : double** transpose(double** matrix)
	{
		int i;
		double** result = (double**)malloc(sizeof(double*) * cp);

		for(i=0;i<cp;i++)
		{
			int j;
			double* result1 = (double*)malloc(sizeof(double) * cp);
			for(j=0;j<cp;j++)
			{
				result1[j] = matrix[j][i];
			}
			result[i] = result1;
		}
		return result;
	}
	
	protected : double** transpose(double** matrix, int n, int k)
	{
		int i;
		double** result = (double**)malloc(sizeof(double*) * k);

		for(i=0;i<k;i++)
		{
			int j;
			double* result1 = (double*)malloc(sizeof(double) * n);
			for(j=0;j<n;j++)
			{
				result1[j] = matrix[j][i];
			}
			result[i] = result1;
		}
		return result;
	}
	
	protected : double** transpose(double** matrix, int n, int min, int max, int k)
	{
		int i;
		double** result = (double**)malloc(sizeof(double*) * k);

		for(i=0;i<k;i++)
		{
			int j;
			double* result1 = (double*)malloc(sizeof(double) * n);
			for(j=min;j<=max;j++)
			{
				result1[j] = matrix[j][i];
			}
			result[i] = result1;
		}
		return result;
	}

	private : void Initialize(double* vP)
	{
		size = cp * sizeof(double);
		avP = (double*)malloc(size);
		memcpy(avP, vP, size);

		ret_val = MAX_NOCONV;
		adStep = 1;
		adFunc = -1E+100;
	}

	public :  double* MaxBFGS(double* vP) throw(RuntimeException)
	{
		// first process default arguments settings
		Initialize(vP);

		int itno;

		// start of MaxBFGS
		itno = -1;

		bool fstepok, fconv, fok;
		bool fhesreset = false;
    	double  dsteplen, dmxstep, dfprev, doldstep;
    	double d_2 = 0;
    	double d_2i, d_3;

    	double* vscore = (double*)malloc(size);
    	double* vdelta = (double*)malloc(size);
    	double* vgamma = (double*)malloc(size);
    	double* vhgamm = (double*)malloc(size);
    	double* vdeltaprev = (double*)malloc(size);
    	double* vprev =  (double*)malloc(size);

    	fill(vscore, vscore + cp, 0);
    	fill(vdelta, vdelta + cp, 0);
    	fill(vhgamm, vhgamm + cp, 0);

    	dsteplen = 1.0;
    	dmxstep = 1.0;
    	

    	
    	if (minvhess_set)
    	{
    		minvhess = minvhess_1;        	
        	
    	}
    	else
    	{
    		minvhess = unit(cp);
    	}
    	

    	bool go_on = true;

    	do                      // start iterating, first loop is initialization
    	{
    		itno++;  fstepok = true;  fconv = false;
    		//cout << itno << "\t" <<  s_mxIter << endl;
    		doldstep = dsteplen;
    		dsteplen = dmxstep;
    		adStep = dsteplen;
    		dfprev = adFunc;       // keep old dfunc; compute new parameters
    		memcpy(vprev, avP, sizeof(double) * cp);

    		multiply_to(vdeltaprev, vdelta, dsteplen);

    		add_to(avP, vdeltaprev);		  // new trial value; first time: push_back 0
    		memcpy(vgamma, vscore, size);

                                         // evaluate function: 1 is success
    		try {
    			adFunc = Function(avP);
    			fok = true;
    		}
    		catch (RuntimeException &e)
    		{
    			if (itno == 0)
    			{
    				Finalize(vscore, vhgamm, vdelta, vprev, vdeltaprev, vgamma, minvhess);
    				throw RuntimeException("MaxBFGS(): function evaluation failed at starting values");
    			}
    			else
    			{
    				fok = false;
    				adFunc = -1E+300;
    			}
    		}

    		int i;

    		if (itno > 0)
    		{
    			double adfunc1 = adFunc;

    			if (((adFunc < dfprev) || (fok == false) || (dsteplen * norm(vdelta) > 10 * (0.1 + norm(vgamma)) )) || (std::isnan(adFunc))
    						)
    			{

    				// failure to evaluate or improve: determine steplength in (0,1]
    				// now also for large step
    				try {
    					fLineSearch(vdelta, dfprev, vgamma);
    					dsteplen = adStep;
    					Function(avP);
    				}
    				catch (RuntimeException &e) {
    					// no improvement in line search
    					memcpy(vscore, vgamma, size);
    					// also reset old score
    					if (!fhesreset)                  // don't do twice in a row
    					{
    						fhesreset = true; // try once more, with a unit Hessian
    						memcpy(vdelta, vscore, size);                          // vdelta = q
    						
    						continue;                                  // try again
    					}
    					else
    					{
    						double* hulp_subtract = subtract(avP , vprev);
    						if (itno > 1)
    						{
    							fconv = fIsStrongConvergence(avP, vscore, hulp_subtract, s_dEps1);
    							free(hulp_subtract);
    							if (fconv)
    								fstepok = true;
    							else
    							{
    								fstepok = false;
    								fconv = fIsWeakConvergence(avP, vscore, s_dEps1);
    							}
    						}

    						break;                                // stop iterating
    					}
    				}
    				if (!fok)            // yes: failed to evaluate at steplength 1
    					dmxstep = max(dsteplen, 0.1);//switch to restr. step method


    				multiply_to(vdelta, dsteplen);           // actual delta after line search
    			}
    		}


    		fhesreset = false;
    		if (itno > 0)                      // adjust restricted step method
    		{
    			if (dsteplen >= dmxstep)             // grow maximum steplength
    			{
    				dmxstep *= 2;
    				if (dmxstep > 1)
    					dmxstep = 1.0;
    			}
    			else if (dmxstep < 1 && dsteplen < dmxstep / 2)       // shrink
    			{
    				dmxstep /= 2;
    				if (dmxstep < 1e-6)
    					dmxstep = 1e-6;
    			}
    		}


    		try {
    			double* df = dFunction(avP);
    			memcpy(vscore, df, size) ;
    			free(df);
    		}
    		catch (RuntimeException &e) {
    			throw RuntimeException("MaxBFGS(): numerical score failed");
    		}

    		subtract_to(vgamma, vscore);  /* g = change in score */

    		if (itno > 0)
    		{	   // Hnew = Hold + [(1 + g'Hg/d'g)/d'g]dd' - (dg'H + Hgd')/d'g
                  // first time: d_2 == d_3 == 0, so Hessian doesn't change
    			multiply_to(vhgamm, minvhess, vgamma);

    			d_2 = internal_product(vdelta, vgamma);							 // d'g
    			d_3 = internal_product(vgamma, vhgamm);						    // g'Hg


    			if (d_2 <= 0)
    			{
    				d_2 = 0;
    				for(i=0;i<cp;i++)
    					free(minvhess[i]);
    				free(minvhess);

    				minvhess = unit(cp);                       // reset Hessian
    			}
    			else
    			{
    				d_2i = 1 / d_2;                             // d_2i = 1/d'g
    				d_3 = (1 + d_3 * d_2i) * d_2i;        // (1 + g'Hg/d'g)/d'g

    				double* hulp_mult2 = multiply(vdelta, d_3 / 2);
    				double* hulp_mult3 = multiply(vhgamm, d_2i);
    				double** hulp_mult = multiply1(hulp_mult2, vdelta);
    				free(hulp_mult2);
    				double** hulp_mult1 = multiply1(hulp_mult3, vdelta);
    				free(hulp_mult3);
    				double** mup = subtract1(hulp_mult, hulp_mult1);
    				double** trans = transpose(mup);

    				add_to(mup, trans);
    				add_to(minvhess, mup);

    				int i;
    				for(i=0;i<cp;i++)
    				{
    					free(hulp_mult[i]);
    					free(hulp_mult1[i]);
    					free(mup[i]);
    					free(trans[i]);
    				}
    				free(hulp_mult);
    				free(hulp_mult1);
    				free(mup);
    				free(trans);
    			}
    		}

    		multiply_to(vdelta, minvhess, vscore);                          // vdelta = Hq

    		if (itno <= 0)
    			memcpy(vdeltaprev, vdelta, sizeof(double) * cp);
    		else
    			subtract_to(vdeltaprev, avP, vprev);

    		
    		if (itno > 1)
    		{
    			fconv = fIsStrongConvergence(avP, vscore, vdeltaprev, s_dEps1);
    		}
    		
    		if (!fconv && s_iPrint > 0 && (((double)itno / (double)s_iPrint) - itno / s_iPrint) == 0)
    			MaxMonitor("BFGS", itno, avP, vscore, vdeltaprev, adFunc, adFunc,
    					dsteplen, 0, ret_val, s_bCompact);
    		
 
/*
    		if (fconv)
    		{
    			int i;
				//if (!minvhess_set)
					minvhess_1 = unit(cp);
    			for(i=0;i<cp;i++)
    			{
    			//	cout << i << endl;
    				memcpy(minvhess_1[i], minvhess[i], cp);
    			}
    		}
  */  		
    		if (check_convergence)
    			go_on =  !(fconv || itno >= s_mxIter);
    		else
    			go_on = !(itno >= s_mxIter);

    	} while (go_on);


    ret_val = iConvergenceCode(fconv, fstepok, itno >= s_mxIter);
    
    if (itno == s_mxIter)
    {
    	bool hulp = fIsWeakConvergence(avP, vscore, s_dEps2);
    	
    
    	if (hulp)
    		ret_val = MAX_WEAK_CONV;
    }

    if (s_iPrint > 0)
    {
        MaxMonitor("BFGS", itno, avP, vscore, vdeltaprev, adFunc, adFunc,
			dsteplen, 0, ret_val, s_bCompact);
    }


    Finalize(vscore, vhgamm, vdelta, vprev, vdeltaprev, vgamma, minvhess);
    N = itno;
    
    return avP;
}

private : void Finalize(double* vscore, double* vhgamm, double* vdelta, double* vprev,
		double* vdeltaprev, double* vgamma, double** minvhess)
{
    free(vscore);
    free(vhgamm);
    free(vdelta);
    free(vprev);
    free(vdeltaprev);
    free(vgamma);

    int i;
    for(i=0;i<cp;i++)
    	free(minvhess[i]);
    free(minvhess);
}
             // now have analytical score if used, else get numerical later


/*----------------------------- fLineSearch --------------------------------*/
    void fLineSearch(double* vDelta,
        double dFuncBase, double* vScoreBase) throw(RuntimeException)
    {
        double dstep = adStep;
        double f;

        do                         /* move towards zero with linear line search */
        {
        	dstep /= 2;

        	double* mult_hulp = multiply(vDelta, -dstep);
            add_to(avP, mult_hulp);
            free(mult_hulp);
            try
            {
            	adFunc = Function(avP);
            	if (adFunc > dFuncBase)
            	{
            		/* function & better function value; continue while improvement */
            		double* mult_hulp =  multiply(vDelta, -dstep / 2);
            		double* avp_new = add(avP, mult_hulp);
            		free(mult_hulp);
            		try
            		{
            			f = Function(avp_new);
            			while (f > adFunc)
            			{
            				dstep /= 2;
            				double* mult_hulp = multiply(vDelta, -dstep);
            				add_to(avP, mult_hulp);
            				free(mult_hulp);
            				adFunc = f;
            				double* mult_hulp1 = multiply(vDelta, -dstep / 2);
            				add_to(avp_new, avP, mult_hulp1);
            				free(mult_hulp1);
            				try {
            					f = Function(avp_new);
            				}
            				catch (RuntimeException &e) {
            					break;
            				}
            			}

            			adStep = dstep;
            			free(avp_new);
                        return;
            		}
            		catch (RuntimeException &e) {
            		}
            		free(avp_new);
            	}
            }
            catch (RuntimeException &e) {
            }
        } while (dstep >= s_dStepLinearGap);
                                       /* if we're here, the line search failed */
                 /* reset function value and parameters to that at steplength 0 */
        double* mult_hulp = multiply(vDelta, -dstep);
        avP = add(avP, mult_hulp);     /* this removes the remaining step length */
        free(mult_hulp);
        adFunc = dFuncBase; /* dFuncBase was function value of previous iter */
        adStep = dstep;
        throw RuntimeException("Line search failed");
    }

	private : bool fIsStrongConvergence(double* vP, double* vScore, double* vDelta,
    					double dEps1)
    {
    	vector< double > vpp;
    	int i;
    	for(i=0;i<cp;i++)
    	{
    		if (vP[i] == 0)
    			vpp.push_back(1.);
    		else
    		{
    			double hulp = cp * cp;
    			vpp.push_back(1 / hulp + fabs(vP[i]));
    		}
    	}

    	for(i=0;i<cp;i++)
    	{
    		if ((fabs(vScore[i] * vpp[i]) > dEps1) || (abs(vDelta[i] / vpp[i]) > 10 * dEps1))
    			return false;
    	}

    	return true;
    }

	private : bool fIsWeakConvergence(double* vP, double* vScore, double dEps2)
    {
    	vector< double > vpp;
    	int i;
    	for(i=0;i<cp;i++)
    	{
    		if (vP[i] == 0)
    			vpp.push_back(1.);
    		else
    		{
    			double hulp = cp * cp;
    			vpp.push_back(1 / hulp + fabs(vP[i]));
    		}
    	}

    	for(i=0;i<cp;i++)
    	{
    		if (fabs(vScore[i] * vpp[i]) > dEps2)
    			return false;
    	}
    	return true;
    }

	private : RetVal iConvergenceCode(bool fConv, bool fStepOK, bool fItMax)
    {
        if (!fStepOK)
        {
        	if (fConv)
        		return MAX_WEAK_CONV;
            return MAX_LINE_FAIL;
        }
        else if (fConv)
            return MAX_CONV;
        else
            return MAX_MAXIT;
    }

    public : void print_vector(double* vector) {
    	int i;
    	int n = cp;
    	int row_index = 0;
    	for(i=0;i<n;i++)
    	{
    		row_index++;
    		cout << "\t" << setprecision(5)<< fixed <<  vector[i];
    		if (row_index == 6)
    		{
    			cout << "\n";
    			row_index = 0;
    		}
    	}
    	cout << "\n";
    }

    public : static void print_vector(double* vector, int n) {
    	int i;

    	int row_index = 0;
    	for(i=0;i<n;i++)
    	{
    		row_index++;
    		cout << "\t" << setprecision(5)<< fixed <<  vector[i];
    		if (row_index == 6)
    		{
    			cout << "\n";
    			row_index = 0;
    		}
    	}
    	cout << "\n";
    }

    private : void MaxMonitor(string sMethod, int cIter, double* vP, double* vScore, double* vDelta,
    	    double dFuncInit, double dFunc, double steplen, int condhes, RetVal ret_val,
    		bool bCompact)
    	{
    		cout.precision(8);
    	    if (cIter <= 0)
    	        cout << "\nStarting values" << endl;
    	    else
    	    {
    	    	cout << "\nPosition after " <<  cIter << " ";
    	    	cout << sMethod;//
    	    	cout <<  " iterations" << endl;

    	        if (ret_val != MAX_NOCONV)
    	        {
    	            cout << "Status: ";
    	        	cout <<   MaxConvergenceMsg(ret_val) << endl;
    	        }
    	    }
    	    cout << "parameters" << endl;
    	    print_vector(vP);

    	    if (cp > 0)
    	        cout << "gradients" << endl;

    	    print_vector(vScore);


    	    if (cIter <= 0)
    	    {
    	    	cout <<  "Initial Function = "<< setprecision(10) << dFuncInit;

    	    }
    	    else
    	    {
    	    	cout << "function value = "<< setprecision(10) << dFunc;
    	        if (condhes != 0)
    	            cout << "  condhes = "<< condhes;
    	        if (steplen != 1)
    	            cout << "  steplen = "<< steplen;
    	    }
    	    cout << "\n";
    	}

    private : string MaxConvergenceMsg(RetVal iCode)
    {
        if (iCode == MAX_CONV)
            return "Strong convergence";
        else if (iCode == MAX_WEAK_CONV)
            return "Weak convergence (no improvement in line search)";
        else if (iCode == MAX_MAXIT)
            return "No convergence (maximum no of iterations reached)";
        else if (iCode == MAX_LINE_FAIL)
            return "No convergence (no improvement in line search)";
        else if (iCode == MAX_FUNC_FAIL)
            return "No convergence (function evaluation failed)";
        else
            return "No convergence";
    }
};


#endif




