/*
 * IndirectEstimation.cpp
 *
 *  Created on: Jun 4, 2013
 *      Author: Aico van Vuuren
 */

#ifndef INDIRECT_INCLUDED
#define INDIRECT_INCLUDED

#include "maximize.cpp"
#include "NumJacobian.cpp"
#include "loadmat.cpp"
#include "savemat.cpp"
#include "hazards_function.cpp"
#include "sample.cpp"
#include "probit.cpp"
#include "logit.cpp"
#include <stdlib.h>
#include "maxsan.cpp"
#include "retval.cpp"
#if _MSC_VER > 1500
#define isinf(x) (!_finite(x))
#define isnan(x) (_isnan(x))

#endif

using namespace std;


extern "C" void dgetri_ ( int* n, const void* A, int* lda,
    int* ipiv, const void* work, int* lwork, int* info );

extern "C" void dgetrf_( int* m, int* n, const void* a, int* lda,
    int* ipiv, int* info );

class IndirectEstimation : public Maximize, public NumJacobian, public Maximize_SAN {
	double**  T_1_totaal_a; // Unemployment duration that ends in local labor market per spell
	double**  T_1_totaal_b;	// Unemployment duration that ends in non-local labor market per spell
	double**  D_1_totaal_a;	// Dummy for local labor market	per spell
	double**  D_1_totaal_b;	// Dummy for non-local labor market	per spell
	double**  D_2_totaal; // Dummy for right censoring per spell
	double**  T_2_totaal; // Dummy for employment spell per spell
	double**  H2_totaal; // Dummy for homeownership of the unemployed per spell
	double**  H4_totaal; // Dummy for homeownership of the employed per spell 
	double**  X;  // X variables
	double* vP_Theta;  // Set of coefficients for the hazard rate model
	public : double rss_dividor;
	string directory;
	double MaxValue;	// Max value of minus the criterion function 
	int cp1_indirect, n, k;
	public : int cp_indirect;
	private : double Lambda_H;	// Smoothing parameter
	public : bool printing;
	double Rho, Beta_W; // Parameters
	double* b;	 
	int myid, numprocs;
	int n_size;	  // number of observations
	MPI_Status status;
	int Min, Max, increment;
	private : double* u2;
	private : double* u2_1;
	private : double* u3;
	private : double* u_x;
	private : double** u16;
	private : double** u17;
	private : double** u14;
	private : double** u14a;
	private : double** u14b;
	private : double** u15;
	private : double** u15a;
	private : double** u_u_spells;
	private : double** u_e_spells;
	private : double* u_w;
	private: int* U_spells_totaal;
	private : int* E_spells_totaal;
	private : string intermed_string;
	private : double** g_all;
	private : double* G2;
	private : double* G3;
	private : double* G4;
	private : double* G5;
	private : double* G1;
	private : double* W;
	private : double** mhess1;
	private : double** mhess;
	private : double** mhess2;
	private : double** mhess2_2;
	private : double** mhess2_3;
	private : double** mhess2_4;
	private : bool* variables;
	private : double P0000_0, P0000;
	private : double P1000_0, P1000;
	private : double P0100_0, P0100;
	private : double P0010_0, P0010;
	private : double P1100_0, P1100;
	private : double P1010_0, P1010;
	private : double P0110_0, P0110;
	private : double P1110_0, P1110;
	private : double P0001_0, P0001;
	private : double P1001_0, P1001;
	private : double P0101_0, P0101;
	private : double P0011_0, P0011;
	private : double P1101_0, P1101;
	private : double P1011_0, P1011;
	private : double P0111_0, P0111;
	private : double P1111_0, P1111;
	private : double* b_0;		// set of coefficients for the distribution regresssion - first quintile  
	private : double* b2_0;	   // set of coefficients for the distribution regression - second quintile
	private : double* b3_0;	  // set of coefficients for the distribution regression - third quintile
	private : double* b4_0;	  // set of coefficients for the distribution regresion - fourth quintile
	private : int N1;
	private : double* v_dsteplen;
	private : double* vP1;

	public : IndirectEstimation(double Rho, int n, int k, int cp,
					bool* variables, int cp1_indirect, string directory, int myid, int numprocs,
					string intermed_string,
					MPI_Status status) throw(IOException)
		{
			printing = true;
			n_size = 0;
			rss_dividor = 1000;
			MaxValue = -1e+9;
			this->directory = directory;
	//		this->Mu = Mu;
			this->Rho = Rho;
//			this->Sigma = Sigma;
			this->intermed_string = intermed_string;
			this->variables = variables;
			cp_indirect = 0;
			v_dsteplen = zeros(1);
			int i;
			for(i=0;i<cp1_indirect;i++)
			{
				if (variables[i])
					cp_indirect++;
			}
			LoadData load;
			double* vp_theta_1;
			try {
				vp_theta_1 = load.LoadTxt_vector(directory + "vp_theta.txt");
				int hulp = load.m;
			}
			catch (IOException &e) {
				throw e;
				return;
			}

			int k2 = HazardsFunction::x_size;
			int k2_1 = HazardsFunction::x_size_1;

			vP_Theta = (double*)malloc((3+4 * k2+47) * sizeof(double));

			memcpy(vP_Theta, vp_theta_1, 3 * sizeof(double));
			memcpy(&vP_Theta[3], &vp_theta_1[3], k2 * sizeof(double));
			memcpy(&vP_Theta[3 + k2], &vp_theta_1[3 + k2_1], k2 * sizeof(double));
			memcpy(&vP_Theta[3 + 2 * k2], &vp_theta_1[3 + 2 * k2_1], k2 * sizeof(double));
			memcpy(&vP_Theta[3 + 3 * k2], &vp_theta_1[3 + 3 * k2_1], k2 * sizeof(double));
			memcpy(&vP_Theta[3 + 4 * k2], &vp_theta_1[3 + 4 * k2_1], 47 * sizeof(double));
			free(vp_theta_1);

			b = (double*)malloc(sizeof(double) * (HazardsFunction::x_size+1));
			b_0 = load.LoadTxt_vector(directory + "b1_dist.txt");

			this->cp1_indirect = cp1_indirect;
			this->n = n;
			this->k = k;
			this->cp = cp;
			this->cp_SAN = cp;
			this->myid = myid;
			this->numprocs = numprocs;
			this->status = status;
			increment = n/numprocs;
			Min = myid * increment;
			Max = myid < numprocs-1? (myid+1) * increment-1 : n-1;
			vP1 = (double*)malloc(sizeof(double) * HazardsFunction::x_size_1);

			mt19937 mt(642012511);
			uniform_real_distribution<double> dist(0.1);
			u2 = new double[n];
			u3 = new double[n];
			u2_1 = new double[n];
			u_x = new double[n];
 			u16 = new double*[n];
			u17 = new double*[n];
			u14 = new double*[n];
			u14a = new double*[n];
			u14b = new double*[n];
			u15a = new double*[n];
			u15 = new double*[n];
			u_u_spells = new double*[n];
			u_e_spells = new double*[n];
			u_w = new double[n];
	
			for(int i=0;i<n;i++)
			{
				u2[i] = dist(mt);
				u2_1[i] = dist(mt);
				u3[i] = dist(mt);
				u_x[i] = dist(mt);
				u_w[i] = dist(mt);
				u16[i] = new double[k];
 				u17[i] = new double[k];
				u14[i] = new double[k];
				u14a[i] = new double[k];
				u14b[i] = new double[k];
				u15[i] = new double[k];
				u15a[i] = new double[k];
				u_u_spells[i] = new double[k];
				u_e_spells[i] = new double[k];
				
				for(int j=0;j<k;j++)
				{
					u16[i][j] = dist(mt);
					u17[i][j] = dist(mt);
					u14[i][j] = dist(mt);
					u14a[i][j] = dist(mt);
					u14b[i][j] = dist(mt);
					u15[i][j] = dist(mt);
					u15a[i][j] = dist(mt);
					u_u_spells[i][j] = dist(mt);
					u_e_spells[i][j] = dist(mt);
				}
			}			
			try {
				g_all = load.LoadTxt(directory + "g.txt",n); // data for the "sampled" x's 
				mhess = load.LoadTxt(directory + "mhess.txt"); // hessian for the hazard rates model
				mhess2 = load.LoadTxt(directory + "mhess1_1.txt"); // hessian for the distribution regression at Q20
				mhess2_2 = load.LoadTxt(directory + "mhess1_2.txt"); // hessian for the distribution regression at Q40
				mhess2_3 = load.LoadTxt(directory + "mhess1_3.txt"); // hessian for the distribution regression at Q60
				mhess2_4 = load.LoadTxt(directory + "mhess1_4.txt"); // hessian for the distribution regression at Q80
				b2_0 = load.LoadTxt_vector(directory + "b2_dist.txt"); // coefficients of the distribution regression at Q20
				b3_0 = load.LoadTxt_vector(directory + "b3_dist.txt"); // coefficients of the distribution regression at Q40
				b4_0 = load.LoadTxt_vector(directory + "b4_dist.txt"); // coefficients of the distribution regression at Q60
			}
			catch (IOException &e) {
				throw IOException(e.filename);
			}

			n_size = load.m;
			int k1 = HazardsFunction::x_size;
			vector<double> p;
			double p2 = 0;

			for(i=3+4*k1+29;i<=3+4 * k1+43;i++)
			{
				double p1 = exp(vP_Theta[i]);
				p.push_back(p1);
				p2 += p1;
			}

			P0000_0 = p[0] / (1 + p2);
			P1000_0 = p[1] / (1 + p2);
			P0100_0 = p[2] / (1 + p2);
			P0010_0 = p[3] / (1 + p2);
			P1100_0 = p[4] / (1 + p2);
			P1010_0 = p[5] / (1 + p2);
			P0110_0 = p[6] / (1 + p2);
			P1110_0 = p[7] / (1 + p2);
			P0001_0 = p[8] / (1 + p2);
			P1001_0 = p[9] / (1 + p2);
			P0101_0 = p[10] / (1 + p2);
			P0011_0 = p[11] / (1 + p2);
			P1101_0 = p[12] / (1 + p2);
			P1011_0 = p[13] / (1 + p2);
			P0111_0 = p[14] / (1 + p2);
			P1111_0 = 1  / (1 + p2);


		}

	~IndirectEstimation()
	{
		free(b);
		free(vP_Theta);
		free(b_0);
		free(u2);
		free(u2_1);
		free(u3);
		free(u_x);
		int i;
		for(i=0;i<k;i++)
		{
			free(u16[i]);
			free(u17[i]);
			free(u14[i]);
			free(u14a[i]);
			free(u14b[i]);
			free(u15[i]);
			free(u15a[i]);
		}
		free(u16);
		free(u17);
		free(u14);
		free(u14a);
		free(u14b);
		free(u15);
		free(u15a);
	}

	double** calculate_inverse(double** m, int k) {
		double* m1 = (double*)malloc(sizeof(double) * k * k);
		int i;

		for(i=0;i<k;i++)
		{
			int j;
			for(j=0;j<k;j++)
			{
				m1[i*k + j] = m[i][j];
			}
		}
		

	    int lda, arows, acols;
	    int* ipiv = (int*)malloc(sizeof(int) * k * k);
	    int info;
	    arows=k;
	    acols=k;
	    // Set the leading dimension of A
	    lda = k;
		    // Set the dimension of the "workspace" array WORK
	    // see http://www.netlib.org/lapack-dev/lapack-coding/program-style.html#workspace
	    int lwork=1000;
	    double* work = (double*)malloc(sizeof(double) * lwork);
	    // Allocate memory for arrays:
	    // Pivot indices array
	    memset(ipiv, 0, sizeof(int)*k*k);
	    // Work (workspace) array
	    memset(work, 0, sizeof(double)*lwork);

		dgetrf_( &arows, &acols, m1, &lda, ipiv, &info );
		
		if (!(info == 0))
			throw RuntimeException("dgetrf failed");
		
		
		dgetri_ (&k, m1, &lda, ipiv, work, &lwork, &info);
		
		
		if (!(info == 0))
			throw new RuntimeException("dgetri failed");

	    free(work);
		free(ipiv);
		
		
		double** m2 =(double**)malloc(sizeof(double*) * k);
		for(i=0;i<k;i++)
			m2[i] = (double*)malloc(sizeof(double) * k);
	
		
		for(i=0;i<k;i++)
		{
			int j;
			for(j=0;j<k;j++)
				m2[i][j] = m1[k * i + j];
		}
		
	
		free(m1);
		return m2;
	}
	
	
	public : int Get_size()
	{
		return n_size;
	}
	
	private : double sum(double* x, int Min, int Max)
	{
		int i;
		double result = 0;
		for(i=Min;i<=Max;i++)
			result += x[i];
		return result;
	}
	
	private : double sum_2(double* x, int Min, int Max)
	{
		int i;
		double result = 0;
		for(i=Min;i<=Max;i++)
			result += x[i] * x[i];
		return result;
	}

	private : double sum_3(double* x, int Min, int Max)
	{
		int i;
		double result = 0;
		for(i=Min;i<=Max;i++)
			result += x[i] * x[i] * x[i];
		return result;
	}
	
	private : double sum_4(double* x, int Min, int Max)
	{
		int i;
		double result = 0;
		for(i=Min;i<=Max;i++)
			result += x[i] * x[i] * x[i] * x[i];
		return result;
	}
	
	private : double sum_cross(double* x, double* y, int Min, int Max)
	{
		int i;
		double result;
		for(i=Min;i<=Max;i++)
			result += x[i] * y[i];
		return result;
	}	
	

	private : double* Num1Derivative_1(double* vP) throw(RuntimeException)
/* 	Calculates derovatives (numerically) of the indirect inference criterion function 
* 	In: parameter values
*	Out: vector of derivatives
*/
	{
	    int i;
	    double p, h, fm, fp, d, hr, right, left, hl;

	    
	    int size = cp * sizeof(double);
	    double* v = (double*)malloc(size);

		double f = Function(vP);

	    for (i = 0; i < cp; i++)    // get 1st derivative by central difference
	    {
	        p = (double)vP[i];

	        h = 1e-4;//dFiniteDiff1(p);
	        d = p + h;

	        vP[i] = d;
			hr = h;
			hl = h;
	
			try {

				right = Function(vP);
				
				fp = right;
				vP[i] = p;
				while ((abs(fp - f) > 100 * h) || std::isnan(fp))
				{
					h = h / 2;
					cout << h << endl;
					fp = Function(vP);
					vP[i] = p;
				}
				v[i] = (fp-f) / h;
		
		    }
			catch (RuntimeException &e)
			{
				try {
					left = Function(vP);
					fm = left;
					vP[i] = p;                         // restore original parameter
					try
					{
						f = Function(vP);	      // not: try to get it
						 v[i] = (f - fm) / hl; // take left difference
					}
					catch (RuntimeException &e1) {
						throw RuntimeException("Unable to evaluate numerical derivative");
					}

				}
				catch (RuntimeException &e2) {
					throw RuntimeException("Unable to evaluate numerical derivative");
				}
			}
	    } // for loop end

		return v;
	}

	public : virtual double* dFunction(double* vP)  throw(RuntimeException)   {

		printing = false;
		double* result;
		try {
			result =  Num1Derivative_1(vP);
		}
		catch (RuntimeException &e)
		{
			throw RuntimeException(e.ExceptionText);
		}
		printing = true;
		return result;
	}

	public: void MultivariateFunction(double* f, double* variable_list, int n_size) throw(RuntimeException)  
	{

		try {
			Initialize(variable_list);
		}
		catch (RuntimeException &e) {
			cout << "MultivariateFunction failed" << endl;
			exit(-1);
		}

		double* avp;

			HazardsFunction hazards_function(T_1_totaal_a, T_1_totaal_b, D_1_totaal_a, D_1_totaal_b, D_2_totaal, T_2_totaal,
					H2_totaal, H4_totaal, X, vP_Theta, n, k, cp1_indirect, variables, myid, numprocs, status,
					U_spells_totaal, E_spells_totaal, G2, G3, G4, Min, Max, Lambda_H);


			int i_1 = 0;

			double* vp1 = (double*)malloc(sizeof(double) * cp_indirect);

			for(int i=0;i<cp1_indirect;i++)
			{
				if (variables[i])
				{
					vp1[i_1] = vP_Theta[i];
					i_1++;
				}
			}


			if (printing)
			{
				hazards_function.Set_s_mxIter(2000);
				hazards_function.Set_MaxControl(1E-4, 1E-4);
			}
			else
			{
				hazards_function.Set_s_mxIter(N1);

				hazards_function.Set_MaxControl(1E-11, 1E-11);
			}


			hazards_function.Set_Print(0);
			avp =  hazards_function.MaxBFGS(vp1);

			double* weights = new double[n];
			double hulp_weights = 0;
			for(int i=Min;i<=Max;i++)
			{
				double hulp = (exp(H4_totaal[i][0] / Lambda_H) / (1 + exp(H4_totaal[i][0] / Lambda_H)));
				weights[i] = 1 - hulp;
				hulp_weights += weights[i];
			}


			if (!myid)
			{
				double hulp1_weights;
				for(int i=1;i<numprocs;i++)
				{
					MPI_Recv(&hulp1_weights, 1, MPI_DOUBLE, i, 1, MPI_COMM_WORLD, &status);
					hulp_weights += hulp1_weights;
				}

				for(int i=1;i<numprocs;i++)
				{
					MPI_Send(&hulp_weights, 1, MPI_DOUBLE, i, 1, MPI_COMM_WORLD);
				}
			}
			else
			{
				MPI_Send(&hulp_weights, 1, MPI_DOUBLE, 0, 1, MPI_COMM_WORLD);
				MPI_Recv(&hulp_weights, 1, MPI_DOUBLE, 0, 1, MPI_COMM_WORLD, &status);
			}

			hulp_weights /= n;

			for(int i=Min;i<=Max;i++)
			{
				weights[i] /= hulp_weights;
			}

			double* b1;
			double* b2;
			double* b3;
			double* b4;


				b1 = bereken_b(b_0, 6.779259, weights, myid, numprocs, status);
				b2 = bereken_b(b2_0,  7.352602, weights, myid, numprocs, status);
				b3 = bereken_b(b3_0,  7.660664, weights, myid, numprocs, status);
				b4 = bereken_b(b4_0,  7.916261 , weights, myid, numprocs, status);

		delete [] weights;
		penalty_function1(f, avp, b2, b3, b4);

		if (myid == 0)
			cout << f[0] << endl;		    	
			
		free(vp1);
		free(avp);
		free(b2);
		free(b3);
		free(b4);
				
		Finalize();
	}


	private :  double calculate_cross(double** x, int cp, int k, int l)
	{
		int i;
		double result = 0;
		for(i=0;i<n; i++)
		{
			result += x[k][i] * x[l][i] / n;
		}
		return result;
	}
	
	private : double* bereken_b(double* vP, double q, double* weights, int myid, int numprocs, MPI_Status status) throw(RuntimeException)
	{
			int i;
			int x_size_2 = HazardsFunction::x_size_2+1;

			double* vp = new double[x_size_2];

			memcpy(vp, vP, x_size_2 * sizeof(double));

			double** x = new double*[n];

			for(i=Min;i<=Max;i++)
			{
					x[i] = new double[x_size_2];
					memcpy(x[i], X[i], x_size_2 * sizeof(double));
			}

			double* y = new double[n];
			double* hulp = zeros(x_size_2);
			double hulp_weights = 0;
			for(i=Min;i<=Max;i++)
			{
				double hulp2 = (q-W[i]) / 1e-2;
				double hulp1 = exp(hulp2);
				hulp1 /= (1+hulp1);
				y[i] = hulp2 > 10? 1: hulp1;


				double* hulp3 = Maximize::multiply(x[i], y[i] * weights[i], x_size_2);
				Maximize::add_to(hulp, hulp3, x_size_2);

				hulp_weights += weights[i];
				delete [] hulp3;
			}

			if (!myid)
			{
				double* hulp1 = new double[x_size_2];
				double hulp1_weights;
				for(i=1;i<numprocs;i++)
				{
					MPI_Recv(hulp1, x_size_2, MPI_DOUBLE, i, 0, MPI_COMM_WORLD, &status);
					MPI_Recv(&hulp1_weights, 1, MPI_DOUBLE, i, 1, MPI_COMM_WORLD, &status);
					Maximize::add_to(hulp, hulp1, x_size_2);
					hulp_weights += hulp1_weights;
				}

				delete [] hulp1;

				for(i=1;i<numprocs;i++)
				{
					MPI_Send(hulp, x_size_2, MPI_DOUBLE, i, 0, MPI_COMM_WORLD);
					MPI_Send(&hulp_weights, 1, MPI_DOUBLE, i, 1, MPI_COMM_WORLD);
				}
			}
			else
			{
				MPI_Send(hulp, x_size_2, MPI_DOUBLE, 0, 0, MPI_COMM_WORLD);
				MPI_Send(&hulp_weights, 1, MPI_DOUBLE, 0, 1, MPI_COMM_WORLD);
				MPI_Recv(hulp, x_size_2, MPI_DOUBLE, 0, 0, MPI_COMM_WORLD, &status);
				MPI_Recv(&hulp_weights, 1, MPI_DOUBLE, 0, 1, MPI_COMM_WORLD, &status);
			}


			for(i=0;i<x_size_2;i++)
			{
				if ((hulp[i] > hulp_weights-1e-9) || (hulp[i] < 1e-9))
				{
					cout << "DR falied" << endl;
					throw RuntimeException("DR failed - perfect prediction");
				}
			}
			Logit logit(y, x, weights, n, x_size_2, myid, numprocs, status);
			logit.Set_s_mxIter(100);

			//if (myid > 0)
				logit.Set_Print(0);

			logit.Set_MaxControl(5e-2,5e-2);
			double* avp = logit.MaxBFGS(vp);

			for(i=Min;i<=Max;i++)
			{
				delete [] x[i];
			}

			delete [] x;
			delete [] y;
			return avp;
	}

	private : double* bereken_b2(int myid, int m) {
		int i;
		double* ba = (double*)malloc(sizeof(double)*HazardsFunction::x_size);
		double** x = transpose(X, n, Min, Max, HazardsFunction::x_size);
		double* w1 = (double*)malloc(sizeof(double)*n);
	
		for(i=Min;i<=Max;i++)
		{
			w1[i] = 1.;
			
			int j;
			for(j = 0;j<m;j++)
				w1[i] = w1[i] * W[i];
		}
		
		int k1 = HazardsFunction::x_size; 
	
	
		double** x2 = (double**)malloc(sizeof(double) * (k1));
		double* xt = (double*)malloc(sizeof(double) * (k1));
		for(i=0;i<k1;i++)
			x2[i] = (double*)malloc(sizeof(double)*(k1));			
			
		for(i=0;i<(k1);i++)
		{
			double* hulp_x_i = (double*)malloc(sizeof(double) * n);
			memcpy(hulp_x_i, x[i], sizeof(double)* n);			
			int j;
			for(j=0;j<(k1);j++)
			{
				double* hulp_x_j = (double*)malloc(sizeof(double) * n);
				memcpy(hulp_x_j, x[j], sizeof(double)* n);
				
				double hulp = internal_product(hulp_x_i, hulp_x_j, Min, Max);
				x2[i][j] = hulp;
				free(hulp_x_j);
			}
			free(hulp_x_i);
		}
		
				
		for(i=0;i<k1;i++)
		{
			double* hulp_x_i = (double*)malloc(sizeof(double) * n);
			memcpy(hulp_x_i, x[i], sizeof(double)* n);
			double hulp = internal_product(hulp_x_i, w1, Min, Max);
			xt[i] = hulp;		
			free(hulp_x_i);
		}
			
		
		int tag = 9;
		if (myid == 0)
		{	
			for(i=1;i<numprocs;i++)
			{
				int j;
				double** x2_hulp = (double**)malloc((k1) * sizeof(double*));
					
				for(j=0;j<k1;j++)
				{
					double* hulp1 = (double*)malloc((k1) * sizeof(double));
					MPI_Recv(hulp1, k1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD, &status);
					x2_hulp[j] = hulp1;
				}	
				
				add_to(x2, x2_hulp, k1);

				double *xt_hulp =  (double*)malloc((k1) * sizeof(double));
				MPI_Recv(xt_hulp, k1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD, &status);
				add_to(xt, xt_hulp, k1);
				for(j=0;j<k1;j++)
					free(x2_hulp[j]);
				free(x2_hulp);
				free(xt_hulp);
			}
				
			double** x2_inv;
				
			try {
				x2_inv = calculate_inverse(x2, k1);
			}
			catch (RuntimeException &e) {
				cout << "singular" << endl;
			}
								
			for(i=0;i<k1;i++)
			{	
				double hulp = internal_product(x2_inv[i], xt, k1);
				ba[i] = hulp;		
			}
			
	
			for(i=0;i<k1;i++)
				free(x2_inv[i]);	
			
			free(x2_inv);
			
			
			for(i=1;i<numprocs;i++)
				MPI_Send(ba, k1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
		
		}
		else
		{				
			int j;
			for(j=0;j<k1;j++)
			{
				MPI_Send(x2[j], k1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD);
			}
			MPI_Send(xt, k1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD);
							
			MPI_Recv(ba, k1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
		}		
			
		
		for(i=0;i<k1;i++)
		{
			free(x2[i]);
			
			free(x[i]);
		}
	
		
		free(x);
		free(x2);
		free(xt);
		free(w1);
		
		return ba;
	}
	
	private : double penalty_function(double* avp, double* b1, double* b2, double* b3, double* b4)
	/* Calculates the penalty function for indirect inference 
	*/
	{
		int i;
		double total = 0;
		int i_1 = 0;
		int k1 = HazardsFunction::x_size;

		for(i=0;i<3+4*k1+29;i++)
		{
			if (variables[i])
			{					
				if (i<=2)
					total -= (avp[i_1] - vP_Theta[i]) * (avp[i_1] - vP_Theta[i]) * 10;
				else
				{
					if ((i== 3) || (i==15) || (i == 27) || (i==39))
					{
						total -= (avp[i_1] - vP_Theta[i]) * (avp[i_1] - vP_Theta[i]) * 1000;
					}
					else
						total -= (avp[i_1] - vP_Theta[i]) * (avp[i_1] - vP_Theta[i]);
				}
				i_1++;
			}

		}

		double* vp_theta_hulp = (double*)malloc(cp1_indirect * sizeof(double));
		memcpy(vp_theta_hulp, vP_Theta, cp1_indirect * sizeof(double));
			
		i_1 = 0;
			
		for(i=0;i<cp1_indirect;i++)
		{
			if (variables[i])
			{
				vp_theta_hulp[i] = avp[i_1];
				i_1++;
			}
		}			
			
		vector<double> p;
		double p2 = 0;

		for(i=3+4*k1+29;i<=3+4 * k1+43;i++)
		{
			double p1 = exp(vp_theta_hulp[i]);
			p.push_back(p1);
			p2 += p1;
		}

		P0000 = p[0] / (1 + p2);
		P1000 = p[1] / (1 + p2);
		P0100 = p[2] / (1 + p2);
		P0010 = p[3] / (1 + p2);
		P1100 = p[4] / (1 + p2);
		P1010 = p[5] / (1 + p2);
		P0110 = p[6] / (1 + p2);
		P1110 = p[7] / (1 + p2);
		P0001 = p[8] / (1 + p2);
		P1001 = p[9] / (1 + p2);
		P0101 = p[10] / (1 + p2);
		P0011 = p[11] / (1 + p2);
		P1101 = p[12] / (1 + p2);
		P1011 = p[13] / (1 + p2);
		P0111 = p[14] / (1 + p2);
		P1111 = 1  / (1 + p2);

		
		free(vp_theta_hulp);
		
	
		total -= (P0000-P0000_0) * (P0000-P0000_0);
		total -= (P1000-P1000_0) * (P1000-P1000_0);
		total -= (P0100-P0100_0) * (P0100-P0100_0);
		total -= (P0010-P0010_0) * (P0010-P0010_0);
		total -= (P1100-P1100_0) * (P1100-P1100_0);
		total -= (P1010-P1010_0) * (P1010-P1010_0);
		total -= (P0110-P0110_0) * (P0110-P0110_0);
		total -= (P1110-P1110_0) * (P0110-P1110_0);
		total -= (P0001-P0001_0) * (P0001-P0001_0);
		total -= (P1001-P1001_0) * (P1001-P1001_0);
		total -= (P0101-P0101_0) * (P0101-P0101_0);
		total -= (P0011-P0011_0) * (P0011-P0011_0);
		total -= (P1101-P1101_0) * (P1101-P1101_0);
		total -= (P1011-P1011_0) * (P1011-P1011_0);
		total -= (P0111-P0111_0) * (P0111-P0111_0);
		
		if (b4[0] > 13)
			return -1e+12;
		for(i=0;i<HazardsFunction::x_size_2+1;i++)
			total -= (b1[i] - b_0[i]) * (b1[i] - b_0[i]) / 4;

		for(i=0;i<HazardsFunction::x_size_2+1;i++)
			total -= (b2[i] - b2_0[i]) * (b2[i] - b2_0[i]) / 4;
		
		for(i=0;i<HazardsFunction::x_size_2+1;i++)
			total -= (b3[i] - b3_0[i]) * (b3[i] - b3_0[i]) / 4;
		
		for(i=0;i<HazardsFunction::x_size_2+1;i++)
			total -= (b4[i] - b4_0[i]) * (b4[i] - b4_0[i]) / 4;

		return total;
	}

	private : double penalty_function1(double* f, double* avp, double* b2, double* b3, double* b4)
/* Calculates penalty function for indirect inference
*/
	{
		int i;
		double total = 0;
		int i_1 = 0;
		int k1 = HazardsFunction::x_size;
		for(i=0;i<3+3*k1+28;i++)
		{
			if (variables[i])
			{					
				double hulp = mhess[i][i];
				if ((i <= 3+4 * HazardsFunction::x_size))
							f[i_1] = avp[i_1];
						else
						{
							double hulp_avp = exp(avp[i_1]);
							double hulp_vp = exp(vP_Theta[i]);
							
							f[i_1] = hulp_avp;
						}
						i_1++;
			}
		}

	
			
		vector<double> p;
		double p2 = 0;
		double* vp_theta_hulp = (double*)malloc(cp1_indirect * sizeof(double));
		memcpy(vp_theta_hulp, vP_Theta, cp1_indirect * sizeof(double));
			
		

		for(i=3+3*k1+28;i<=3+3 * k1+43;i++)
		{
			double p1 = exp(vp_theta_hulp[i]);
			p.push_back(p1);
			p2 += p1;
		}


		P0000 = p[0] / (1 + p2);
		P1000 = p[1] / (1 + p2);
		P0100 = p[2] / (1 + p2);
		P0010 = p[3] / (1 + p2);
		P1100 = p[4] / (1 + p2);
		P1010 = p[5] / (1 + p2);
		P0110 = p[6] / (1 + p2);
		P1110 = p[7] / (1 + p2);
		P0001 = p[8] / (1 + p2);
		P1001 = p[9] / (1 + p2);
		P0101 = p[10] / (1 + p2);
		P0011 = p[11] / (1 + p2);
		P1101 = p[12] / (1 + p2);
		P1011 = p[13] / (1 + p2);
		P0111 = p[14] / (1 + p2);
		P1111 = 1  / (1 + p2);

		free(vp_theta_hulp);
	
		double p_hulp[15] = {P0000, P1000, P0100, P0010, P1100, P1010, P0110,
				P0001, P1001, P0101, P0011, P1101, P1011, P0111};
		
		
		int j = 0;
		for(i=3+4*k1+28;i<=3+4*k1+43;i++)
		{			
			if (variables[i])
			{
				f[i_1] = p_hulp[j];
				i_1++;
			}
			j++;
		}
		

		memcpy(&f[cp_indirect], b, sizeof(double) * (k1));
		memcpy(&f[cp_indirect+k1], b2, sizeof(double) * k1);
		memcpy(&f[cp_indirect+2*k1], b3, sizeof(double) * k1);
		memcpy(&f[cp_indirect+3*k1], b4, sizeof(double) * k1);

		return total;
	}

	private : void do_printing(double* avp, double* b1, double* b2, double* b3, double* b4)
/* Prints intermediate values information
*/
	{
		//return;
		int k1 = HazardsFunction::x_size;
		int i;
	

		int i_1 = 0;
		for(i=0;i<3+4*k1+29;i++)
		{

			if (variables[i])
			{
				//double hulp = mhess[i][i];
				//if ((i <= 3+4*k1))
					if (i <= 2)
						cout << avp[i_1] << "\t" << vP_Theta[i] <<"\t" <<  (avp[i_1] - vP_Theta[i]) * (avp[i_1] - vP_Theta[i]) * 10 << "\t"  << endl;
					else
					{
						if ((i==15) ||(i==27))
							cout << avp[i_1] << "\t" << vP_Theta[i] <<"\t" <<  (avp[i_1] - vP_Theta[i]) * (avp[i_1] - vP_Theta[i]) * 10 << "\t"  << endl;
						else
							cout << avp[i_1] << "\t" << vP_Theta[i] <<"\t" <<  (avp[i_1] - vP_Theta[i]) * (avp[i_1] - vP_Theta[i])  << "\t"  << endl;
					}
				i_1++;
			}
		}

		cout << "P-values:" << endl;
	
		cout << P0000 << "\t" << P0000_0 << "\t"  << (P0000-P0000_0) * (P0000-P0000_0) * mhess[3 + 3 * k1 + 28][3 + 3 * k1 + 28] << endl;
		cout << P1000 << "\t" << P1000_0 << "\t" << (P1000-P1000_0) * (P1000-P1000_0) * mhess[3 + 3 * k1 + 29][3 + 3 * k1 + 29] << endl;
		cout  << P0100 << "\t" << P0100_0 << "\t" << (P0100-P0100_0) * (P0100-P0100_0) * mhess[3 + 3 * k1 + 30][3 + 3 * k1 + 30] << endl;
		cout  << P0010 << "\t" << P0010_0 << "\t" << (P0010-P0010_0) * (P0010-P0010_0) * mhess[3 + 3 * k1 + 31][3 + 3 * k1 + 31] << endl;
		cout  << P1100 << "\t" << P1100_0 << "\t" << (P1100-P1100_0) * (P1100-P1100_0) * mhess[3 + 3 * k1 + 32][3 + 3 * k1 + 32] << endl;
		cout  << P1010 << "\t" << P1010_0 << "\t" << (P1010-P1010_0) * (P1010-P1010_0) * mhess[3 + 3 * k1 + 33][3 + 3 * k1 + 33] << endl;
		cout  << P0110 << "\t" << P0110_0 << "\t" << (P0110-P0110_0) * (P0110-P0110_0) * mhess[3 + 3 * k1 + 34][3 + 3 * k1 + 34] << endl;
		cout << P0001 << "\t" << P0001_0 << "\t"  << (P0001-P0001_0) * (P0001-P0000_0) * mhess[3 + 3 * k1 + 28][3 + 3 * k1 + 28] << endl;
		cout << P1001 << "\t" << P1001_0 << "\t" << (P1001-P1001_0) * (P1001-P1001_0) * mhess[3 + 3 * k1 + 29][3 + 3 * k1 + 29] << endl;
		cout  << P0101 << "\t" << P0101_0 << "\t" << (P0101-P0101_0) * (P0101-P0101_0) * mhess[3 + 3 * k1 + 30][3 + 3 * k1 + 30] << endl;
		cout  << P0011 << "\t" << P0011_0 << "\t" << (P0011-P0011_0) * (P0011-P0011_0) * mhess[3 + 3 * k1 + 31][3 + 3 * k1 + 31] << endl;
		cout  << P1101 << "\t" << P1101_0 << "\t" << (P1101-P1101_0) * (P1101-P1101_0) * mhess[3 + 3 * k1 + 32][3 + 3 * k1 + 32] << endl;
		cout  << P1011 << "\t" << P1011_0 << "\t" << (P1011-P1011_0) * (P1011-P1011_0) * mhess[3 + 3 * k1 + 33][3 + 3 * k1 + 33] << endl;
		cout  << P0111 << "\t" << P0111_0 << "\t" << (P0111-P0111_0) * (P0111-P0111_0) * mhess[3 + 3 * k1 + 34][3 + 3 * k1 + 34] << endl;


		for(i=0;i<HazardsFunction::x_size_2+1;i++)
			cout << b1[i]  << "\t" << b_0[i] << "\t"<<(b1[i] - b_0[i])  * (b1[i] - b_0[i]) / 100  << endl;

		for(i=0;i<HazardsFunction::x_size_2+1;i++)
				cout << b2[i]  << "\t" << b2_0[i] << "\t"<<(b2[i] - b2_0[i])  * (b2[i] - b2_0[i]) /100 << endl;

		for(i=0;i<HazardsFunction::x_size_2+1;i++)
				cout << b3[i]  << "\t" << b3_0[i] << "\t"<<(b3[i] - b3_0[i])  * (b3[i] - b3_0[i])/100 << endl;

		for(i=0;i<HazardsFunction::x_size_2+1;i++)
				cout << b4[i]  << "\t" << b4_0[i] << "\t"<<(b4[i] - b4_0[i])  * (b4[i] - b4_0[i])/100 << endl;

	}
	
	public : double Function(double* variable_list) throw(RuntimeException) {
/* Criterion function for the Indirect Inference Method 
	In: variable_list, vector of the parameters
	Out: Criterion value
*/		
		double total = 0;
		int i;


		if (myid == 0)
		{
			SaveData save;
			try {
				save.SaveTxt_vector("param_last23.txt", variable_list, cp);
			}
			catch (IOException &e) {
				cerr << "Could not write to " << e.filename;
			}
		}
		int check;
		
		try {
			Initialize(variable_list);
			check = 0;
		}
		catch (RuntimeException &e) {
			check = 1;
		}	

		if (myid == 0)
		{
		
			for (i=1;i<numprocs;i++)
			{
				int check1;
				MPI_Recv(&check1, 1, MPI_INT, i, 0, MPI_COMM_WORLD, &status);
				check += check1;
			}
			
			for (i=1;i<numprocs;i++)	
			{
				MPI_Send(&check, 1, MPI_INT, i, 1, MPI_COMM_WORLD);
			}
		}
		else
		{
			MPI_Send(&check, 1, MPI_INT, 0, 0, MPI_COMM_WORLD);
			MPI_Recv(&check, 1, MPI_INT, 0, 1, MPI_COMM_WORLD, &status);
		}
		
		if (check > 0)
		{
			cout << "Sampling failed" << endl;
			return log(0.);
		}
		
		double* avp;
		
		HazardsFunction hazards_function(T_1_totaal_a, T_1_totaal_b, D_1_totaal_a, D_1_totaal_b, D_2_totaal, T_2_totaal,
				H2_totaal, H4_totaal, X, vP_Theta, n, k, cp1_indirect, variables, myid, numprocs, status,
				U_spells_totaal, E_spells_totaal, G2, G3, G4, Min, Max, Lambda_H);


		int i_1 = 0;

		double* vp1 = (double*)malloc(sizeof(double) * cp_indirect);
	
		for(i=0;i<cp1_indirect;i++)
		{
			if (variables[i])
			{
				vp1[i_1] = vP_Theta[i];
				i_1++;
			}
		}
	
	
		hazards_function.Set_Print(0);
		if (printing)
		{
			hazards_function.Set_s_mxIter(2000);
			hazards_function.Set_MaxControl(5E-2, 5E-2);
		}
		else
		{
			hazards_function.Set_s_mxIter(N1);			
			hazards_function.Set_MaxControl(1E-11, 1E-11);
		}

		try {
			avp =  hazards_function.MaxBFGS(vp1);
		}
		catch (RuntimeException &e)
		{
			cout << "reached line 1372" << endl;
			return log(0.);
		}

		if (myid == 0)
			cout << "Number of iterations:\t"<< hazards_function.N << endl;

	
		if ((printing) && !((hazards_function.Get_ReturnValue() == MAX_CONV) || (hazards_function.Get_ReturnValue() == MAX_WEAK_CONV)))
		{
			if (myid == 0)
				cout << "no convergence" << endl;
			
			free(avp);
			free(vp1);
			Finalize();
			return log(0.);
		}

		double* weights = new double[n];
		double hulp_weights = 0;
		for(i=Min;i<=Max;i++)
		{
			double hulp = (exp(H4_totaal[i][0] / Lambda_H) / (1 + exp(H4_totaal[i][0] / Lambda_H)));
			weights[i] = 1 - hulp;
			hulp_weights += weights[i];
		}


		if (!myid)
		{
			double hulp1_weights;
			for(i=1;i<numprocs;i++)
			{
				MPI_Recv(&hulp1_weights, 1, MPI_DOUBLE, i, 1, MPI_COMM_WORLD, &status);
				hulp_weights += hulp1_weights;
			}

			for(i=1;i<numprocs;i++)
			{
				MPI_Send(&hulp_weights, 1, MPI_DOUBLE, i, 1, MPI_COMM_WORLD);
			}
		}
		else
		{
			MPI_Send(&hulp_weights, 1, MPI_DOUBLE, 0, 1, MPI_COMM_WORLD);
			MPI_Recv(&hulp_weights, 1, MPI_DOUBLE, 0, 1, MPI_COMM_WORLD, &status);
		}

		hulp_weights /= n;

		for(i=Min;i<=Max;i++)
		{
			weights[i] /= hulp_weights;
		}

		double* b1;
		double* b2;
		double* b3;
		double* b4;

		try {
			b1 = bereken_b(b_0, 6.779259, weights, myid, numprocs, status);
			b2 = bereken_b(b2_0,  7.352602, weights, myid, numprocs, status);
			b3 = bereken_b(b3_0,  7.660664, weights, myid, numprocs, status);
			b4 = bereken_b(b4_0,  7.916261 , weights, myid, numprocs, status);
		}
		catch (RuntimeException &e)
		{
			cout << "reached line 1469" << endl;
			return log(0.);
		}

		delete [] weights;

		total += penalty_function(avp, b1, b2, b3, b4);
	

		if (((total) > MaxValue) && (myid == 0))
		{
			SaveData save;

			try {
				save.SaveTxt_vector(intermed_string, variable_list, cp);
			}
			catch (IOException &e) {
				cerr << "Could not open file " << e.filename << endl;
			}
		}
		
	
		if (((total) >= MaxValue) && printing)
		{
			  MaxValue = total;
			N1 = hazards_function.N;
			
		
		}

		if ((printing) && (myid == 0))
			do_printing(avp,b1, b2, b3, b4);
		

		if (myid == 0)
			cout << "Function value:\t"<< (total) / rss_dividor << endl;

		free(vp1);
		free(avp);
		delete [] b1;
		free(b2);
		free(b3);
		free(b4);

		Finalize();


		return (total) / rss_dividor;
	}



	private : void Finalize()
	{
		
		int i;
		
	   for(i=Min;i<=Max;i++)
	    {
	    	free(H2_totaal[i]);
	    	free(H4_totaal[i]);
	    	free(T_1_totaal_a[i]);
	    	free(T_1_totaal_b[i]);
	    	free(D_1_totaal_a[i]);
	    	free(D_1_totaal_b[i]);
	    	free(T_2_totaal[i]);
	    	free(D_2_totaal[i]);
	    	free(X[i]);
	    }
	    
	    
	    free(H2_totaal);
	    free(H4_totaal);
	    free(T_1_totaal_a);
	    free(T_1_totaal_b);
	    free(D_1_totaal_a);
	    free(D_1_totaal_b);
	    free(T_2_totaal);
	    free(D_2_totaal);
	    free(X);
	    free(U_spells_totaal);
	    free(E_spells_totaal);
	    free(G1);
	    free(G2);
	    free(G3);
	    free(G4);
	    free(G5);	 
	    free(W);				  
	}
	
	private : void Initialize(double* variable_list) throw(RuntimeException)
/*
	Initialization of the Indirect inference method - Sampling
*/
	{

		int k1 = HazardsFunction::x_size-1;
		double* Beta1 = (double*)malloc(sizeof(double) * k1);
		double* Beta2 = (double*)malloc(sizeof(double)*k1);
		double* Beta_W = (double*)malloc(sizeof(double)*k1);
		double* Beta_U = (double*)malloc(sizeof(double)*k1);
		int x_size_2 = HazardsFunction::x_size_2;

		int tag = 9;
		double mu_v1, mu_v2, sigma_v1, sigma_v2, Rho_1, P, Mu_u, B, Sigma_u, K, Mu, Sigma, Tau;
		if (myid == 0)
		{

			memcpy(Beta1, variable_list, HazardsFunction::x_size_2 * sizeof(double));
			if (x_size_2 < k1)
				memcpy(&Beta1[x_size_2], Maximize::zeros(k1-x_size_2), (k1-x_size_2) * sizeof(double));
			memcpy(Beta2, &variable_list[x_size_2], x_size_2 * sizeof(double));
			if (x_size_2 < k1)
				memcpy(&Beta2[x_size_2], Maximize::zeros(k1-x_size_2), (k1-x_size_2) * sizeof(double));

			mu_v1 = variable_list[2 * x_size_2];
			mu_v2 = variable_list[2 * x_size_2 + 1];
			sigma_v1 = exp(variable_list[2 * x_size_2 + 2]);
			sigma_v2 = exp(variable_list[2 * x_size_2 + 3]);
					
			Rho_1 = -1 + 2 * exp(variable_list[3 * x_size_2 + 8]) / (1 + exp(variable_list[3 * x_size_2 + 8]));
			P = exp(variable_list[3*x_size_2 + 6]) / (1 + exp(variable_list[3 * x_size_2 + 6]));
		
			Mu_u = exp(variable_list[2 * x_size_2 + 4]);
			Mu_u -= 0.5;
			memcpy(Beta_U, &variable_list[2 *x_size_2+5], sizeof(double) * x_size_2);
			if (x_size_2 < k1)
				memcpy(&Beta_U[x_size_2], Maximize::zeros(k1-x_size_2), (k1-x_size_2) * sizeof(double));
		
			B = variable_list[3 * x_size_2 + 5];
			K = exp(variable_list[3 * x_size_2 +7]);
			Mu = exp(variable_list[3 * x_size_2 + 9]);
		
			memcpy(Beta_W, &variable_list[3 * x_size_2 + 10], x_size_2 * sizeof(double));
			if (x_size_2 < k1)
				memcpy(&Beta_W[x_size_2], Maximize::zeros(k1-x_size_2), (k1-x_size_2) * sizeof(double));
		
		
			Sigma = exp(variable_list[4 * x_size_2 + 10]);
			Tau = exp(variable_list[4 * x_size_2 + 11]);

			int i;
			for(i=1;i<numprocs;i++)
			{
				MPI_Send(Beta1, k1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(Beta2, k1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
			
				MPI_Send(&mu_v1, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&mu_v2, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&sigma_v1, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&sigma_v2, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&Rho_1, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&P, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&Mu_u, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(Beta_U, k1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&B, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&K, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&Mu, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(Beta_W, k1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&Sigma, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
				MPI_Send(&Tau, 1, MPI_DOUBLE, i, tag, MPI_COMM_WORLD);
			}
		}
		else {
			MPI_Recv(Beta1, k1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(Beta2, k1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
		
			MPI_Recv(&mu_v1, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&mu_v2, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&sigma_v1, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&sigma_v2, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&Rho_1, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&P, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&Mu_u, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(Beta_U, k1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&B, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&K, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			
	
			MPI_Recv(&Mu, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(Beta_W, k1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&Sigma, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
			MPI_Recv(&Tau, 1, MPI_DOUBLE, 0, tag, MPI_COMM_WORLD, &status);
		}
		
		Sigma_u = 1;
	
		int* n_employment_spells = (int*)malloc(sizeof(int) * 2);
		int* n_unemployment_spells = (int*)malloc(sizeof(int) * 2);

		n_employment_spells[0] = n;
		n_employment_spells[1] = n;
		n_unemployment_spells[0] = n;
		n_unemployment_spells[1] = n;

		
		try
		{
			Sample sample(Mu, Beta_W, Sigma, B, K, Rho, Beta1, Beta2, mu_v1, mu_v2, sigma_v1,
					sigma_v2, Mu_u, Beta_U, Sigma_u, P, Rho_1, Tau, n_employment_spells, n_unemployment_spells, k, k1,
					u2, u2_1, u3, u_x, u16, u17, u14, u14a, u14b, u15, u15a, u_u_spells,
					u_e_spells, g_all, u_w,
					Min, Max, numprocs);
			



			Lambda_H = sample.Lambda_H;
			H2_totaal = sample.Get_H_1_totaal();
			H4_totaal = sample.Get_H_2_totaal();
			T_1_totaal_a = sample.Get_T_1_totaal_a();
			T_1_totaal_b = sample.Get_T_1_totaal_b();
			D_1_totaal_a = sample.Get_D_1_totaal_a();
			D_1_totaal_b = sample.Get_D_1_totaal_b();
			T_2_totaal = sample.Get_T_2_totaal();
			D_2_totaal = sample.Get_D_2_totaal();
			X = sample.Get_X();
			E_spells_totaal = sample.Get_E_spells_totaal();
			U_spells_totaal = sample.Get_U_spells_totaal();
			G1 = sample.Get_G1();
			G2 = sample.Get_G2();
			G3 = sample.Get_G3();
			G4 = sample.Get_G4();
			G5 = sample.Get_G5();
			W = sample.Get_W();
		
		}
		catch (SampleException &e)
		{
			cout << "Sampling failed" << endl;
			throw RuntimeException("Sampling failed");
		}


		free(Beta1);
		free(Beta2);
		free(Beta_W);
		free(Beta_U);
		free(n_employment_spells);
		free(n_unemployment_spells);
	}
};


#endif

