/*
 * BisectionSolver.cpp
 *
 *  Created on: Jun 4, 2013
 *      Author: Aico van Vuuren
 */

#ifndef BISECTION_INCLUDED
#define BISECTION_INCLUDED

#include <cmath>
#include "maximize.cpp"
using namespace std;



class BisectionSolver  {
    /** Default absolute accuracy. */
	private : double DEFAULT_ABSOLUTE_ACCURACY;
	private : double DEFAULT_MAX_ITER;
	private: double absoluteAccuracy;
	private: int max_iterations;

    /**
     * Construct a solver with default accuracy (1e-6).
     */
    public : BisectionSolver() {
    	DEFAULT_ABSOLUTE_ACCURACY = 1e-6;
    	DEFAULT_MAX_ITER = 10000;
    	this->absoluteAccuracy = DEFAULT_ABSOLUTE_ACCURACY;
    	max_iterations = DEFAULT_MAX_ITER;
    }
    /**
     * Construct a solver.
     *
     * @param absoluteAccuracy Absolute accuracy.
     */
    public : BisectionSolver(double absoluteAccuracy) {
    	DEFAULT_ABSOLUTE_ACCURACY = 1e-6;
    	DEFAULT_MAX_ITER = 10000;
        this->absoluteAccuracy = absoluteAccuracy;
    }

    /**
     * {@inheritDoc}
     */

	public: virtual double value(double variable_list) throw(RuntimeException) = 0;

    void verifyInterval(double min, double max)
    {
    	double fmin = value(min);
    	double fmax = value(max);

    	if (fmin * fmax > 0)
    		throw RuntimeException("Minimum and max value have the same sign");
    }

    private : double midpoint(double x, double y)
    {
    	return x + (y - x)/ 2;
    }

    public : double doSolve(double min_0, double max_0) {
        double min = min_0;
        double max = max_0;
        try {
        	verifyInterval(min, max);
        }
        catch (RuntimeException &e)
        {
        	throw RuntimeException(e.ExceptionText);
        }

        double m;
        double fm;
        double fmin;

        int index = 0;

        while (true) {
        	index ++;
        	if (index > max_iterations)
        	{
        		throw RuntimeException("Maximum number of iterations reached");
        	}
            m = midpoint(min, max);
            fmin = value(min);
            fm = value(m);

            if (fm * fmin > 0) {
                // max and m bracket the root.
                min = m;
            } else {
                // min and m bracket the root.
                max = m;
            }

            if (abs(max - min) <= absoluteAccuracy) {
                m = midpoint(min, max);
                return m;
            }
        }
        return 1.;
    }
};



#endif





