/* Estimating the inequality reduction mobility measure based on
   Atkinson's inequality measures conditional on age using (gaussian)
   WARPing density estimation

   Inputs:  e   (p,1)-vector of parameters for the Atkinson measure
            sy1 (n1,1)-vector, points of density evaluation in the
                first period
            sy2 (n2,1)-vector, same for second period
            sya (na,1)-vector, same for aggregate income
            f1  (n1,k)-matrix, (unconditional) densities at incomes sy1
                and various ages
            f2  (n2,k)-matrix, same for second period
            fa  (na,k)-matrix, same for aggregate income
   Output:      (k,p)-maxtrix of conditional mobility measures, one row per
                age, one column per parameter
   Remarks:     All the inputs except the vector e are returns of the
                WARPing algorithm "warpnrm2.g". If the evaluation points
                are negative they are deleted. */

proc c_msa(e,sy1,sy2,sya,f1,f2,fa);
  local a1,a2,aa,d1,d2,da,i,m1,m2,ma,z;
  d1=sy1[2]-sy1[1];
  d2=sy2[2]-sy2[1];
  da=sya[2]-sya[1];
  f1=f1./(d1.*sumc(f1)');
  f2=f2./(d2.*sumc(f2)');
  fa=fa./(da.*sumc(fa)');
  sy1=sy1+d1/2;
  sy2=sy2+d2/2;
  sya=sya+da/2;
  f1=delif(f1,sy1.<=0); sy1=delif(sy1,sy1.<=0);
  f2=delif(f2,sy2.<=0); sy2=delif(sy2,sy2.<=0);
  fa=delif(fa,sya.<=0); sya=delif(sya,sya.<=0);
  m1=sumc(sy1.*f1.*d1);
  m2=sumc(sy2.*f2.*d2);
  ma=sumc(sya.*fa.*da);
  z=zeros(cols(f1),rows(e));
  i=0;
  do until i==rows(e);
    i=i+1;
    a1=1-sumc(((sy1./m1')^(1-e[i])).*f1.*d1)^(1/(1-e[i]));
    a2=1-sumc(((sy2./m2')^(1-e[i])).*f2.*d2)^(1/(1-e[i]));
    aa=1-sumc(((sya./ma')^(1-e[i])).*fa.*da)^(1/(1-e[i]));
    z[.,i]=1-(ma.*aa)./(m1.*a1+m2.*a2);
  endo;
  retp(z);
endp;
