function [Dg_ineq,Dg_eq] = moments_gradient(theta,J1,J2,KMSoptions)
%% USER-SPECIFIED FUNCTION: Gradient of the moment function
% The moment functions are in the form
%
%       E_P[m(W_i,theta)] = E_P[f(W_i)] + g(theta)
%
% where
%
%       E_P[m_j(W_i,theta)] <= 0 for j = 1,...,J1
%       E_P[m_j(W_i,theta)] = 0  for j = J1+1,...,J1+J2
%
% This function computes Dm(W_i,theta) = Dg(theta).
%
% NOTE: We used the separability assumption above, so that Dm does not
% depend on data.
%
% INPUT:
%   theta         dim_p-by-1 vector of parameters
%
%   J1            Integer number of moment inequalities
%
%   J2.           Integer number of moment equalities
%
%   KMSoptions.   This is a structure of additional inputs.  The user can
%                 add parameters to KMSoptions, say KMSoptions.params,
%                 and call KMSoptions.params in the user-specified
%                 functions.
%                 For example, in the 2x2 entry game, we include the
%                 support for the covariates and the probability that
%                 a particular support point occurs.

% OUTPUT:
%   Dg_ineq     J1-by-dim_p vector of gradients of moment inequalities
%                 Dg_j(theta) for j = 1,...,J1
%
%   Dg_eq       2*J2-by-dim_p vector of gradients of moment equalities
%                 Dg_j(theta) for j = 1,...,J2
%
% Below is a list of examples of moment functions.

% Load ancilliary parameters
uH = KMSoptions.uH;
S = length(uH);
kk = KMSoptions.kk;
xx = KMSoptions.xx;
xs  = size(xx,2);

% Extract parameters
a = theta(1);
g = theta(2);
b = theta(3:end);

% Calculate gradient
l = kron(exp(-xx*b),(uH).^a);
D = (1+g*l).^(-1/g-1);
Grad = [D.*l.*repmat(log(uH),kk,1) ...            % derivative wrt Weibull parameter
		1/g^2*D.*(l*g-log(1+g*l).*(1+g*l)) ...    % derivative wrt Gamma distr. parameter    
	    -kron(xx,ones(S,1)).*repmat(D.*l,1,xs)];              % derivative with respect to beta
ys = length(uH);
Grad1 = Grad; Grad1(1:ys:kk*ys,:) = [];
Grad2 = Grad; Grad2(ys:ys:kk*ys,:) = [];
Dg_ineq = [Grad; -Grad; -Grad1+Grad2; Grad; -Grad];
Dg_eq   = [];
