function [f_ineq,f_eq,f_ineq_keep,f_eq_keep,paired_mom,J1,J2,J3] = moments_w(W,KMSoptions)
%% USER-SPECIFIED FUNCTION: Moment function that depends on data only
% The moment (in)equalities are in the form
%
%       E_P[m(W_i,theta)] = E_P[f(W_i)] + g(theta)
%
% where
%
%       E_P[m_j(W_i,theta)] <= 0 for j = 1,...,J1
%       E_P[m_j(W_i,theta)] = 0  for j = J1+1,...,J1+J2
%
% This function computes the estimator for  E_P[f(W_i)], which is:
% f_j = (1/n)sum_i f_j(W_i).
%
% The user needs to specify this function.  Examples are given below.
%
% INPUT:
%   W.              Data vector W = [w_1, w2, ... , w_K], where w_k is n-by-1
%                   and n is the sample size.
%
%   KMSoptions.     This is a structure of additional inputs.  The user can
%                   add parameters to KMSoptions, say KMSoptions.params,
%                   and call KMSoptions.params in the user-specified
%                   functions.
%                   For example, in the 2x2 entry game, we include the
%                   support for the covariates and the probability that
%                   a particular support point occurs.
%
% OUTPUT:
%   f_ineq      J1-by-1 vector of moment inequalities f_j(W) for j=1,...,J1
%
%   f_eq        2*J2-by-1 vector of moment inequalities f_j(W) for j=1,...,J2
%               Note that we re-write the moment equalities as two moment
%               inequalities.  Thus, we have f_eq = [f(W) ; - f(W)], where
%               f(W) is the vector of moment equalities.
%
%   f_ineq_keep,f_eq_keep      
%               J1-by-1 and 2*J2-by-1 vector of moments to keep.  
%               If empirical moment j is too close to the boundary of the 
%               support of f_j, then we drop this moment in all future 
%               calculations.  The jth entry of f_keep is equal to 1 if we
%               keep f_j, otherwise it is equal to 0.  
%               As a concrete example, suppose that f_j(W_i) is a Bernoulli
%               random variable, so that E_P[f_j(W_i)] bounded between [0,1].
%               If f_j close to 0 or 1, we set f_keep(j,1) = 0.  
%               NB: If this boundary issue is not a problem in your
%               application, set f_keep = ones(J,1).
%
%   paired_mom  J3-by-1 vector of paired moment inequalities.
%               Let k=1,...,J3 by the kth pair of moment inequalities that
%               are highly correlated.  Say that these moment inequalities
%               are indexed by j1,j2 in {1,...,J1}.  Then
%                      paired_mom(j1,1) = paired_mom(j2,1) = k.
%
%   J1          Integer number of moment inequalities
%
%   J2.         Integer number of moment equalities
%
%   J3.         Integer number of pairs of paired moment inequalities.
%               Nb: require that 2*J3 <= J1.
%
% Below is a list of examples of moment functions.

% Threshhold for f_j being to boundary threshhold.
f_keep_threshold  = KMSoptions.f_keep_threshold ;

% Load ancilliary parameters
uH = KMSoptions.uH;
S = length(uH); uH = uH';
kk = KMSoptions.kk;
n = KMSoptions.n;
xx = KMSoptions.xx;
c = KMSoptions.c;

% Load variables
y = W(:,1);
d = W(:,2);
x = W(:,3:end);

% Loop over discrete values of x to calculate moments
M1 = []; M2 = []; M3 = []; M4 = []; M5 = [];
for i = 1:kk
    z = xx(i,:);
    xind = find(all(abs(x-z(ones(n,1),:))<0.0001,2));
    if isempty(xind)
        %warning('Value x=%f not found in the data \n',z) 
        M1 = [M1; -ones(S,1)]; % If value of x not in data, force corresponding
        M2 = [M2; -ones(S,1)]; % inequalities to be satisfied.
        M3 = [M3; -ones(S-1,1)]; 
        M4 = [M4; -ones(S,1)];
        M5 = [M5; -ones(S,1)];
        continue
    else
    xfreq = length(xind);
    ym = y(xind);
    dm = d(xind);
    pd = mean(dm);
    if xfreq ~= 1
        pdy = sum(dm(:,ones(1,S)).*(ym(:,ones(1,S))<=uH(ones(xfreq,1),:)))./xfreq;
        m1 = sum((ym(:,ones(1,S))>uH(ones(xfreq,1),:)))./xfreq;
        m2 = -1 + pdy;
        m3 = pdy(2:end) - pdy(1:end-1);
        m4 = - min([pdy/(pd-c)*(pd>c)+(pd<=c); (pdy+c)/(pd+c); pdy+1-pd]);
        m5 = max([pdy/(pd+c); (pdy-c)/(pd-c)*(pd>c); pdy]);
    else
        pdy = dm(:,ones(1,S)).*(ym(:,ones(1,S))<=uH(ones(xfreq,1),:));
        m1 = (ym(:,ones(1,S))>uH(ones(xfreq,1),:));
        m2 = -1 + pdy;
        m3 = pdy(2:end) - pdy(1:end-1);
        m4 = - min([pdy/(pd-c)*(pd>c)+(pd<=c); (pdy+c)/(pd+c); pdy+1-pd]);
        m5 = max([pdy/(pd+c); (pdy-c)/(pd-c)*(pd>c); pdy]);
    end
    M1 = [M1; m1(:)];
    M2 = [M2; m2(:)];
    M3 = [M3; m3(:)];
    M4 = [M4; m4(:)];
    M5 = [M5; m5(:)];
    end
end
f_ineq = [M1 ; M2 ; M3; M4; M5];
f_eq = [];
paired_mom = [];
J1 = 2*length(uH)*kk + (length(uH)-1)*kk + 2*length(uH)*kk;
J2 = 0;
J3 = 0;
    
% Select moments to keep.  Note that since f_ineq and f_eq are bounded
% between [0,1]. If they are close to the boundary, then we drop 
% them using the 'f_ineq_keep' and 'f_eq_keep' vectors.
f_ineq_keep = zeros(J1,1);
f_eq_keep = zeros(2*J2,1);
UB = 1;
LB = 0;
ind = find( abs(f_ineq) > LB + f_keep_threshold & abs(f_ineq) < UB - f_keep_threshold);
f_ineq_keep(ind,1) = 1;
%ind = find( abs(f_eq) > LB + f_keep_threshold & abs(f_eq) < UB - f_keep_threshold);
%f_eq_keep(ind,1) = 1; 
