function [eff] = deasel(X,Y,ori,rts,prdl)
%
% ======
% deasel
% ======
%
% Selection of a DEA model and computation of the efficiency
% for all observations after selection nondominated observation
% 
%         Usage
%         [eff]=deasel(X,Y,ori,rts,prdl)
%         IN   :
%         ------
%            X   : Matrix of input(s)  (n x p)
%            Y   : Matrix of output(s) (n x k)
%            ori : Orientation of the computation
%                     'I'    = Input orientation
%                     'O'    = output orientation
%            rts : Assumption on returns to scale (text)
%                     'NIRS' = Non increasing rts
%                     'NDRS' = Non decreasing rts
%                     'CRS'  = Constant rts
%                     'VRS'  = Variable rts
%            prdl: Selection of primal or dual approach
%                     'P'    = Primal
%                     'D'    = Dual
%         OUT  :
%         ------
%	Eff    : Vector of efficiency scores (n x 1)
% 
% Use : LP, dea_i_p, dea_o_p, dea_i_d, dea_o_d, fdhsel
% Proposed by L. Simar
% Adapted by P. Vanden Eeckaut (June 1998), D. Mitrut (August 2001)
% up-dated and checked by L. Simar (december 2002)
%
% INSTITUT DE STATISTIQUE - UNIVERSITE CATHOLIQUE DE LOUVAIN

% Note on lp:
% Input inefficiency measures using the DUAL
% Need to input f,A,b
% for solving in x minf'x s.t. Ax<=b
% then call x = lp(f,A,b)
% equality constraints must be writen in first N rows
% then call x=lp(f,A,b,Vlb,Ulb,[],N)


% X matrix n,p of inputs
% Y matrix n,q of outputs
[n,p] = size(X);
[n,q] = size(Y);


[XS,YS,IS] = fdhsel(X,Y,rts);ns=length(IS);
%XS=X;YS=Y;ns=n;IS=[1:n]';

t0=cputime;
eff=[];
% --------------------
% test for orientation
% --------------------
p = [strcmp('I', ori),strcmp('O', ori)];
if sum(p)==0
	disp( ' !!! ori option is not correct use I or O !!!');
	return
end
% ------------------------
% test for return to scale
% ------------------------
p = [strcmp('CRS', rts),strcmp('NIRS', rts),strcmp('NDRS', rts),strcmp('VRS', rts),strcmp('VRS2', rts)];
if sum(p)==0
	disp( ' !!! rts option is not correct use CRS, NIRS, NDRS or VRS !!!');
	return
end
% --------------------
% test for primal dual
% --------------------
p = [strcmp('P', prdl),strcmp('D', prdl)];
if sum(p)==0
	disp( ' !!! prdl option is not correct use P or D !!!');
	return
end
%
%
% Primal
%
if strcmp('P',prdl)
   fprintf('\n * Primal approach \n')
   % Input orientation
	if strcmp('I',ori)
		disp(' * Input orientation');
      disp('   -----------------');
      t0=cputime;
		for k = 1:n
			xeff=dea_i_p(X(k,:)',Y(k,:)',XS,YS,rts);
         eff=[eff;xeff];
         j=sum(find(IS==k));
         if (j~=0)&(abs(xeff-1)>10e-8)
            XS = [XS([1:j-1],:);XS([j+1:ns],:)];
            YS = [YS([1:j-1],:);YS([j+1:ns],:)];
            IS=IS([[1:j-1]';[j+1:ns]']);
            ns=ns-1;
         end
         fprintf(' Unit : %5.0f Eff. : %6.4f \n',[k,xeff]);
      end
      t=cputime-t0;
	end
   % output orientation
   if strcmp('O',ori)
      disp(' * Output orientation');
      disp('   ------------------');
      t0=cputime;
		for k = 1:n
			xeff=dea_o_p(X(k,:)',Y(k,:)',XS,YS,rts);
         eff=[eff;xeff];
         j=sum(find(IS==k));
         if (j~=0)&(abs(xeff-1)>10e-8)
            XS = [XS([1:j-1],:);XS([j+1:ns],:)];
            YS = [YS([1:j-1],:);YS([j+1:ns],:)];
            IS=IS([[1:j-1]';[j+1:ns]']);
            ns=ns-1;
         end
         fprintf(' Unit : %5.0f Eff. : %6.4f \n',[k,xeff]);
      end
      t=cputime-t0;
	end
end
% Dual
if strcmp('D',prdl)
   fprintf('\n * Dual approach \n')
   % Input orientation
   if strcmp('I',ori)
		disp(' * Input orientation');
      disp('   -----------------');
      t0=cputime;
		for k = 1:n
			xeff=dea_i_d(X(k,:)',Y(k,:)',XS,YS,rts);
         eff=[eff;xeff];
         j=sum(find(IS==k));
         if (j~=0)&(abs(xeff-1)>10e-8)
            XS = [XS([1:j-1],:);XS([j+1:ns],:)];
            YS = [YS([1:j-1],:);YS([j+1:ns],:)];
            IS=IS([[1:j-1]';[j+1:ns]']);
            ns=ns-1;
         end
         fprintf(' Unit : %5.0f Eff. : %6.4f \n',[k,xeff]);
      end
      t=cputime-t0;
	end
	% Output orientation
   if strcmp('O',ori)
		disp(' * Output orientation')
      disp('   ------------------')
      t0=cputime;
		for k = 1:n
	 		xeff=dea_o_d(X(k,:)',Y(k,:)',XS,YS,rts);
         eff=[eff;xeff];
         j=sum(find(IS==k));
         if (j~=0)&(abs(xeff-1)>10e-8)
            XS = [XS([1:j-1],:);XS([j+1:ns],:)];
            YS = [YS([1:j-1],:);YS([j+1:ns],:)];
            IS=IS([[1:j-1]';[j+1:ns]']);
            ns=ns-1;
         end 
         fprintf(' Unit : %5.0f Eff. : %6.4f \n',[k,xeff]);
      end
      t=cputime-t0;
	end
end
fprintf('\n Necessary time to compute: %10.4f seconds \n',t);
return
