/***************************************************************/
/*                                                             */
/*     VAR analysis: simulate the marginal density             */
/*                                                             */
/*                                                             */
/*                       Frank Schorfheide                     */
/*                                                             */
/***************************************************************/

/* Filename:  varpost1.g
** created:   05/14/97
*/
library pgraph;
cls;




/* Import data on output growth and inflation: series (nobs,2)
** observations from 1950:I to 1997:IV
*/
nobs   = rows(series);
p      = 4;                  /* number of lags                       */
k      = 1 + 2*p;            /* number of coefficients per equation  */
lagmax = 4;
T0     = 1+2*lagmax+2;
ndraw  = 10000;

/* Set beginning and end dates
*/

T0grid   = 16 | 31 | 51 | 71 | 91 | 111 | 131 | 151 | 171 | 11 | 31 | 51 |91 ;
Tendgrid = (nobs-lagmax) | 51 | 71 | 91 | 111| 131 | 151 | 171 | ones(5,1)*(nobs-lagmax);


/* Start outer loop
*/

margdens = zeros(rows(T0grid),1);

loopct = 1;
do until loopct > 1;

T0   = T0grid[loopct];
Tend = Tendgrid[loopct];

nobs   = rows(series);      
ti     = seqa(1950.0,0.25,nobs);
YY     = series[1+lagmax:nobs,.];
XX     = ones(nobs-lagmax, 1+2*p);
ti     = ti[1+lagmax:nobs,.];

i = 1;
do until i > p;
   XX[.,2+(i-1)*2:1+i*2] = series[lagmax-i+1:nobs-i,.];
   i = i+1;
endo;

nobs = nobs - lagmax;

/* Adjust sample length
*/
YY = YY[1:Tend,.];
XX = XX[1:Tend,.];
ti = ti[1:Tend,.];
nobs = rows(YY);

goto m1;

/* Method 1: use formulae for ln p(Y) derived from regression model
** with improper priors. Subtract ln p(Yt0) for some initialization
** period 0..t0 
*/
M1:
  
   YYT = YY[1:T0,.];
   XXT = XX[1:T0,.];

   /* Compute OLS estimates
   */
   CC_hat  = inv(XX'*XX)*XX'*YY;
   C_hat   = vec(CC_hat);
   SIG_hat = (YY-XX*CC_hat)'*(YY-XX*CC_hat)/nobs;

   CC_hatT  = inv(XXT'*XXT)*XXT'*YYT;
   C_hatT   = vec(CC_hatT);
   SIG_hatT = (YYT-XXT*CC_hatT)'*(YYT-XXT*CC_hatT)/T0;

   kapt = (2*pi)^(-2*(nobs-k)/2)*2^((nobs-k))*sqrt(pi)*gamma((nobs-k)/2)*gamma((nobs-k-1)/2);
   kapt0 =(2*pi)^(-2*(T0-k)/2)*2^((T0-k))*sqrt(pi)*gamma((T0-k)/2)*gamma((T0-k-1)/2);
   pYt = ln(kapt)  - ((nobs-k)/2)*ln(det(nobs*Sig_hat)) - ln(det(XX'*XX));
   PYt0= ln(kapt0) - ((T0-k)/2)  *ln(det(T0*Sig_hatT))  - ln(det(XXT'*XXT));
   lndatadens = pYt - pYt0;

   RMSE = sqrt(diag( (YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CC_hat)'*(YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CC_hat)/(nobs-T0) ));

goto eofcomp;

/* Method 2: compute posterior probability treating the first T0 observations
** fixed. These are enough observations to make the posterior of C,\Sigma proper
** so that p(Y|Yk) is indeed a proper probability distribution which is computed
** recursively using a prediction error decomposition
*/
M2:

   /* We start at period k+2 to predict observation k+3
   */
   t = T0;
   lndatadens = 0;

   do until t == nobs;

      /* Compute the OLS estimate
      */
      XXt     = XX[1:t,.];
      YYt     = YY[1:t,.];
      CC_hat  = inv(XXt'*XXt)*XXt'*YYt;
      C_hat   = vec(CC_hat);
      SIG_hat = (YYt-XXt*CC_hat)'*(YYt-XXt*CC_hat)/t;

      /* evaluate the predictive distribution at obs t+1
      */
      a  = 1 + XX[t+1,.]*inv(XXt'*XXt)*XX[t+1,.]';
      nu = t-k-1;
      V  = a*t/nu*SIG_hat;
      lnkap= (nu/2)*ln(nu) + ln( gamma((nu+2)/2) ) - ln( pi*gamma(nu/2) );
      lnpdf= lnkap - 0.5*ln(det(V)) -((nu+2)/2)*ln(nu + (YY[t+1,.] - XX[t+1,.]*CC_hat)*inv(V)*(YY[t+1,.] - XX[t+1,.]*CC_hat)' );
      
      lndatadens = lndatadens + lnpdf;

      locate 1,1;
      "Step" t;
      lndatadens;
      lnkap;
      /*
      enter = keyw;
      */
      t = t+1;
      
   endo;

goto eofcomp;

/* Method 3: compute posterior probability via importance sampling.
** Generate draws from the posterior parameter distribution and compute
** a geometric mean of the inverse likelihood function.
*/
M3:


/* Compute OLS estimates
*/
CC_hat  = inv(XX'*XX)*XX'*YY;
C_hat   = vec(CC_hat);
SIG_hat = (YY-XX*CC_hat)'*(YY-XX*CC_hat)/nobs;

/* Simulation loop
*/

lndat = zeros(ndraw,1);

i = 1;
do until i > ndraw;
   
      /* draw from the marginal posterior of SIG
      */
      H = nobs*Sig_hat;
      Z = chol(inv(H))'*rndn(2,nobs-k);
      Sig_sim = inv(Z*Z');

      /* draw from the conditional posterior of C
      */
      VC     = Sig_sim.*.inv(XX'*XX);
      C_sim  = C_hat + chol(VC)'*rndn(2*k,1);
      CCt_sim= reshape(C_sim,2,k);
      
      /* Evaluate likelihood function
      */
      lndat[i] = -((nobs-T0))*ln(2*pi) - ((nobs-T0)/2)*ln(det(Sig_sim)) 
                 - 0.5*sumc(diag(  inv(Sig_sim)*( (YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CCt_sim')'*(YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CCt_sim'))));
      lndatadens = ln(1/( sumc(1/ exp(lndat[1:i]-20) )/i))+20;

      /* Screen output
      */
      locate 1,1;
      "Simulation step" i;
      "Likelihood     " lndat[i];
      "Score" lndatadens;
    i = i+1;
endo;

lhmax =  -((nobs-T0))*ln(2*pi) - ((nobs-T0)/2)*ln(det(Sig_hat)) 
         - 0.5*sumc(diag(  inv(Sig_hat)*( (YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CC_hat)'*(YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CC_hat))));

goto eofcomp;

/* Method 4: Laplace Approximation
*/
M4:

/* Compute OLS estimates
*/
CC_hat  = inv(XX'*XX)*XX'*YY;
C_hat   = vec(CC_hat);
SIG_hat = (YY-XX*CC_hat)'*(YY-XX*CC_hat)/nobs;
SIG_mod = nobs/(nobs+2+1)*SIG_hat;


/* Define Duplicator matrix
*/
D2 = ( 1 ~ 0 ~ 0 )|
     ( 0 ~ 1 ~ 0 )|
     ( 0 ~ 1 ~ 0 )| 
     ( 0 ~ 0 ~ 1 );

D2p = inv(D2'*D2)*D2';


/* Compute prior parameters from sample 1...T0
*/
CCprior  = inv(XX[1:T0,.]'*XX[1:T0,.])*XX[1:T0,.]'*YY[1:T0,.];
HHprior = (YY[1:T0,.]-XX[1:T0,.]*CCprior)'*(YY[1:T0,.]-XX[1:T0,.]*CCprior);
nup = T0-k;

/* Evaluate the prior at the posterior mode, plug-in CC_hat and SIG_mod
*/

lnprior = -k*ln(2*pi)-0.5*ln(det(Sig_mod.*.inv(XX[1:T0,.]'*XX[1:T0,.]))) 
          -0.5*sumc(diag( inv(Sig_mod)*(CC_hat - CCprior)'*(XX[1:T0,.]'*XX[1:T0,.])*(CC_hat-CCprior) ))
          -( nup*ln(2) + 0.5*ln(pi) + ln(gamma(nup/2)) + ln(gamma( (nup-1)/2 )) )
          +(nup/2)*ln(det(HHprior)) - ((nup+3)/2)*ln(det(Sig_mod)) - 0.5*sumc(diag( inv(Sig_mod)*HHprior ));

/* Evaluate the likelihood at the posterior mode
*/
lhmode =  -((nobs-T0))*ln(2*pi) - ((nobs-T0)/2)*ln(det(Sig_mod)) 
          - 0.5*sumc(diag(  inv(Sig_mod)*( (YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CC_hat)'*(YY[T0+1:nobs,.]-XX[T0+1:nobs,.]*CC_hat))));

/* Calculate a penalty term
*/
Sig22= 2*D2p*(Sig_mod.*.Sig_mod)*D2p'/(nobs+1+2);

CCpenalty  = 0.5*2*k*ln(2*pi) + 0.5*ln(det(Sig_mod.*.inv(XX'*XX)));
SIGpenalty = 0.5*3*ln(2*pi)   + 0.5*ln(det(Sig22));

lndatadens = lnprior+lhmode+CCpenalty+SIGpenalty;

goto eofcomp;

/* Method 5: compute posterior probability treating the first T0 observations
** fixed. These are enough observations to make the posterior of C,\Sigma proper
** so that p(Y|Yk) is indeed a proper probability distribution which is computed
** recursively using a prediction error decomposition
*/
M5:

   /* We start at period T0 to predict observation T0+1
   */
   t = T0;
   lndatadens = 0;
   lnprobex = 0;

   do until t == nobs;

      /* Compute the OLS estimate
      */
      XXt     = XX[1:t,.];
      YYt     = YY[1:t,.];
      CC_hat  = inv(XXt'*XXt)*XXt'*YYt;
      C_hat   = vec(CC_hat);
      SIG_hat = (YYt-XXt*CC_hat)'*(YYt-XXt*CC_hat)/t;

      /* evaluate the exact predictive distribution at obs t+1
      */
      a  = 1 + XX[t+1,.]*inv(XXt'*XXt)*XX[t+1,.]';
      nu = t-k-1;
      V  = a*t/nu*SIG_hat;
      lnkap= (nu/2)*ln(nu) + ln( gamma((nu+2)/2) ) - ln( pi*gamma(nu/2) );
      lnpdfex= lnkap - 0.5*ln(det(V)) -((nu+2)/2)*ln(nu + (YY[t+1,.] - XX[t+1,.]*CC_hat)*inv(V)*(YY[t+1,.] - XX[t+1,.]*CC_hat)' );

      /* evaluate the predictive distribution at obs t+1 by MC Integration
      */
      lnpdfsim = zeros(ndraw,1);
      i = 1;
      do until i > ndraw;
   
        /* draw from the marginal posterior of SIG
        */
        H = t*Sig_hat;
        Z = chol(inv(H))'*rndn(2,t-k);
        Sig_sim = inv(Z*Z');

        /* draw from the conditional posterior of C
        */
        VC     = Sig_sim.*.inv(XXt'*XXt);
        C_sim  = C_hat + chol(VC)'*rndn(2*k,1);
        CCt_sim= reshape(C_sim,2,k);
      
        /* Evaluate predictive density
        */
        lnpdfsim[i] = -ln(2*pi) - 0.5*ln(det(Sig_sim)) 
                      - 0.5*(YY[t+1,.] - XX[t+1,.]*CCt_sim')*inv(Sig_sim)*(YY[t+1,.] - XX[t+1,.]*CCt_sim')';
        lnpdf = ln(meanc( exp(lnpdfsim[1:i]) ));

        /* Screen output
        */
        locate 10,1;
        "Simulation step" i;
        "Log Likelihood     " lnpdfsim[i];
        "Likelihood         " exp(lnpdfsim[i]);
        "Predictive Density, MC   " lnpdf;
        "Predictive Density, Exact" lnpdfex;

        i = i+1;
      endo;
      
      lndatadens = lndatadens + lnpdf;  
      lnprobex   = lnprobex + lnpdfex;

      locate 1,1;
      "Step" t;
      "Data Density, Exact"  lnprobex;
      "Data Density, MC   "  lndatadens;
     
      enter = keyw;
      
      t = t+1;
      
   endo;


goto eofcomp;


/*************************************************************
**  Loop Control and output
*/
eofcomp:

locate 20,1;
"Loop iteration     " loopct;
"Marginal Data Dens " lndatadens;
margdens[loopct] = lndatadens;

loopct = loopct + 1;
endo;

cls;
"Summary";
"=======";
" ";
"Start T0";
T0grid';
" ";
"End Tend";
Tendgrid';
" ";
"Marginal Data density";
margdens;
