/* filename:    ciapred.g
** description: The program converts the DSGE parameters
**              into impulse responses, etc
** created:     21/10/99
*/

new;
closeall;
library cialib, pgraph, user;
cls;


/* Open files with Parameter Draws
*/
npara = 10;
lpath = "c:\\projects\\active\\dsgesel\\results\\analys1\\";
lmodel = "m2";
lprior = "1";
mhrun  = "1";
mspec = 2;
lpath = lpath $+ "mhrun" $+ mhrun $+ "\\";
opara = lpath $+ lmodel  $+ lprior $+ "par" $+ mhrun;
open fhpara = ^opara for read;

/* Define some Constants
*/
nsim   = 1000;          /* block size */
ncov   = 100;
ncovout= 4;           /* number of lags for the autocovariance function */
nirf   = 40;          /* number of impulse responses */
nvard  = 200;         /* number of MA lags for variance decomposition */
nload  = 1;
nomega = 5;           /* Number of grid points for spectrum */
omegagr= seqa(pi/nomega, pi/nomega, nomega-1);


/* Initialize Output files
*/
oir11 = lpath $+ lmodel $+ lprior $+ "r11" $+ mhrun;
oir12 = lpath $+ lmodel $+ lprior $+ "r12" $+ mhrun;
oir21 = lpath $+ lmodel $+ lprior $+ "r21" $+ mhrun;
oir22 = lpath $+ lmodel $+ lprior $+ "r22" $+ mhrun;
ovard = lpath $+ lmodel $+ lprior $+ "vd"  $+ mhrun;

ospc1 = lpath $+ lmodel $+ lprior $+ "sp1" $+ mhrun;
ospc2 = lpath $+ lmodel $+ lprior $+ "sp2" $+ mhrun;
ocohe = lpath $+ lmodel $+ lprior $+ "coh" $+ mhrun;
ophas = lpath $+ lmodel $+ lprior $+ "pha" $+ mhrun;
ogamm = lpath $+ lmodel $+ lprior $+ "gam" $+ mhrun;
ocorr = lpath $+ lmodel $+ lprior $+ "cor" $+ mhrun;

create fhir11 = ^oir11 with IR11, nirf, 8;
create fhir12 = ^oir12 with IR12, nirf, 8;
create fhir21 = ^oir21 with IR21, nirf, 8;
create fhir22 = ^oir22 with IR22, nirf, 8;
create fhvard = ^ovard with VARD, 2,    8;

create fhspc1 = ^ospc1 with SPC1, nomega-1, 8;
create fhspc2 = ^ospc2 with SPC2, nomega-1, 8;
create fhcohe = ^ocohe with COHE, nomega-1, 8;
create fhphas = ^ophas with PHAS, nomega-1, 8;

create fhgamm = ^ogamm with GAMM, 4*(ncovout+1), 8;
create fhcorr = ^ocorr with CORR, 4*(ncovout+1), 8; 
 
/* Initialization
*/ 

eofloop = 0;
loopct  = 0;

do until eofloop; 

   /* Read blocks of size nsim
   */
   parasim = readr( fhpara, nsim );

   nread = rows(parasim);
      
   Asim11 = zeros(nread,nirf);
   Asim12 = zeros(nread,nirf);
   Asim21 = zeros(nread,nirf);
   Asim22 = zeros(nread,nirf);
   vardsim= zeros(nread,2);

   GAMMsim   = zeros(nread,4*(ncovout+1));
   CORRsim   = zeros(nread,4*(ncovout+1));

   Spec1sim = zeros(nread,nomega-1);
   Spec2sim = zeros(nread,nomega-1);
   Cohsim = zeros(nread,nomega-1); 
   Phassim = zeros(nread,nomega-1);
    

   j = 1;
   do until j > rows(parasim);

      /* For each element of the block, compute the desired statistic
      */

     {TTT, RRR, valid} = ciasolv( parasim[j,.]', mspec);
     sigeps = parasim[j,npara-2]^2;
     sigeta = parasim[j,npara-1]^2;
     nstate = rows(TTT);

     /* create system matrices for state space model
     */

     /* composition of state vector
     ** [dm(t-1), dp(t-1), dy(t-1), dm(t), dp(t), dk(t+1), dh(t), dy(t) ,... ]
     */

     ZZ = zeros(2,nstate);
     ZZ[1,8] = 1; ZZ[1,3] = -1;
     ZZ[2,5] = 1; ZZ[2,2] = -1; ZZ[2,1] = 1;

     HH = (( sigeps ~ -sigeps )|( -sigeps ~ sigeps));
     QQ = ( sigeps ~ 0        )|( 0      ~ sigeta); 
     VV = ( sigeps ~ -sigeps )|( 0       ~ 0      );


     /* compute impulse responses for output and inflation
     */

     IRFeps = zeros(2,nvard);
     IRFeta = zeros(2,nvard); 
  
     /* Define initial state vector vector 
     */
     alpeps = zeros(nstate,1);
     alpeta = zeros(nstate,1);

     /* Total Factor Productivity Shocks
     */
     u_eps   = (1|-1)*sqrt(sigeps) ~ zeros(2,nvard-1);
     v_eps   = (1|0 )*sqrt(sigeps) ~ zeros(2,nvard-1);
     u_eta   = zeros(2,nvard);
     v_eta   = (0|1 )*sqrt(sigeta) ~ zeros(2,nvard-1);

     /* Initialize Loop variable t 
     */
     t=1;

     DO UNTIL t > nvard;

      /* update the vector of state variables alpha
      */
      alpeps = TTT*alpeps + RRR*v_eps[.,t];
      irfeps[.,t] = ZZ*alpeps + u_eps[.,t];

      alpeta = TTT*alpeta + RRR*v_eta[.,t];
      irfeta[.,t] = ZZ*alpeta + u_eta[.,t];

      t=t+1;

     ENDO;
     
     /* Variance Decomposition for output growth and inflation
     */
     vardsim[j,1] = irfeps[1,.]*irfeps[1,.]'/( irfeps[1,.]*irfeps[1,.]' + irfeta[1,.]*irfeta[1,.]');
     vardsim[j,2] = irfeps[2,.]*irfeps[2,.]'/( irfeps[2,.]*irfeps[2,.]' + irfeta[2,.]*irfeta[2,.]');

     /*
     locate 5,1;
     vardsim[j,.];
     */

     /* Convert the growth rate IRF for output into level irf
     */
     t=2;
     do until t > nirf;
       irfeps[1,t] = irfeps[1,t-1]+irfeps[1,t];
       irfeta[1,t] = irfeta[1,t-1]+irfeta[1,t];
       t = t+1;
     endo;

     Asim11[j,.] = irfeps[1,1:nirf];
     Asim12[j,.] = irfeta[1,1:nirf];
     Asim21[j,.] = irfeps[2,1:nirf];
     Asim22[j,.] = irfeta[2,1:nirf];

     /* Compute autocovariances and spectrum
     */

     /* Initialize Autocovariances
     */
     GAMM0 = zeros(4,ncovout+1);
     CORR0 = zeros(4,ncovout+1);

     GA0  = inv(eye(nstate^2) - (TTT.*.TTT))*vec(RRR*QQ*RRR');
     GA0  = reshape(GA0,nstate,nstate);

     Gl   = ZZ*GA0*ZZ' + ZZ*RRR*VV + (ZZ*RRR*VV)' + HH;
     Gstar= ( Gl[1,1] ~ sqrt(Gl[1,1]*Gl[2,2]) )|
          ( sqrt(Gl[1,1]*Gl[2,2]) ~ Gl[2,2] );

     GAMM0[.,1] = vecr(Gl);
     CORR0[.,1] = vecr(Gl./Gstar);
 
     /* Initialize Spectrum
     */
     SPEC1   = ones(1,nomega-1)/(2*pi)*GA0[1,1];   
     SPEC2   = ones(1,nomega-1)/(2*pi)*GA0[2,2];
     SPEC12  = ones(1,nomega-1)/(2*pi)*GA0[1,2];

     TTl = TTT;
     l=1;
     do until l > ncov;
      Gl = ZZ*(TTl*GA0)*ZZ' + ZZ*(TTl*RRR*VV);

      SPEC1    = SPEC1 + (1/pi)*Gl[1,1]*cos(l*omegagr');
      SPEC2    = SPEC2 + (1/pi)*Gl[2,2]*cos(l*omegagr');
      SPEC12   = SPEC12 + (0.5/pi)*Gl[1,2]*exp(-complex(0,1)*l*omegagr')
                        + (0.5/pi)*Gl[2,1]*exp( complex(0,1)*l*omegagr');
      if l LE ncovout;
         GAMM0[.,l+1] = vecr(Gl);           
         CORR0[.,l+1] = vecr(Gl./Gstar);
      endif;
      TTl = TTl*TTT;
      l = l+1; 
     endo;

     GAMMsim[j,.] = vec(GAMM0)';
     CORRsim[j,.] = vec(CORR0)';

     Spec1sim[j,.] = SPEC1;
     Spec2sim[j,.] = SPEC2;
     Cohsim[j,.]   = ( (real(SPEC12))^2 + (imag(SPEC12))^2)./ ( SPEC1 .* SPEC2 );
     Phassim[j,.]  = atan( real(SPEC12)./imag(SPEC12) );

     locate 1,1;
     "Iteration" j "of" nsim;
     j = j+1;
  endo;

  /* Write Output
  */
  wr = writer(fhir11,Asim11);
  wr = writer(fhir12,Asim12);
  wr = writer(fhir21,Asim21);
  wr = writer(fhir22,Asim22);
  wr = writer(fhvard,vardsim);
  /*
  wr = writer(fhspc1,spec1sim);
  wr = writer(fhspc2,spec2sim);
  wr = writer(fhcohe,cohsim);
  wr = writer(fhphas,phassim);
  */
  wr = writer(fhgamm,gammsim);
  wr = writer(fhcorr,corrsim);

  loopct = loopct + 1;
  locate 2,1;
  "Loop     " loopct;
  eofloop = eof(fhpara);
endo;


closeall fhir11, fhir12, fhir21, fhir22,
         fhspc1, fhspc2, fhcohe, fhphas,
         fhgamm, fhcorr, fhpara;


outfile = lpath $+ lmodel $+ lprior $+ "pre.log";
output file = ^outfile on;

"filename           contents";
"--------------------------------------------------";
"varirf11.out       each row: IRF dY/eps";
"varirf12.out       each row: IRF dY/eta";
"varirf21.out       each row: IRF dP/eps";
"varirf22.out       each row: IRF dP/eta";
" ";
"vargamm.out        each row contains autocovariance fcn";
"varcorr.out        The matrices                        ";
"                   g0(11) g0(12)           g1(11) g1(12)";                    
"                   g0(21) g0(22)           g1(21) g1(22)";
"                   are stored as                        ";
"                   g0(11) g0(12) g0(21) g0(22)  g1(11) g1(12) g1(21) g1(22)";
" ";
"spec1.out spec2.out cohere.out phase.out";
"Each file contains spectral function of omega, evaluated on grid" nomega;

output off;


