/* filename:    ciamh.g
** description: The program generates draws from the posterior
**              distribution via the random walk Metropolis Hastings Algorithm
**              
** created:     02/27/00
** modified:   
*/

library pgraph, user, cialib;
__output = 0;

cls;
outwidth 128;

/* Import data on output growth and inflation: series (nobs,2)
** observations from 1950:I to 1997:IV
*/

nobs  = rows(series);       /* number of observations          */
p     = 4;                  /* number of lags in VAR(p) model  */
YY    = series[1:nobs,.];   /* Note that YY is defined defined over the entire series */
k     = 1 + 2*p;
T0    = p+k+2;
T0    = 4 + 16;

lpath = "c:\\projects\\active\\dsgesel\\gauss\\para";
lmodel = "m2";
lprior = "1";
mhrun  = "2";
npara  = 10;
mspec  = 2;

cc0     = 10;
cc      = 0.2;
nblocks = 9;
nsim    = 10000;


parafile  = lmodel $+ lprior $+ "mod.out";
priorfile = lmodel $+ lprior $+ "pri.out";

load path=^lpath para[npara,1]      = ^parafile;
load path=^lpath prior[npara,3]     = ^priorfile;

pmean  = prior[.,1];
pstdd  = prior[.,2];
pshape = prior[.,3];

/* Load multiplication matrix for parameters
*/
opath = "c:\\projects\\active\\dsgesel\\results\\analys1\\";
omult = opath $+ lmodel $+ lprior $+ "mul";
open fhmult = ^omult for read;

SIGSCALE   = readr( fhmult,npara );
SIGPROP    = SIGSCALE*SIGSCALE';
if mspec == 1;
   SIGPROP = SIGPROP[1:9,1:9];
   SIGPROPDIM = 9;
   PARASEL = eye(npara);
   PARASEL = PARASEL[1:9,.];
else;
   SIGPROPDIM = 10;
   PARASEL = eye(npara);
endif;
SIGPROPINV = inv(SIGPROP);
SIGPROPlndet = ln(det(SIGPROP));

closeall fhmult;

_bounds =
    (0.1  ~ 0.99999 )|
    (0.5  ~ 0.99999 )|
    (-0.5 ~ 0.5     )|
    (-0.5 ~ 0.5     )|
    (1E-5 ~ 0.99999 )|
    (1E-5 ~ 0.99999 )|
    (1E-5 ~ 0.99999 )|
    (1E-6 ~ 100     )|
    (1E-6 ~ 100     )|
    (1E-7 ~ 500     );

/* Compute Posterior Mode
*/
{postmax, likemax} = fcn(para);
"Posterior / Likelihood" postmax likemax;
"Press Key to continue";
kstrt = keyw;
cls;


/* Initialize Output files
*/

ostat = opath  $+ lmodel $+ lprior $+ "sta" $+ mhrun;
opara = opath  $+ lmodel $+ lprior $+ "par" $+ mhrun;
statname = { POST LIKE PROP PROPPOST REJECT };

create fhstat=^ostat with ^statname, 5, 8;
create fhpara=^opara with PARAM, npara, 8;

/******************************************************************
**              Metropolis Hastings Algorithm
*/

/* Initialize PARANEW
*/
valid = 0;
do until valid == 1;
   paranew   = para + cc0*(sigscale*rndn(npara,1));
   {postnew, likenew} = fcn(paranew);
   propdens  = -0.5*SIGPROPDIM*ln(2*pi) - 0.5*SIGPROPlndet - 0.5*SIGPROPDIM*ln(cc0^2)
               -0.5*(paranew - para)'*PARASEL'*SIGPROPINV*PARASEL*(paranew - para)/cc0^2;
   if postnew > -1000;
      valid = 1;
   endif;
endo;   


iblock = 1;
do until iblock > nblocks;

   parasim = zeros(nsim,npara);
   likesim = zeros(nsim,1);
   postsim = zeros(nsim,1);
   rej     = zeros(nsim,1);
   propsim = zeros(nsim,1);
   proppostsim = zeros(nsim,1);

   if iblock == 1;
      j = 2;
      likesim[1,1] = likenew;
      postsim[1,1] = postnew;
      propsim[1,1] = propdens;
      proppostsim[1,1] = postnew;
      parasim[1,.] = paranew';
      postold      = postnew;
      likeold      = likenew;
      paraold      = paranew;     
   else;
      j=1;
   endif;      

   do until j > nsim;

     paranew          = paraold + cc*(sigscale*rndn(npara,1));
     {postnew, likenew} = fcn(paranew);
     propdens         = -0.5*SIGPROPDIM*ln(2*pi) - 0.5*SIGPROPlndet - 0.5*SIGPROPDIM*ln(cc^2)
                        -0.5*(paranew - paraold)'*PARASEL'*SIGPROPINV*PARASEL*(paranew - paraold)/cc^2;
     propsim[j,1]     = propdens;
     proppostsim[j,1] = postnew;

     r = minc(1 | exp( postnew - postold));
     if rndu(1,1) < r;
       /* Accept proposed jump
       */
       postsim[j,1] = postnew;
       likesim[j,1] = likenew;
       parasim[j,.] = paranew';
       paraold = paranew;
       postold = postnew;
       likeold = likenew;
     else;
       /* Reject proposed jump
       */
       likesim[j,1] = likeold;
       postsim[j,1] = postold;
       parasim[j,.] = paraold';
       rej[j] = 1;
     endif;

     locate 1,1;
     "Block" iblock "of" nblocks;
     "Simulation step" j "of" nsim;
     "Rejection: r / rej / perct " r rej[j] sumc(rej)/j;
     "Likelihood     " likesim[j,1];
     "Posterior      " postsim[j,1];
     "Proposal Dens  " propsim[j,1];
     "Estimated Score" 1250+ln(1/(sumc(1/exp(likesim[1:j,1]-1250))/j));
     "Parameters     "; parasim[j,.]; 

     j = j+1;
    endo;

    wr = writer(fhstat,postsim~likesim~propsim~proppostsim~rej);
    wr = writer(fhpara,parasim);

    iblock = iblock+1;
endo;  

closeall fhstat, fhpara;
end;

/****************************************************/
/*                 PROCEDURES                       */
/****************************************************/

proc (2) = fcn(para);
local lnpY, lnprio, obsmean, obsvar, lnpost;

    /* Fix parameter values here
    */
 
    if (para < _bounds[.,2]) and (para > _bounds[.,1]);

       {lnpy, obsmean, obsvar } = evalcia( para, mspec, T0 ,YY);
       lnpy   = real(lnpy);
       lnprio = priodens( para, pmean, pstdd, pshape);
       lnpost = real(lnpy + lnprio);

    else;
      
       lnpost = -1E6;
       lnpy   = -1E6;

    endif;
    
retp(lnpost,lnpy);  
endp;

