% SMM
% This program constructs the final set used in the estimation of the model
% in the paper "Skewness Risk and Bond Prices" by Francisco Ruge-Murcia,
% which is forthcoming in the JOURNAL OF APPLIED ECONOMETRICS

clear all


%--------------------------------------------------------------------------%
% READ AND DETREND DATA 
disp('Read and detrend data')
% Construct date %
	quarter = (kron(ones(1,68),(1:4)))';
	year = (kron((1947:2014),ones(1,4)))';
	obs = year+quarter/10;

% Consumer price index (monthly from 1947M1 to 2015M2)
	load CPIAUCSL.txt -ascii
	qcpi = CPIAUCSL(1:end-2,:);  % drop the two observations from 2015
    qcpi = reshape(qcpi(:,2),3,size(qcpi,1)/3);
    qcpi = transpose(mean(qcpi));
    qcpi = [obs qcpi];  % quarterly from 1947Q1 to 2014Q4

% Compute rate of inflation
	pcpi = [qcpi(2:size(qcpi,1),1) (qcpi(2:size(qcpi,1),2)./qcpi(1:size(qcpi,1)-1,2))]; % quarterly rate
	cpi = qcpi;
	qcpi = qcpi(2:size(qcpi,1),:);
    
% Consumption (quarterly from 1947Q1 to 2014Q4)
	load PCND.txt -ascii
	load PCESV.txt -ascii
	load PCDG.txt -ascii
	load PCEC.txt -ascii
    PCESV(:,2) = PCESV(:,2)/4;  % since data is annualized divide by 4 to convert into quarterly
    PCND(:,2) = PCND(:,2)/4;
    PCDG(:,2) = PCDG(:,2)/4;
    PCEC(:,2) = PCEC(:,2)/4;
	pce = [PCND(:,1) PCND(:,2)+PCESV(:,2)];
    
% Investment (quarterly from 1947Q1 to 2014Q4)
% Investment is mesured by private nonresidential fixed investment    
	load PNFI.txt -ascii
    PNFI(:,2) = PNFI(:,2)/4;  % since data is annualized divide by 4 to convert into quarterly
 
% Population (monthly from 1948M1 to 2015M3)
	load CNP16OV.txt -ascii
	pop = CNP16OV;
    pop = pop/1000;  % in millions
	pop = pop(1:end-3,:);  % drop the three observations from 2015
    pop = reshape(pop(:,2),3,size(pop,1)/3);
    pop = transpose(mean(pop));
    pop = [obs(5:end) pop];  % quarterly from 1948Q1 to 2014Q4
    
% Construct real, per-capita variables and rates of growth using CPI
    realc = [pce(5:end,1) pce(5:end,2)./cpi(5:end,2)./pop(:,2)];   % quarterly from 1948Q1 to 2014Q4
    reali = [PNFI(5:end,1) (PNFI(5:end,2)+PCDG(5:end,2))./cpi(5:end,2)./pop(:,2)];   % quarterly from 1948Q1 to 2014Q4
    growc = [pce(6:end,1) realc(2:end,2)./realc(1:end-1,2)];  % quarterly from 1948Q2 to 2014Q4  
    growi = [PNFI(6:end,1) reali(2:end,2)./reali(1:end-1,2)];  % quarterly from 1948Q2 to 2014Q4         
    
% 3-Month T-Bill rate (monthly from 1934M1 to 2015M3)
% The raw data is at the annual rate
	load TB3MS.txt -ascii
	tb3m = TB3MS(301:end,:);
    tb3m = tb3m(1:end-3,:);  % drop the three observations from 2015
    tb3m = reshape(tb3m(:,2),3,size(tb3m,1)/3);
    tb3m = transpose(mean(tb3m));
	tb3m = (1+tb3m/100).^(1/4); % transform raw data to a quarterly rate    
    
% 6-Month T-Bill rate (monthly from 1958M12 to 2015M3)
% The raw data is at the annual rate
	load TB6MS.txt -ascii
	tb6m = TB6MS(2:end,:); % drop the observation from 1958
    tb6m = tb6m(1:end-3,:);  % drop the three observations from 2015
    tb6m = reshape(tb6m(:,2),3,size(tb6m,1)/3);
    tb6m = transpose(mean(tb6m));
	tb6m = (1+tb6m/100).^(1/4); % transform raw data to a quarterly rate    
    
% Fama-Bliss discount bond yields (monthly from 1952M6 to 2014M12)
% The raw data is at the annual rate
	load FAMA.txt -ascii
	fama = FAMA(2:end,2:end);% drop the observation from June 1952 and the column with the date
    fb1y = reshape(fama(:,1),3,size(fama,1)/3);
    fb1y = transpose(mean(fb1y));
	fb1y = (1+fb1y/100).^(1/4); % transform raw data to a quarterly rate   
    fb2y = reshape(fama(:,2),3,size(fama,1)/3);
    fb2y = transpose(mean(fb2y));
	fb2y = (1+fb2y/100).^(1/4); % transform raw data to a quarterly rate  
    fb3y = reshape(fama(:,3),3,size(fama,1)/3);
    fb3y = transpose(mean(fb3y));
	fb3y = (1+fb3y/100).^(1/4); % transform raw data to a quarterly rate  
    fb4y = reshape(fama(:,4),3,size(fama,1)/3);
    fb4y = transpose(mean(fb4y));
	fb4y = (1+fb4y/100).^(1/4); % transform raw data to a quarterly rate     
    fb5y = reshape(fama(:,5),3,size(fama,1)/3);
    fb5y = transpose(mean(fb5y));
	fb5y = (1+fb5y/100).^(1/4); % transform raw data to a quarterly rate         
    
% Put data set together
% Quarterly data from 1959Q1 to 2014Q4
% The columns of data are: 
% Col 1  = date
% Col 2  = real per-capita consumption (or its growth rate)
% Col 3  = real per-capita investment (or its growth rate)
% Col 4  = rate of inflation
% Col 5  = 3-month T-Bill rate
% Col 6  = 6-month T-Bill rate
% Col 7  = 1-year Fama-Bliss discount-bond rate
% Col 8  = 2-year Fama-Bliss discount-bond rate
% Col 9  = 3-year Fama-Bliss discount-bond rate
% Col 10 = 4-year Fama-Bliss discount-bond rate
% Col 11 = 5-year Fama-Bliss discount-bond rate

data = obs(49:end,:);
data = [data growc(44:end,2)];  
data = [data growi(44:end,2)];  
data = [data pcpi(48:end,2)];  
data = [data tb3m];
data = [data tb6m];
data = [data fb1y(27:end)];
data = [data fb2y(27:end)];
data = [data fb3y(27:end)];
data = [data fb4y(27:end)];
data = [data fb5y(27:end)];

% Filter data using deterministic detrending
% Whether logs are taken or not must be consistent with linearization below
qtdata = [data(:,1) 100*log(data(:,2:end))];

disp('end')
return