/*

If you have questions, contact:

David E. Rapach
Department of Economics
Saint Louis University
3674 Lindell Boulevard
Saint Louis, MO 63108-3397
rapachde@slu.edu
http://pages.slu.edu/faculty/rapachde

*/

new;
output file=c:\research\garchbreak\Oos_us_other_n01.out reset;
load data[6453,1]=c:\research\garchbreak\Data_us_d.txt;
rex=ln(data[2:6453]./data[1:6452])*100; @ US continuous returns @
p=500; @ number of out-of-sample observations @
r=rows(rex)-p; @ in-sample period @
load pe_gj=c:\research\garchbreak\Data_pe_us_gj;
pars_gj=pe_gj[.,3:6];
hhat_gj=pe_gj[.,7];
load pe_ms=c:\research\garchbreak\Data_pe_us_ms;
pars_ms=pe_ms[.,3:10];
sims=2000;
alpha=0.05;
s=1; @ initial forecast horizon @
smax=120; @ maximum forecast horizon @
do until s>smax;
   fc_gj=zeros(p-(s-1),s); @ for storing GJR-GARCH forecasts @ 
   fc_ms=zeros(p-(s-1),s); @ for storing expanding MS-GARCH forecasts @
   fcv_gj=zeros(p-(s-1),1); @ for storing GJR-GARCH VaR forecasts @ 
   fcv_ms=zeros(p-(s-1),1); @ for storing expanding MS-GARCH VaR forecasts @
   iter=0;
   do until iter>p-s;
      b_garch_gj=pars_gj[iter+1,.]';
      ffc_gj=gjr_garch11_fc(b_garch_gj,rex[r+iter],hhat_gj[iter+1],s);
      fc_gj[iter+1,.]=ffc_gj';
      fcv_gj[iter+1]=gjr_garch11_var_fc(b_garch_gj,ffc_gj[1],s,alpha,sims);
      b_garch_ms=pars_ms[iter+1,.]';
      ffc_ms=msgarch11_fc(b_garch_ms,rex[1:r+iter],s);
      fc_ms[iter+1,.]=ffc_ms';
      fcv_ms[iter+1]=msgarch11_var_fc(b_garch_ms,rex[1:r+iter],s,alpha,sims);
      iter=iter+1;
   endo;
   rexoos=rex[r+1:r+p];
   rex2oos=rex[r+1:r+p]^2;
   format 8,6;
   "Number of in-sample observations  = " r;?;
   "Forecast horizon                  = " s;?;
   "Number of out-of-sample forecasts = " p-(s-1);?;

   "Average squared error loss function, mean (ratios)";?;

   {se_loss_gj,mse_gj}=loss_se_avg(fc_gj,rex2oos);
   {se_loss_ms,mse_ms}=loss_se_avg(fc_ms,rex2oos);
   if s==1;
      mse_ex=0.178606;
   elseif s==20;
      mse_ex=4.86863;
   elseif s==60;
      mse_ex=25.2864;
   elseif s==120;
      mse_ex=76.5956;
   endif;
   format 8,6;
   "Expanding GARCH     = " mse_ex;?;
   "Expanding GJR-GARCH = " mse_gj~(mse_gj/mse_ex);?;
   "Expanding MS-GARCH  = " mse_ms~(mse_ms/mse_ex);?;

   "5% VaR loss function, mean (ratios)";?;

   {var_loss_gj,mvar_gj}=loss_var(fcv_gj,rexoos,s,alpha);   
   {var_loss_ms,mvar_ms}=loss_var(fcv_ms,rexoos,s,alpha);
   if s==1;
      mvar_ex=0.0560488;
   elseif s==20;
      mvar_ex=0.223620;
   elseif s==60;
      mvar_ex=0.445925;
   elseif s==120;
      mvar_ex=0.459859;
   endif;
   "alpha               = " alpha;?;
   "Expanding GARCH     = " mvar_ex;?;
   "Expanding GJR-GARCH = " mvar_gj~(mvar_gj/mvar_ex);?;
   "Expanding MS-GARCH  = " mvar_ms~(mvar_ms/mvar_ex);?;

   if s==1;
      s=s+19;
   elseif s==20;
     s=s+40;
   elseif s==60;
      s=s+60;
   elseif s==120;
      s=s+120;
   endif;
endo;

/* Procedure definition area */

/**********************  PROC VARLAGS  *****************************
**   last update: 5 Dec 95      previous: 15 June 94
**   AUTHOR		 
**        Alan G. Isaac
**   FORMAT		 
**        {x,xlags} = varlags(var,lags)
**   INPUT		 
**        var  - T x K matrix
**        lags - scalar, number of lags of var (a positive integer)
**   OUTPUT		 
**        x -     (T - lags) x K matrix, the last T-lags rows of var
**        xlags - (T - lags) x lags*cols(var) matrix,
**                being the 1st through lags-th
**                values of var corresponding to the values in x
**                i.e, the appropriate rows of x(-1)~x(-2)~etc.
**   GLOBAL VARIABLES: none
**********************************************************************/
proc(2)=varlags(var,lags);
    local xlags;
    xlags = shiftr((ones(1,lags) .*. var)',seqa(1-lags,1,lags)
                                            .*. ones(cols(var),1),miss(0,0))';
    retp(trimr(var,lags,0),trimr(xlags,0,lags));
endp;

/****************************************************************
PROC: GJR_GARCH11_FC

The procedure generates a vector of volatility forecasts from
1 to s steps ahead for a stationary GJR-GARCH(1,1) model using eq.
(4.124) in Franses and van Dijk (2000).

Format: h_ts=gjr_garch11_fc(theta,e,hhat,s)

Input:

theta = vector of estimated GJR-GARCH parameters (omega|alpha|gamma|beta)
e     = scalar, obs of e_t
hhat  = scalar, estimate of h_t
s     = longest forecast horizon

Output:

h_ts = s-vector forecasts of h_t+1|t,...,h_t+s|t

Reference

P.H. Franses and D. van Dijk (2000), Non-Linear Time Series
Models in Empirical Finance (Cambridge, UK: Cambridge University
Press)
****************************************************************/
proc(1)=gjr_garch11_fc(theta,e,hhat,s);
   local omega,alpha,gama,beta,h_ts,iter;
   omega=theta[1];
   alpha=theta[2];
   gama=theta[3];
   beta=theta[3];
   h_ts=zeros(s,1);
   if e>0;
      h_ts[1]=omega+gama*e^2+beta*hhat;
   else;
      h_ts[1]=omega+alpha*e^2+beta*hhat;
   endif;
   if s>1;
      iter=2;
      do until iter>s;
         h_ts[iter]=omega+(((alpha+gama)/2)+beta)*h_ts[iter-1];
         iter=iter+1;
      endo;
   endif;
   retp(h_ts);
endp;

/*****************************************************************
PROC: GJR_GARCH11_VAR_FC

This procedure calculates a VaR quantile forecast for aggregate
returns for a stationary GJR-GARCH(1,1) model at a horizon of s
using a standard normal distribution for the standardized residuals.
The process is assumed to have zero conditional and unconditional
means.

Format: varq=gjr_garch11_var_fc(b_garch,fc,s,alpha,sims);

Intput:

b_garch = 4-vector of GJR-GARCH(1,1) estimates (omega|alpha|gamma|beta)
fc      = one-step-ahead conditional variance forecast
s       = forecast horizon
alpha   = probability of loss
sims    = number of simulations

Output:

varq = VaR quantile forecast
*****************************************************************/
proc(1)=gjr_garch11_var_fc(b_garch,fc,s,alpha,sims);
   local tretstar,itersims,zstar,hstar,estar,iters,alphai,varq;
   tretstar=ones(sims,1);
   itersims=1;
   do until itersims>sims;
      zstar=rndn(s,1);
      if s>2;
         hstar=fc|zeros(s-1,1);
      else;
         hstar=fc;
      endif;
      estar=zeros(s,1);
      estar[1]=sqrt(hstar[1])*zstar[1];
      if s>2;
         iters=2;
         do until iters>s;
            if estar[iters-1]>0;
               hstar[iters]=b_garch[1]+b_garch[3]*estar[iters-1]^2
                  +b_garch[4]*hstar[iters-1];
            else;
               hstar[iters]=b_garch[1]+b_garch[2]*estar[iters-1]^2
                  +b_garch[4]*hstar[iters-1];
            endif;
            estar[iters]=sqrt(hstar[iters])*zstar[iters];
            iters=iters+1;
         endo;
      endif;
      tretstar[itersims]=sumc(estar);
      itersims=itersims+1;
   endo;
   alphai=round(alpha*sims);
   tretstar=sortc(tretstar,1);
   varq=tretstar[alphai];
   retp(varq);
endp;

/****************************************************************
PROC: MSGARCH11_FC

The procedure generates volatility forecasts for from 1 to s steps
ahead for a two-state Markov-switching GARCH(1,1) model using the
specification in Hass et al. (2004) assuming a conditional mean of
zero.

Format: fc=msgarch11_fc(theta,e,s)

Input:

theta = parameter values
e     = data vector
s     = forecast horizon

Output:

fc = s-vector of forecasts of h_t+1|t,...,h_t+s|t

References

M. Hass et al. (2004), "A New Approach to Markov-Switching
GARCH Models," Journal of Financial Econometrics, 4, 493-530
****************************************************************/
proc(1)=msgarch11_fc(theta,e,s);
   local bigt,e2,omega1,alpha1,beta1,omega2,alpha2,beta2,p11,p22,
   bigp,mean_e2,alpha1part,h1,alpha2part,h2,h,eta1,eta2,eta,xit1t,
   xitt,iter,fc,xitst,ffc,iters;
   bigt=rows(e);
   e2=e^2;
   omega1=theta[1];
   alpha1=theta[2];
   beta1=theta[3];
   omega2=theta[4];
   alpha2=theta[5];
   beta2=theta[6];
   p11=theta[7];
   p22=theta[8];
   bigp=(p11~(1-p22))|((1-p11)~p22);
   mean_e2=meanc(e2);
   alpha1part=omega1+alpha1*missrv(lagn(e2,1),mean_e2);
   h1=recserar(0|alpha1part,mean_e2,beta1);
   h1=h1[2:bigt+1];
   alpha2part=omega2+alpha2*missrv(lagn(e2,1),mean_e2);
   h2=recserar(0|alpha2part,mean_e2,beta2);
   h2=h2[2:bigt+1];
   h=h1~h2;
   eta1=(1/sqrt(2*pi))*(sqrt(h1)^(-1)).*exp(-e2./(2*h1));
   eta2=(1/sqrt(2*pi))*(sqrt(h2)^(-1)).*exp(-e2./(2*h2));
   eta=eta1~eta2;
   xit1t=zeros(bigt,2);
   xitt=zeros(bigt,2);
   xit1t[1,.]=0.5~0.5;
   xitt[1,.]=(xit1t[1,.].*eta[1,.])/(xit1t[1,.]*eta[1,.]');
   iter=2;
   do until iter>bigt;
      xit1t[iter,.]=xitt[iter-1,.]*bigp';
      xitt[iter,.]=(xit1t[iter,.].*eta[iter,.])/(xit1t[iter,.]*eta[iter,.]');
      iter=iter+1;
   endo;
   xitst=zeros(s,2);
   ffc=zeros(s,2);
   fc=zeros(s,1);
   xitst[1,.]=xitt[bigt,.]*bigp';
   ffc[1,.]=(omega1+alpha1*e2[bigt]+beta1*h1[bigt])~
            (omega2+alpha2*e2[bigt]+beta2*h2[bigt]);
   fc[1]=xitst[1,.]*ffc[1,.]';
   if s>1;
      iters=2;
      do until iters>s;
         xitst[iters,.]=xitst[iters-1,.]*bigp';
         ffc[iters,.]=(omega1+(alpha1+beta1)*ffc[iters-1,1])~
                      (omega2+(alpha2+beta2)*ffc[iters-1,2]);
         fc[iters]=xitst[iters,.]*ffc[iters,.]';
         iters=iters+1;
      endo;
   endif;
   retp(fc);
endp;

/*****************************************************************
PROC: MSGARCH11_VAR_FC

This procedure calculates a VaR quantile forecast for
aggregate returns at a horizon of p for a Markov-switching
GARCH(1,1) model using simulations. The process is assumed to
have zero conditional and unconditional variances and be
conditionally normally distributed in each regime.

Format: varq=msgarch11_var_fc(theta,e,s,alpha,sims);

Input:

theta = parameter values
e     = data vector
s     = forecast horizon
alpha = probability of loss
sims  = number of simulations

Output:

varq = vector of VaR quantile forecasts
*****************************************************************/
proc(1)=msgarch11_var_fc(theta,e,s,alpha,sims);
   local bigt,e2,omega1,alpha1,beta1,omega2,alpha2,beta2,p11,
   p22,bigp,mean_e2,alpha1part,h1,alpha2part,h2,h,eta1,eta2,eta,
   xit1t,xitt,iter,estar,itersims,hstar,regime,iters,tret,alphai,
   varq;
   bigt=rows(e);
   e2=e^2;
   omega1=theta[1];
   alpha1=theta[2];
   beta1=theta[3];
   omega2=theta[4];
   alpha2=theta[5];
   beta2=theta[6];
   p11=theta[7];
   p22=theta[8];
   bigp=(p11~(1-p22))|((1-p11)~p22);
   mean_e2=meanc(e2);
   alpha1part=omega1+alpha1*missrv(lagn(e2,1),mean_e2);
   h1=recserar(0|alpha1part,mean_e2,beta1);
   h1=h1[2:bigt+1];
   alpha2part=omega2+alpha2*missrv(lagn(e2,1),mean_e2);
   h2=recserar(0|alpha2part,mean_e2,beta2);
   h2=h2[2:bigt+1];
   h=h1~h2;
   eta1=(1/sqrt(2*pi))*(sqrt(h1)^(-1)).*exp(-e2./(2*h1));
   eta2=(1/sqrt(2*pi))*(sqrt(h2)^(-1)).*exp(-e2./(2*h2));
   eta=eta1~eta2;
   xit1t=zeros(bigt,2);
   xitt=zeros(bigt,2);
   xit1t[1,.]=0.5~0.5;
   xitt[1,.]=(xit1t[1,.].*eta[1,.])/(xit1t[1,.]*eta[1,.]');
   iter=2;
   do until iter>bigt;
      xit1t[iter,.]=xitt[iter-1,.]*bigp';
      xitt[iter,.]=(xit1t[iter,.].*eta[iter,.])/(xit1t[iter,.]*eta[iter,.]');
      iter=iter+1;
   endo;
   estar=zeros(s,sims);
   itersims=1;
   do until itersims>sims;
      hstar=zeros(s,2);
      regime=zeros(s,1);
      hstar[1,.]=(omega1+alpha1*e2[bigt]+beta1*h[bigt,1])~
                 (omega2+alpha2*e2[bigt]+beta2*h[bigt,2]);
      if rndu(1,1)<xitt[bigt,1];
         regime[1]=1;
         estar[1,itersims]=sqrt(hstar[1,1])*rndn(1,1);
      else;
         regime[1]=2;
         estar[1,itersims]=sqrt(hstar[1,2])*rndn(1,1);
      endif;
      iters=2;
      do until iters>s;
         hstar[iters,.]=(omega1+alpha1*estar[iters-1,itersims]^2
                        +beta1*hstar[iters-1,1])~
                        (omega2+alpha2*estar[iters-1,itersims]^2
                        +beta2*hstar[iters-1,2]);
         if regime[iters-1]==1;
            if rndu(1,1)<p11;
               regime[iters]=1;
               estar[iters,itersims]=sqrt(hstar[iters,1])*rndn(1,1);
            else;
               regime[iters]=2;
               estar[iters,itersims]=sqrt(hstar[iters,2])*rndn(1,1);
            endif;
         elseif regime[iters-1]==2;
            if rndu(1,1)<(1-p22);
               regime[iters]=1;
               estar[iters,itersims]=sqrt(hstar[iters,1])*rndn(1,1);
            else;
               regime[iters]=2;
               estar[iters,itersims]=sqrt(hstar[iters,2])*rndn(1,1);
            endif;   
         endif;
         iters=iters+1;
      endo;
      itersims=itersims+1;
   endo;
   tret=sumc(estar);
   alphai=round(alpha*sims);
   tret=sortc(tret,1);
   varq=tret[alphai];
   retp(varq);
endp;

/*****************************************************************
PROC: LOSS_SE_AVG

This procedure calculates values and the mean of the squared
error loss function based on averaging from Starica and Granger
(2005).

Format: {loss_series,loss_mean}=loss_se_avg(fc,a);

Intput:

fc = n x p matrix of conditional variance forecasts
a  = vector of actual values for squared returns

Output:

loss_series = vector of loss series
loss_mean   = mean loss

Reference

C. Starica and C. Granger (2005), "Nonstationarities in
Stock Returns," Review of Economics and Statistics, 87(3),
503-522
*****************************************************************/
proc(2)=loss_se_avg(fc,a);
   local n,p,fcavg,aavg,iter,loss_series,loss_mean;
   n=rows(fc);
   p=cols(fc);
   fcavg=sumc(fc');
   aavg=zeros(n,1);
   iter=1;
   do until iter>n;
      aavg[iter]=sumc(a[iter:iter+(p-1)]);
      iter=iter+1;
   endo;
   loss_series=(fcavg-aavg)^2;
   loss_mean=meanc(loss_series);
   retp(loss_series,loss_mean);
endp;

/*****************************************************************
PROC: LOSS_VAR

This procedure calculates values for the VaR-based loss
function described in Section 4.3 of Gonzalez-Rivera et al.
(2004) for aggregate returns. It also calculates the mean
loss.

Format: {loss_series,loss_average}=loss_var(fc,a,s,alpha);

Intput:

fc    = vector of VaR quantile forecasts
a     = vector of actual values for one-period returns
s     = forecast horizon
alpha = probability of loss

Output:

loss_series  = vector of VaR-based loss function values
loss_average = mean loss

Reference

G. Gonzalez-Rivera, T.-H. Lee, and S. Mishra (2004),
"Forecasting Volatility: A Reality Check Based on Option
Prcing, Utility Function, Value-at-Risk, and Predictive
Likelihood," International Journal of Forecasting, 20(4),
629-645
*****************************************************************/
proc(2)=loss_var(fc,a,s,alpha);
   local n,aagg,iter,d_alpha,loss_series,loss_average;
   n=rows(fc);
   aagg=zeros(n,1);
   iter=1;
   do until iter>n;
      aagg[iter]=sumc(a[iter:iter+(s-1)]);
      iter=iter+1;
   endo;
   d_alpha=aagg.<fc;
   loss_series=(alpha-d_alpha).*(aagg-fc);
   loss_average=meanc(loss_series);
   retp(loss_series,loss_average);
endp;
