/*

If you have questions, contact:

David E. Rapach
Department of Economics
Saint Louis University
3674 Lindell Boulevard
Saint Louis, MO 63108-3397
rapachde@slu.edu
http://pages.slu.edu/faculty/rapachde

*/

new;
tstart=hsec;
output file=c:\research\garchbreak\Oos_sw_s60_fhs.out reset;
load data[6618,1]=c:\research\garchbreak\Data_sw_d.txt;
rex=ln(data[2:6618]./data[1:6617])*100; @ Switzerland continuous returns @
p=500; @ number of out-of-sample observations @
r=rows(rex)-p; @ in-sample period @
load pe_ex=c:\research\garchbreak\Data_pe_sw_ex;
load pe_fi=c:\research\garchbreak\Data_pe_sw_fi;
load pe_50=c:\research\garchbreak\Data_pe_sw_50;
load pe_25=c:\research\garchbreak\Data_pe_sw_25;
load pe_br=c:\research\garchbreak\Data_pe_sw_br;
first_ex=pe_ex[.,2];
first_fi=pe_fi[.,2];
first_50=pe_50[.,2];
first_25=pe_25[.,2];
first_br=pe_br[.,2];
pars_ex=pe_ex[.,3:5];
pars_fi=pe_fi[.,3:6];
pars_50=pe_50[.,3:5];
pars_25=pe_25[.,3:5];
pars_br=pe_br[.,3:5];
hhat_ex=pe_ex[.,6];
hhat_fi=pe_fi[.,7];
hhat_50=pe_50[.,6];
hhat_25=pe_25[.,6];
hhat_br=pe_br[.,6];
s=60; @ forecast horizon @
sims=2000;
alpha=0.05;
bl=500;
fcv_ex=zeros(p-(s-1),1); @ for storing expanding GARCH forecasts @
fcv_rm=zeros(p-(s-1),1); @ for storing RiskMetrics forecasts @
fcv_fi=zeros(p-(s-1),1); @ for storing expanding FIGARCH forecasts @
fcv_50=zeros(p-(s-1),1); @ for storing 0.50 rolling GARCH forecasts @
fcv_25=zeros(p-(s-1),1); @ for storing 0.25 rolling GARCH forecasts @
fcv_br=zeros(p-(s-1),1); @ for storing GARCH w/breaks forecasts @
fcv_ma=zeros(p-(s-1),1); @ for storing MA forecasts @
iter=0;
do until iter>p-s;

   /* Expanding GARCH forecast */

   b_garch_ex=pars_ex[iter+1,.]';
   ffc_ex=garch11_fc(b_garch_ex,rex[r+iter],hhat_ex[iter+1],s);
   h_ex=garch11_hhat(b_garch_ex,rex[first_ex[iter+1]:r+iter]);
   z_ex=rex[first_ex[iter+1]:r+iter]./sqrt(h_ex);
   fcv_ex[iter+1]=
      garch11_var_fc_boot(b_garch_ex,ffc_ex[1],s,z_ex[rows(z_ex)-(bl-1):rows(z_ex)],alpha,sims);

   /* RiskMetrics forecast */

   lambda=0.94;
   r2vec=rex[1:r+iter]^2;
   r2vec=rev(r2vec);
   tauseq=seqa(0,1,rows(r2vec));
   lambdatau=lambda^tauseq;
   ffc_rm=(1-lambda)*lambdatau'r2vec*ones(s,1);
   fcv_rm[iter+1]=constant_var_fc(ffc_rm[1],s,alpha,sims);

   /* Expanding FIGARCH forecast */

   b_garch_fi=pars_fi[iter+1,.]';
   ffc_fi=figarch11_fc(b_garch_fi,rex[1:(r+iter)],hhat_fi[iter+1],s);
   h_fi=figarch11_hhat(b_garch_fi,rex[first_fi[iter+1]:r+iter]);
   z_fi=rex[first_fi[iter+1]:r+iter]./sqrt(h_fi);
   fcv_fi[iter+1]=
      figarch11_var_fc_boot(b_garch_fi,ffc_fi[1],rex[first_fi[iter+1]:r+iter],s,z_fi[rows(z_fi)-(bl-1):rows(z_fi)],alpha,sims);

   /* 0.50 rolling GARCH forecast */

   b_garch_50=pars_50[iter+1,.]';
   ffc_50=garch11_fc(b_garch_50,rex[r+iter],hhat_50[iter+1],s);
   h_50=garch11_hhat(b_garch_50,rex[first_50[iter+1]:r+iter]);
   z_50=rex[first_50[iter+1]:r+iter]./sqrt(h_50);
   fcv_50[iter+1]=
      garch11_var_fc_boot(b_garch_50,ffc_50[1],s,z_50[rows(z_50)-(bl-1):rows(z_50)],alpha,sims);

   /* 0.25 rolling GARCH forecast */

   b_garch_25=pars_25[iter+1,.]';
   ffc_25=garch11_fc(b_garch_25,rex[r+iter],hhat_25[iter+1],s);
   h_25=garch11_hhat(b_garch_25,rex[first_25[iter+1]:r+iter]);
   z_25=rex[first_25[iter+1]:r+iter]./sqrt(h_25);
   fcv_25[iter+1]=
      garch11_var_fc_boot(b_garch_25,ffc_25[1],s,z_25[rows(z_25)-(bl-1):rows(z_25)],alpha,sims);

   /* GARCH w/breaks forecast */

   b_garch_br=pars_br[iter+1,.]';
   if b_garch_br[2]==0;
      fcc_br=b_garch_br[1]*ones(s,1);
      h_br=b_garch_br[1]*ones(r+iter-first_br[iter+1]+1,1);
      z_br=rex[first_br[iter+1]:r+iter]./sqrt(h_br);
      if rows(z_br)<bl;
         fcv_br[iter+1]=constant_var_fc_boot(b_garch_br[1],s,z_br,alpha,sims);
      else;
         fcv_br[iter+1]=
            constant_var_fc_boot(b_garch_br[1],s,z_br[rows(z_br)-(bl-1):rows(z_br)],alpha,sims);
      endif;
   else;
      ffc_br=garch11_fc(b_garch_br,rex[r+iter],hhat_br[iter+1],s);
      h_br=garch11_hhat(b_garch_br,rex[first_br[iter+1]:r+iter]);
      z_br=rex[first_br[iter+1]:r+iter]./sqrt(h_br);
      if rows(z_br)<bl;
         fcv_br[iter+1]=
            garch11_var_fc_boot(b_garch_br,ffc_br[1],s,z_br,alpha,sims);
      else;
         fcv_br[iter+1]=
            garch11_var_fc_boot(b_garch_br,ffc_br[1],s,z_br[rows(z_br)-(bl-1):rows(z_br)],alpha,sims);
      endif;
   endif;

   /* Moving average forecast */

   r2ma=meanc(rex[r+iter-249:r+iter]^2);
   ffc_ma=r2ma*ones(s,1);
   h_ma=r2ma*ones(250,1);
   z_ma=rex[r+iter-249:r+iter]./sqrt(h_ma);
   fcv_ma[iter+1]=constant_var_fc_boot(ffc_ma[1],s,z_ma,alpha,sims);

   iter=iter+1;
endo;
rexoos=rex[r+1:r+p];
rex2oos=rex[r+1:r+p]^2;
format 8,6;
"Number of in-sample observations  = " r;?;
"Forecast horizon                  = " s;?;
"Number of out-of-sample forecasts = " p-(s-1);?;

/* 5% VaR */

fcv_all=fcv_ex~fcv_50~fcv_25~fcv_br~fcv_ma;
fcv_me=meanc(fcv_all');
fcv_tm=(1/3)*(sumc(fcv_all')-minc(fcv_all')-maxc(fcv_all'));
{avg_ex,per_ex}=var_stats(fcv_ex,rexoos,s);
{avg_rm,per_rm}=var_stats(fcv_rm,rexoos,s);
{avg_fi,per_fi}=var_stats(fcv_fi,rexoos,s);
{avg_50,per_50}=var_stats(fcv_50,rexoos,s);
{avg_25,per_25}=var_stats(fcv_25,rexoos,s);
{avg_br,per_br}=var_stats(fcv_br,rexoos,s);
{avg_ma,per_ma}=var_stats(fcv_ma,rexoos,s);
{avg_me,per_me}=var_stats(fcv_me,rexoos,s);
{avg_tm,per_tm}=var_stats(fcv_tm,rexoos,s);

"5% VaR (average quantile and percentages)";?;

format 8,6;
"alpha              = " alpha;?;
"Expanding GARCH    = " avg_ex~per_ex;?;
"RiskMetrics        = " avg_rm~per_rm;?;
"Expanding FIGARCH  = " avg_ma~per_ma;?;
"0.50 rolling GARCH = " avg_50~per_50;?;
"0.25 rolling GARCH = " avg_25~per_25;?;
"GARCH w/breaks     = " avg_br~per_br;?;
"Moving average     = " avg_ma~per_ma;?;
"Mean               = " avg_me~per_me;?;
"Trimmed mean       = " avg_tm~per_tm;?;

"5% VaR loss function, mean (ratios)";?;

{var_loss_ex,mvar_ex}=loss_var(fcv_ex,rexoos,s,alpha);
{var_loss_rm,mvar_rm}=loss_var(fcv_rm,rexoos,s,alpha);
{var_loss_fi,mvar_fi}=loss_var(fcv_fi,rexoos,s,alpha);
{var_loss_50,mvar_50}=loss_var(fcv_50,rexoos,s,alpha);
{var_loss_25,mvar_25}=loss_var(fcv_25,rexoos,s,alpha);
{var_loss_br,mvar_br}=loss_var(fcv_br,rexoos,s,alpha);
{var_loss_ma,mvar_ma}=loss_var(fcv_ma,rexoos,s,alpha);
{var_loss_me,mvar_me}=loss_var(fcv_me,rexoos,s,alpha);
{var_loss_tm,mvar_tm}=loss_var(fcv_tm,rexoos,s,alpha);
format 8,6;
"alpha              = " alpha;?;
"Expanding GARCH    = " mvar_ex;?;
"RiskMetrics        = " mvar_rm~(mvar_rm/mvar_ex);?;
"Expanding FIGARCH  = " mvar_fi~(mvar_fi/mvar_ex);?;
"0.50 rolling GARCH = " mvar_50~(mvar_50/mvar_ex);?;
"0.25 rolling GARCH = " mvar_25~(mvar_25/mvar_ex);?;
"GARCH w/breaks     = " mvar_br~(mvar_br/mvar_ex);?;
"Moving average     = " mvar_ma~(mvar_ma/mvar_ex);?;
"Mean               = " mvar_me~(mvar_me/mvar_ex);?;
"Trimmed mean       = " mvar_tm~(mvar_tm/mvar_ex);?;
tend=hsec-tstart;
sec_tend=tend/100;
min_tend=sec_tend/60;
hr_tend=min_tend/60;
"Time to run program = " tend;
"Seconds             = " sec_tend;
"Minutes             = " min_tend;
"Hours               = " hr_tend;?;

/* Procedure definition area */

/**********************  PROC VARLAGS  *****************************
**   last update: 5 Dec 95      previous: 15 June 94
**   AUTHOR		 
**        Alan G. Isaac
**   FORMAT		 
**        {x,xlags} = varlags(var,lags)
**   INPUT		 
**        var  - T x K matrix
**        lags - scalar, number of lags of var (a positive integer)
**   OUTPUT		 
**        x -     (T - lags) x K matrix, the last T-lags rows of var
**        xlags - (T - lags) x lags*cols(var) matrix,
**                being the 1st through lags-th
**                values of var corresponding to the values in x
**                i.e, the appropriate rows of x(-1)~x(-2)~etc.
**   GLOBAL VARIABLES: none
**********************************************************************/
proc(2)=varlags(var,lags);
    local xlags;
    xlags = shiftr((ones(1,lags) .*. var)',seqa(1-lags,1,lags)
                                            .*. ones(cols(var),1),miss(0,0))';
    retp(trimr(var,lags,0),trimr(xlags,0,lags));
endp;

/****************************************************************
PROC: GARCH11_FC

The procedure generates a vector of volatility forecasts from
1 to s steps ahead for a stationary GARCH(1,1) model using eq.
(4.117) in Franses and van Dijk (2000).

Format: h_ts=garch11_fc(theta,e,hhat,s)

Input:

theta = vector of estimated GARCH parameters (omega|alpha|beta)
e     = scalar, obs of e_t
hhat  = scalar, estimate of h_t
s     = longest forecast horizon

Output:

h_ts = s-vector forecasts of h_t+1|t,...,h_t+s|t

Reference

P.H. Franses and D. van Dijk (2000), Non-Linear Time Series
Models in Empirical Finance (Cambridge, UK: Cambridge University
Press)
****************************************************************/
proc(1)=garch11_fc(theta,e,hhat,s);
   local omega,alpha,beta,h_t1,h_ts,iter,sig2;
   omega=theta[1];
   alpha=theta[2];
   beta=theta[3];
   h_t1=omega+alpha*e^2+beta*hhat;
   h_ts=h_t1;
   sig2=omega/(1-alpha-beta);
   if s>1;
      iter=2;
      do until iter>s;
         h_ts=h_ts|(sig2+(alpha+beta)^(iter-1)*(h_t1-sig2));
         iter=iter+1;
      endo;
   endif;
   retp(h_ts);
endp;

/*****************************************************************
proc: GARCH11_HHAT
*****************************************************************/
proc garch11_hhat(x,e);
   local bigt,mean_e2,alphapart,h;
   bigt=rows(e);
   mean_e2=meanc(e^2);
   alphapart=x[1]+x[2]*missrv(lagn(e^2,1),mean_e2);
   h=recserar(0|alphapart,mean_e2,x[3]);
   h=h[2:bigt+1];
   retp(h);
endp;

/****************************************************************
PROC: FIGARCH11_FC

The procedure generates a vector of volatility forecasts from
1 to s steps ahead for a FIGARCH(1,d,1).

Format: h_ts=figarch11_fc(params,e,hhat,s)

Input:

params = vector of estimated FIGARCH(1,d,1) parameters
         (omega|phi|beta|d)
e      = vector of e_t observations
hhat   = scalar, estimate of h_t
s      = longest forecast horizon

Output:

h_ts = s-vector forecasts of h_t+1|t,...,h_t+s|t
****************************************************************/
proc(1)=figarch11_fc(params,e,hhat,s);
   local x,bigt,tl,e2,vd,vp,vp1,theta1,fc,iter;
   x=params;
   bigt=rows(e);
   tl=1000;
   e2=e^2;
   vd=x[4];
   vp=1|cumprodc(seqa(-vd,1,tl-1)./seqa(1,1,tl-1));
   vp1=conv(1|-x[2],vp,1,tl);
   theta1=vp1[2:tl];
   fc=zeros(s,1);
   fc[1]=x[1]+x[3]*(hhat-e2[bigt])-sumc(theta1.*rev(e2[(bigt-(tl-2)):bigt]));
   iter=2;
   do until iter>s;
      fc[iter]=x[1]-sumc(theta1[1:(iter-1)].*rev(fc[1:(iter-1)]))
               -sumc(theta1[iter:(tl-1)].*rev(e2[(bigt-(tl-(iter+1))):bigt]));
      iter=iter+1;
   endo;
   retp(fc);
endp;

/*****************************************************************
PROC: FIGARCH11_HHAT
*****************************************************************/
proc figarch11_hhat(x,e);
   local bigt,tl,e2,h0,e3,vd,vp,vp1,kn,vp2,vw,vw1,h1,h,z;
   bigt=rows(e);
   tl=1000;
   e2=e^2;
   h0=(e'e)/bigt;
   e3=h0|e2[1:bigt-1];
   vd=x[4];
   vp=1|cumprodc(seqa(-vd,1,tl-1)./seqa(1,1,tl-1));
   vp1=conv(1|-x[2],vp,1,tl);
   kn=ones(tl,1);
   vp2=conv(vp1,kn,tl+1,2*tl-1);
   vp2=vp2|zeros(bigt-tl+1,1);
   vw=conv(vp1,e2,1,bigt);
   vw1=vw+vp2*h0;
   h1=x[1]+e2[1]-vw1[1];
   h=recserar(0|x[1]+e2-x[3]*e3-vw1,h1,x[3]);
   h=h[2:bigt+1];
   retp(h);
endp;

/*****************************************************************
PROC: GARCH11_VAR_FC_BOOT

This procedure calculates a VaR quantile forecast for aggregate
returns for a GARCH(1,1) model at a horizon of s using a bootstrap
procedure. The process is assumed to have zero conditional and
unconditional means.

Format: varq=garch11_var_fc_boot(b_garch,fc,s,zhat,alpha,sims);

Intput:

b_garch = 3-vector of GARCH(1,1) estimates (omega|alpha|beta)
fc      = one-step-ahead conditional variance forecast
s       = forecast horizon
zhat    = vector of standardized residuals for resampling
alpha   = probability of loss
sims    = number of simulations

Output:

varq = VaR quantile forecast
*****************************************************************/
proc(1)=garch11_var_fc_boot(b_garch,fc,s,zhat,alpha,sims);
   local tretstar,itersims,zstar,hstar,estar,iters,alphai,varq;
   zhat=zhat-meanc(zhat);
   tretstar=ones(sims,1);
   itersims=1;
   do until itersims>sims;
      zstar=zhat[ceil(rows(zhat)*rndu(s,1))];
      if s>2;
         hstar=fc|zeros(s-1,1);
      else;
         hstar=fc;
      endif;
      estar=zeros(s,1);
      estar[1]=sqrt(hstar[1])*zstar[1];
      if s>2;
         iters=2;
         do until iters>s;
            hstar[iters]=b_garch[1]+b_garch[2]*estar[iters-1]^2
               +b_garch[3]*hstar[iters-1];
            estar[iters]=sqrt(hstar[iters])*zstar[iters];
            iters=iters+1;
         endo;
      endif;
      tretstar[itersims]=sumc(estar);
      itersims=itersims+1;
   endo;
   alphai=round(alpha*sims);
   tretstar=sortc(tretstar,1);
   varq=tretstar[alphai];
   retp(varq);
endp;

/*****************************************************************
PROC: CONSTANT_VAR_FC

This procedure calculates a VaR quantile forecast for aggregate
returns for a constant variance model at a horizon of s using a
standard normal distribution for the standardized residuals. The
process is assumed to have zero conditional and unconditional means.

Format: varq=constant_var_fc(fc,s,alpha,sims);

Intput:

fc      = variance forecast
s       = forecast horizon
alpha   = probability of loss
sims    = number of simulations

Output:

varq = VaR quantile forecast
*****************************************************************/
proc(1)=constant_var_fc(fc,s,alpha,sims);
   local tretstar,itersims,estar,alphai,varq;
   tretstar=ones(sims,1);
   itersims=1;
   do until itersims>sims;
      estar=sqrt(fc)*rndn(s,1);
      tretstar[itersims]=sumc(estar);
      itersims=itersims+1;
   endo;
   alphai=round(alpha*sims);
   tretstar=sortc(tretstar,1);
   varq=tretstar[alphai];
   retp(varq);
endp;

/*****************************************************************
PROC: CONSTANT_VAR_FC_BOOT

This procedure calculates a VaR quantile forecast for aggregate
returns for a constant variance model at a horizon of s using a
bootstrap procedure. The process is assumed to have zero conditional
and unconditional means.

Format: varq=constant_var_fc_boot(fc,s,zhat,alpha,sims);

Intput:

fc      = variance forecast
s       = forecast horizon
zhat    = vector of standardized residuals for resampling
alpha   = probability of loss
sims    = number of simulations

Output:

varq = VaR quantile forecast
*****************************************************************/
proc(1)=constant_var_fc_boot(fc,s,zhat,alpha,sims);
   local tretstar,itersims,estar,alphai,varq;
   zhat=zhat-meanc(zhat);
   tretstar=ones(sims,1);
   itersims=1;
   do until itersims>sims;
      estar=sqrt(fc)*zhat[ceil(rows(zhat)*rndu(s,1))];
      tretstar[itersims]=sumc(estar);
      itersims=itersims+1;
   endo;
   alphai=round(alpha*sims);
   tretstar=sortc(tretstar,1);
   varq=tretstar[alphai];
   retp(varq);
endp;

/*****************************************************************
PROC: FIGARCH11_VAR_FC_BOOT

This procedure calculates a VaR quantile forecast for aggregate
returns for a FIGARCH(1,d,1) model at a horizon of s using a
bootstrap procedure. The process is assumed to have zero
conditional and unconditional means.

Format: varq=figarch11_var_fc_boot(b_figarch,fc,e,s,zhat,alpha,sims);

Intput:

b_figarch = 4-vector of FIGARCH(1,d,1) estimates (omega|alpha|beta|d)
fc        = one-step-ahead conditional variance forecast
e         = T-vector of observations
s         = forecast horizon
zhat      = vector of standardized residuals for resampling
alpha     = probability of loss
sims      = number of simulations

Output:

varq = VaR quantile forecast
*****************************************************************/
proc(1)=figarch11_var_fc_boot(b_figarch,fc,e,s,zhat,alpha,sims);
   local bigt,tl,vd,vp,vp1,tretstar,itersims,zstar,hstar,estar,
   iters,elagstar,e2lagstar,alphai,varq;
   zhat=zhat-meanc(zhat);
   bigt=rows(e);
   tl=1000;
   vd=b_figarch[4];
   vp=1|cumprodc(seqa(-vd,1,tl-1)./seqa(1,1,tl-1));
   vp1=conv(1|-b_figarch[2],vp,1,tl);
   vp1=vp1[2:rows(vp1)];
   tretstar=ones(sims,1);
   itersims=1;
   do until itersims>sims;
      zstar=zhat[ceil(rows(zhat)*rndu(s,1))];
      if s>2;
         hstar=fc|zeros(s-1,1);
      else;
         hstar=fc;
      endif;
      estar=zeros(s,1);
      estar[1]=sqrt(hstar[1])*zstar[1];
      if s>1;
         iters=2;
         do until iters>s;
            elagstar=e[bigt-tl+iters+1:bigt]|estar[1:iters-1];
            e2lagstar=(rev(elagstar))^2;
            hstar[iters]=b_figarch[1]+b_figarch[3]*hstar[iters-1]
               -b_figarch[3]*e2lagstar[1]-sumc(vp1.*e2lagstar);
            estar[iters]=sqrt(hstar[iters])*zstar[iters];
            iters=iters+1;
         endo;
      endif;
      tretstar[itersims]=sumc(estar);
      itersims=itersims+1;
   endo;
   alphai=round(alpha*sims);
   tretstar=sortc(tretstar,1);
   varq=tretstar[alphai];
   retp(varq);
endp;

/*****************************************************************
PROC: VAR_STATS

This procedure calculates the average Value at Risk quantile
forecasts and the percentage of times the actual return is less
than the Value at Risk quantile forecast.

Format: {varq_avg,varpercent}=var_stats(fc,a,s);

Intput:

fc = vector of VaR quantile forecasts
a  = vector of actual values for returns
s  = forecast horizon

Output:

varq_avg   = average VaR quantile forecast
varpercent = percent
*****************************************************************/
proc(2)=var_stats(fc,a,s);
   local n,varq_avg,aagg,iter,varpercent;
   n=rows(fc);
   varq_avg=meanc(fc);
   aagg=zeros(n,1);
   iter=1;
   do until iter>n;
      aagg[iter]=sumc(a[iter:iter+(s-1)]);
      iter=iter+1;
   endo;
   varpercent=sumc(aagg.<fc)/n;
   retp(varq_avg,varpercent);
endp;

/*****************************************************************
PROC: LOSS_VAR

This procedure calculates values for the VaR-based loss
function described in Section 4.3 of Gonzalez-Rivera et al.
(2004) for aggregate returns. It also calculates the mean
loss.

Format: {loss_series,loss_average}=loss_var(fc,a,s,alpha);

Intput:

fc    = vector of VaR quantile forecasts
a     = vector of actual values for one-period returns
s     = forecast horizon
alpha = probability of loss

Output:

loss_series  = vector of VaR-based loss function values
loss_average = mean loss

Reference

G. Gonzalez-Rivera, T.-H. Lee, and S. Mishra (2004),
"Forecasting Volatility: A Reality Check Based on Option
Prcing, Utility Function, Value-at-Risk, and Predictive
Likelihood," International Journal of Forecasting, 20(4),
629-645
*****************************************************************/
proc(2)=loss_var(fc,a,s,alpha);
   local n,aagg,iter,d_alpha,loss_series,loss_average;
   n=rows(fc);
   aagg=zeros(n,1);
   iter=1;
   do until iter>n;
      aagg[iter]=sumc(a[iter:iter+(s-1)]);
      iter=iter+1;
   endo;
   d_alpha=aagg.<fc;
   loss_series=(alpha-d_alpha).*(aagg-fc);
   loss_average=meanc(loss_series);
   retp(loss_series,loss_average);
endp;
