import java.util.*;

//-------------------------------------------------------------------
//		ModelParameters
//-------------------------------------------------------------------

class ModelParameters
{
 public ModelParameters(double q0,double de,double du,double df,double jd,double jc,double jv,double R,double beta,double gamma,double Scale)
 {
  P=new double[11];
  P[0]=q0;P[1]=de;P[2]=du;P[3]=df;P[4]=jd;P[5]=jc;P[6]=jv;P[7]=R;P[8]=beta;P[9]=gamma;P[10]=Scale;
 }
 
 public double P[];
}

//-------------------------------------------------------------------
//		IntegrationWeights
//-------------------------------------------------------------------

class IntegrationDensity
{
 public IntegrationDensity(double jc,double jv)
 {
  this.jc=jc;this.jv=jv;
  F_minX=-jv*jv/2;
  F_Step=0.00001;
 }
	
 private double f(double x)
 {
  return 1/(Math.sqrt(2*Math.PI)*jv)*Math.exp(-(x+jv*jv/2)*(x+jv*jv/2)/(2*jv*jv));
 }
 
 private double F(double x)
 {
  double a=0,b=1;
  if (x>-jv*jv/2) {x=-jv*jv-x;a=1;b=-1;}
  if (x==F_minX) return a+b*F_min;
  if (x>F_minX) {F_minX=-jv*jv/2;F_min=0.5;}
  while (x<F_minX)
  {
   F_minX-=F_Step;
   F_min-=f(F_minX)*F_Step;
  }
  return a+b*F_min;
 }

 public double F_min=0.5,F_minX,F_Step;

 public double ff(double x)
 {
  return jc*f(x)*Math.exp(-jc*(1-F(x)));
 }

 public double jc,jv;
}

class IntegrationWeights
{
 public IntegrationWeights(Grid G,double jc,double jv)
 {
  this.jc=jc;this.jv=jv;
  
  IntegrationDensity ID=new IntegrationDensity(jc,jv);
  double W_Step=0.00001,W_Thres=0.00001;
  
  Weights=new double[G.numPoints];
  for (int i=0;i<G.numPoints;i++) Weights[i]=0;

  double Int=0,Pos=-jv*jv/2,Dens;
  do
  {
   Dens=ID.ff(Pos);
   Weights[G.getIndex(Math.exp(Pos))]+=Dens*W_Step;
   Pos-=W_Step;
   Int+=Dens;
  }
  while (Dens>W_Thres);

  Pos=-jv*jv/2+W_Step;
  do
  {
   Dens=ID.ff(Pos);
   Weights[G.getIndex(Math.exp(Pos))]+=Dens*W_Step;
   Pos+=W_Step;
   Int+=Dens;
  }
  while (Dens>W_Thres);

  Int*=W_Step;
  Weights[0]+=(1-Math.exp(-jc)-Int)/2;Weights[G.numPoints-1]+=(1-Math.exp(-jc)-Int)/2;

 }

 public double jc,jv,Weights[];

}

//-------------------------------------------------------------------
//		JobOffer
//-------------------------------------------------------------------

class JobOffer
{
 public JobOffer(double jc,double jv,int Precision)
 {
  this.Precision=Precision;
  this.jc=jc;this.jv=jv;
  Data=new int[Precision];
  
  double F=0,kFact=1;
  int    k=0;
  
  for (int i=0;i<Precision;i++)
  {
   while (F*Precision<=(double)i) 
   {
    k+=1;
    kFact*=k;
    F+=Math.exp(k*Math.log(jc))/(kFact*(Math.exp(jc)-1));
   }
   Data[i]=k;
  }
 }
 
 public double getOffer(Random R)
 {
  int Num=Data[R.nextInt(Precision)];
  double best=Double.NEGATIVE_INFINITY;
  for (int i=0;i<Num;i++) best=Math.max(best,Math.exp(R.nextGaussian()*jv-jv*jv/2));
  return best;
 }
 
 private int    Precision,Data[];
 private double jc,jv;
}

//-------------------------------------------------------------------
//		Grids
//-------------------------------------------------------------------

abstract class Grid
// a Grid has at least 1 Point
{
 public Grid() {}
 public int numPoints;
 public double Values[];
 public int getIndex(double value) 
 {
  if (numPoints==1) return 0;
  if (value<=Values[0]) return 0;
  if (value>=Values[numPoints-1]) return numPoints-1;
  int min=0,max=numPoints-1,i;
  
  while (max-min>1)
  {
   i=(min+max)>>1;
   if (value>Values[i]) min=i;
   else max=i;
  }

  if (value-Values[min]<=Values[max]-value) return min;
  return max;
 };

 public final int getInterpolation(double value,int[] Indices,double[] Weights)
 {
  // Indizes sortiert nach Groesse
  int i=getIndex(value);

  if (value==Values[i]) {Indices[0]=i;Weights[0]=1;return 1;}
  if (value>Values[i]) {Indices[0]=i;Indices[1]=i+1;}
  else {Indices[0]=i-1;Indices[1]=i;}
  if ((Indices[0]<0) || (Indices[1]>=numPoints)) {Indices[0]=i;Weights[0]=1;return 1;}

  Weights[1]=(value-Values[Indices[0]])/(Values[Indices[1]]-Values[Indices[0]]);
  Weights[0]=1-Weights[1];
  return 2;
 }

}

class ZeroGrid
extends Grid
{
 public ZeroGrid() 
 {
  numPoints=1;
  Values=new double[1];
  Values[0]=0;
 }
}

class EvenGrid
extends Grid
{
 public EvenGrid(int Num,double min,double max)
 {
  numPoints=Num;
  Values=new double[Num];
  if (Num>1) for (int i=0;i<Num;i++) Values[i]=(min*(Num-i-1)+max*i)/(Num-1);
  else Values[0]=min;
 }
 
 public final int getIndex(double value)
 {
  if (numPoints==1) return 0;
  int i=(int)Math.round((value-Values[0])*(numPoints-1)/(Values[numPoints-1]-Values[0]));
  if (i<0) i=0;
  if (i>=numPoints) i=numPoints-1;
  return i;
 }
}

class EqualWeightsGrid
extends Grid
{
 public EqualWeightsGrid(int Num,double jc,double jv)
 {
  numPoints=Num;
  Values=new double[Num];
  if (Num>1) CritValues=new double[Num-1];

  IntegrationDensity ID=new IntegrationDensity(jc,jv);
  double W_Step=0.00001,W_Thres=0.00001;

  double Int=0,Pos=-jv*jv/2,Dens;
  do
  {
   Dens=ID.ff(Pos);
   Pos-=W_Step;
   Int+=Dens;
  }
  while (Dens>W_Thres);

  double MedInt=Int*W_Step;

  Pos=-jv*jv/2+W_Step;
  do
  {
   Dens=ID.ff(Pos);
   Pos+=W_Step;
   Int+=Dens;
  }
  while (Dens>W_Thres);

  Int*=W_Step;
  MedInt+=(1-Math.exp(-jc)-Int)/2;
  
  int IndexDown=(int)Math.floor(MedInt/(1-Math.exp(-jc))*2*Num),IndexUp=IndexDown+1;

  Int=MedInt;
  double nextCrit=(double)IndexDown*(1-Math.exp(-jc))/(2*Num);
  Pos=-jv*jv/2;
  while (IndexDown>0)
  {
   Int-=ID.ff(Pos)*W_Step;
   Pos-=W_Step;
   if (Int<=nextCrit)
   {
    if ((IndexDown&1)==0) CritValues[(IndexDown>>1)-1]=Math.exp(Pos);
    else Values[IndexDown>>1]=Math.exp(Pos);
    IndexDown--;
    nextCrit=(double)IndexDown*(1-Math.exp(-jc))/(2*Num);
   }
  }
  
  Int=MedInt;
  nextCrit=(double)IndexUp*(1-Math.exp(-jc))/(2*Num);
  Pos=-jv*jv/2+W_Step;
  while (IndexUp<2*Num)
  {
   Int+=ID.ff(Pos)*W_Step;
   Pos+=W_Step;
   if (Int>=nextCrit)
   {
    if ((IndexUp&1)==0) CritValues[(IndexUp>>1)-1]=Math.exp(Pos);
    else Values[IndexUp>>1]=Math.exp(Pos);
    IndexUp++;
    nextCrit=(double)IndexUp*(1-Math.exp(-jc))/(2*Num);
   }
  }

 }
 
 private double[] CritValues;
 
 public final int getIndex(double value)
 {
  if (numPoints==1) return 0;
  if (value<=CritValues[0]) return 0;
  if (value>=CritValues[numPoints-2]) return numPoints-1;
  int min=0,max=numPoints-2,i;
  
  while (max-min>1)
  {
   i=(min+max)>>1;
   if (value>CritValues[i]) min=i;
   else max=i;
  }

  return max;
 }

}

class LinWeightsGrid
extends Grid
{
 public LinWeightsGrid(int Num,double relLast,double jc,double jv,int Num2,double fac2)
 {
  if (Num<2) Num=2;
  numPoints=Num+Num2;
  Values=new double[numPoints];
  CritValues=new double[numPoints-1];
  
  double Sum=Num;for (int i=1;i<Num;i++) Sum+=(relLast-1)*i/(Num-1);
  double[] CritPoints=new double[2*Num-1];
  for (int i=0;i<Num;i++)
  {
   double delta=(1-Math.exp(-jc))*(1+i*(relLast-1)/(Num-1))/(2*Sum);
   CritPoints[2*i]=delta+((i>0)?CritPoints[2*i-1]:0);
   if (i<Num-1) CritPoints[2*i+1]=CritPoints[2*i]+delta;
  } 

  IntegrationDensity ID=new IntegrationDensity(jc,jv);
  double W_Step=0.00001,W_Thres=0.00001;

  double Int,Pos,Dens;
  double MedInt;
  boolean ok;

  do
  {
   ok=true;
   Int=0;Pos=-jv*jv/2;
   do
   {
    Dens=ID.ff(Pos);
    Pos-=W_Step;
    Int+=Dens;
   }
   while (Dens>W_Thres);
   double DIL=Int*W_Step;

   MedInt=Int*W_Step;

   Pos=-jv*jv/2+W_Step;
   do
   {
    Dens=ID.ff(Pos);
    Pos+=W_Step;
    Int+=Dens;
   }
   while (Dens>W_Thres);
   double DIR=Int*W_Step-DIL;
 
   Int*=W_Step;
   MedInt+=(1-Math.exp(-jc)-Int)/2;

   if ((MedInt-DIL>CritPoints[0]) || (MedInt+DIR<CritPoints[2*Num-2])) {ok=false;ID.F_Step*=0.1;W_Thres*=0.1;}
  }
  while (!ok);
  
  
  int IndexDown,IndexUp;
  for (IndexUp=0;(MedInt>CritPoints[IndexUp]) && (IndexUp<2*Num-1);IndexUp++);IndexDown=IndexUp-1;

  Int=MedInt;
  Pos=-jv*jv/2;
  while (IndexDown>=0)
  {
   Int-=ID.ff(Pos)*W_Step;
   Pos-=W_Step;
   if (Int<=CritPoints[IndexDown])
   {
    if ((IndexDown&1)==1) CritValues[IndexDown>>1]=Math.exp(Pos);
    else Values[IndexDown>>1]=Math.exp(Pos);
    IndexDown--;
   }
  }
  
  Int=MedInt;
  Pos=-jv*jv/2+W_Step;
  while (IndexUp<2*Num-1)
  {
   Int+=ID.ff(Pos)*W_Step;
   Pos+=W_Step;
   if (Int>=CritPoints[IndexUp])
   {
    if ((IndexUp&1)==1) CritValues[IndexUp>>1]=Math.exp(Pos);
    else Values[IndexUp>>1]=Math.exp(Pos);
    IndexUp++;
   }
  }

  for (int i=Num;i<numPoints;i++)
  {
   CritValues[i-1]=Values[i-1]*Math.sqrt(fac2);
   Values[i]=Values[i-1]*fac2;
  }
 }
 
 private double[] CritValues;
 
 public final int getIndex(double value)
 {
  if (numPoints==1) return 0;
  if (value<=CritValues[0]) return 0;
  if (value>=CritValues[numPoints-2]) return numPoints-1;
  int min=0,max=numPoints-2,i;
  
  while (max-min>1)
  {
   i=(min+max)>>1;
   if (value>CritValues[i]) min=i;
   else max=i;
  }

  return max;
 }

}

class LogGrid
extends Grid
{
 public LogGrid(int num,double min,double max,double dis)
 {
  numPoints=num;
  Values=new double[num];
  double u=0,op=0,on=0;
  int x=0;
  if ((min<0) && (max>0))
  {
   u=dis;op=max+dis;on=-min+dis;
  }
  else if (min>=0)
  {
   u=on=min+dis;op=max+dis;x=1;
  }
  else if (max<=0)
  {
   u=op=-max+dis;on=-min+dis;x=1;
  }
  else System.out.println("LogGrid Error");
  
  Pu=(int)Math.round(Math.log(on/u)/Math.log(op*on/(u*u))*(num-2*x))+x;
  Po=num-Pu;
  
  Step=Math.log(op*on/(u*u))/(num-x);
  
  for (int i=0;i<Pu;i++) Values[i]=min*Math.exp(-Step*i);
  for (int i=0;i<Po;i++) Values[num-1-i]=max*Math.exp(-Step*i);
    
 }

 public final int getIndex(double value) 
 {
  if (numPoints==1) return 0;
  if (value<=Values[0]) return 0;
  if (value>=Values[numPoints-1]) return numPoints-1;
  int min=0,max=numPoints-1,i;
  
  while (max-min>1)
  {
   i=(min+max)>>1;
   if (value>Values[i]) min=i;
   else max=i;
  }
  if ((Values[min]<=0) && (Values[max]>=0))
  {
   if (value-Values[min]<=Values[max]-value) return min;
   return max;
  }

  if (value>0)
  {  
   if (value*value<=Values[min]*Values[max]) return min;
   return max;
  }
  if (value*value<=Values[min]*Values[max]) return max;
  return min;
 }

 double Step;
 int Pu,Po;
}

class SmallerGrid
extends Grid
{
 public SmallerGrid(Grid Basis,int Size)
 {
  numPoints=(int)Math.ceil(Basis.numPoints/Size)+1;
  Values=new double[numPoints];
  for (int i=0;i<numPoints-1;i++) Values[i]=Basis.Values[Size*i];
  Values[numPoints-1]=Basis.Values[Basis.numPoints-1];
 }
}

class PLGrid
extends Grid
{
 public PLGrid(double[] Values)
  {
  numPoints=Values.length;
  this.Values=Values;
 }
}

class ELogGrid
extends Grid
{
 public ELogGrid(int Anz,double Min,double Max,double Scale)
 {
  numPoints=Anz;
  Values=new double[Anz];
  Values[0]=Math.min(Min,Max);Values[Anz-1]=Math.max(Min,Max);
  double Factor=Math.exp(Math.log(Scale)/(Anz-1)),Pos=1;
  for (int i=1;i<Anz-1;i++)
  {
   Pos*=Factor;
   Values[(Max>Min)?i:Anz-1-i]=(Pos-1)/(Scale-1)*(Max-Min)+Min;
  }
 }
}

class CombinedGrid
extends Grid
{
 public CombinedGrid(Grid G1,Grid G2)
 {
  numPoints=0;
  Values=new double[G1.numPoints+G2.numPoints];
  for (int i=0;i<G1.numPoints;i++) Values[numPoints++]=G1.Values[i];
  for (int i=0;i<G2.numPoints;i++) if ((numPoints==0) || (G2.Values[i]>Values[numPoints-1])) Values[numPoints++]=G2.Values[i];
 }
}

//-------------------------------------------------------------------
//		Period
//-------------------------------------------------------------------

class Period
{
 public Period(int Type,int Basis,Grid AssetGrid,Grid MatchGrid,Grid HKGrid,Grid BenefitGrid,Grid PensionGrid,Grid UIEGrid,Grid UIUGrid)
 {
  this.Type=Type;this.Basis=Basis;LastFull=-1;
  this.AssetGrid=AssetGrid;this.MatchGrid=MatchGrid;this.HKGrid=HKGrid;this.BenefitGrid=BenefitGrid;this.PensionGrid=PensionGrid;this.UIEGrid=UIEGrid;this.UIUGrid=UIUGrid;
  
  if (Basis==0) switch (Type)
  {
   case 1:
    ValueE=new float[PensionGrid.numPoints][HKGrid.numPoints][MatchGrid.numPoints][AssetGrid.numPoints];
    ValueU=new float[UIUGrid.numPoints][PensionGrid.numPoints][BenefitGrid.numPoints][HKGrid.numPoints][AssetGrid.numPoints];
    ConsE=new float[UIEGrid.numPoints][PensionGrid.numPoints][HKGrid.numPoints][MatchGrid.numPoints][AssetGrid.numPoints];
    ConsU=new float[UIUGrid.numPoints][PensionGrid.numPoints][BenefitGrid.numPoints][HKGrid.numPoints][AssetGrid.numPoints];
    Size=4*AssetGrid.numPoints*HKGrid.numPoints*PensionGrid.numPoints*(MatchGrid.numPoints*(1+UIEGrid.numPoints)+2*BenefitGrid.numPoints*UIUGrid.numPoints);
    break;
   case 2:
    ConsR=new float[PensionGrid.numPoints][AssetGrid.numPoints];
    Size=4*PensionGrid.numPoints*AssetGrid.numPoints;
  }
  else Size=0;
  
  reset(MatchGrid);
 }
 
 public void reset(Grid MG)
 {
  if (Type==1) MatchGrid=MG;
 }
 
 public Grid  AssetGrid,MatchGrid,HKGrid,BenefitGrid,PensionGrid,UIEGrid,UIUGrid;
 public int   Type,Basis,LastFull;
 public float ValueE[][][][],ValueU[][][][][];
 public float ConsE[][][][][],ConsU[][][][][],ConsR[][];
 
 public int Size;
}


//-------------------------------------------------------------------
//		Institutions
//-------------------------------------------------------------------

class Institutions
{
 public Institutions(int L,int Lr,double q0,double R,double Scale)
 {
  this.r=R-1;this.q0=q0;this.L=L;this.Lr=Lr;this.Scale=Scale;
  CM=new double[L+1];
  for(int i=1;i<=L;i++) CM[i]=1;
 }

 public void reset(double q0,double R,double Scale)
 {
  this.q0=q0;this.r=R-1;this.Scale=Scale;
 }

 public double nextBenefit(double Match,double HK,double UI,int j) {return 0;} //only when entering unemployment
 public double nextPension(double Match,double HK,double Pension,double UI,int e,int j) {return 0;}
 public double nextUI(double Match,double HK,double UI,int e,int j,int en) {return 0;} // when entering employment: Value CONSTANT
 public double empUI=0; // equal to nextUI(*,*,*,0,*,1)
 
 public double firstPension() {return 0;}
 public double firstUI() {return 0;}
 
 public Grid BenefitGrid() {return new ZeroGrid();};
 public Grid PensionGrid() {return new ZeroGrid();};
 public Grid UIEGrid() {return new ZeroGrid();};
 public Grid UIUGrid() {return new ZeroGrid();};
 
 public double Brutto(double y)
 {
  return y;
 }

 public double y(double Assets,double MatchOrBenefit,double HK,double Pension,double UI,int e,int j)
 {
  if ((e==1) && (j<Lr)) return MatchOrBenefit*(q0+HK);
  return 0;
 }

 public int MakePeriods(Period[] P,Grid MG)
 {
  return 0;
 }
 
 double r,q0,Scale,CM[];
 int L,Lr;
}

class DeutschlandHartz
extends Institutions
// Deutschland 1997
// UI: Dauer in Monaten
{
 public DeutschlandHartz(int L,int Lr,double q0,double R,double Scale,double phi)
 {
  super(L,Lr,q0,R,Scale);

  this.phi=phi;
  FamSize=new double[L+1];
  Children=new double[L+1];
  for (int i=0;i<L/24;i++) for (int j=1;j<=24;j++)
  {
   FamSize[i*24+j]=(fs[i]*(25-j)+fs[i+1]*(j-1))/24;
   Children[i*24+j]=(cn[i]*(25-j)+cn[i+1]*(j-1))/24;
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
  }
 }

 public void reset(double q0,double R,double Scale,double phi)
 {
  super.reset(q0,R,Scale);
  
  this.phi=phi;
  for (int i=0;i<60;i++) for (int j=0;j<24;j++)
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
 }

 private double Age(int j)
 {
  return 20+((double)j-1)/24;
 }

 public double Brutto(double y)
 {
  if (y>9753.25/2) return y-1556.975/2;
  if (y>7432.275/2) return (y-458.175/2)/1.134;
  return y/1.2085;
 }

 public double nextBenefit(double Match,double HK,double UI,int j)
 {
  // max. 8200 DM/Monat als Benefit-Basis
  // Benefit speichert EK bis Obergrenze
  return Math.min(Scale*Match*(q0+HK),(8200+1556.975)/2);
 }
 
 public double nextPension(double Match,double HK,double Pension,double UI,int e,int j)
 {
  // 0<=Pension<=1.88718117
  if (j>=Lr) return Pension;
  double yL=0;
  if (e==1) yL=Brutto(Scale*Match*(q0+HK));
  else if (UI>=0) yL=Brutto(Match)*0.8;
  return (Pension*(j-1)+Math.min(yL /*"Brutto-EK"*/,8200/2)/(52143/24))/j;
 }
 
 public double nextUI(double Match,double HK,double UI,int e,int j,int en)
 {
// UI (e==1): Anzahl der vor t bereits gearbeiteten Perioden
// Achtung: guenstigere die Altersgrenzen bis April 97; Ab Mai 97 alles +3 Jahre
  if (e==1)
  {
   UI+=0.5;
   if ((Age(j)<44) && (UI>24)) UI=24;
   else if ((Age(j)<46.34) && (UI>36)) UI=36;
   else if ((Age(j)<51.34) && (UI>44)) UI=44;
   else if ((Age(j)<56) && (UI>52)) UI=52;
   else if (UI>64) UI=64;
   if (en==1) return UI;
   if (UI<12) return 0;
   UI=Math.floor(UI/2);
   if (Age(j)<45) UI=Math.min(UI,12);
   else if (Age(j)<47) UI=Math.min(UI,18);
   else if (Age(j)<52) UI=Math.min(UI,22);
   else if (Age(j)<57) UI=Math.min(UI,26);
   return UI+24;
  }
  if (en==1) return 0;
  if (UI>0) return UI-0.5;
  return UI;
 }

 private double ErtragsAnteil(int j)
 {
  double Age=(j-1)/24+20;
  if (Age<68) return (92-Age)/100;
  if (Age<81) return (91-Age)/100;
  if (Age<85) return (92-Age)/100;
  if (Age<87) return (93-Age)/100;
  if (Age<90) return (94-Age)/100;
  if (Age<92) return 0.05;
  if (Age<94) return 0.04;
  if (Age<97) return 0.03;
  return 0.02;
 }

 public double y(double Assets,double MatchOrBenefit,double HK,double Pension,double UI,int e,int j)
 {
  double Splitting=Math.min(2,FamSize[j]-Children[j]);
  double PL=1.0/24;
  
  double yL=((e==1) && (j<Lr))?Brutto(Scale*MatchOrBenefit*(q0+HK)):0,yK=Math.max(0,(r+0*0.0284)*Assets/(1+r)),yR=0;
  if (e==2) yL=Brutto(MatchOrBenefit); //zur Berechnung des NettoLohns
  
  // SozVers.
  yL-=Math.min(yL,6150*12*PL)*0.0745+Math.min(yL,8200*12*PL)*0.134;

  //Rente
  if (j>=Lr) 
  {
   if (Pension<0.75) Pension=Math.min(1.5*Pension,0.75); //Rente nach Mindesteinkommen, 262, SGB 6
   yR=Pension*45*40.51*12*PL;
  }

  //Freibetraege
  double yS=0;
  yS=Math.max(0,yK-6100*Splitting*PL)+Math.max(yR*ErtragsAnteil(j)-200,0);
  if (Age(j)>=64) yS-=Math.max(0,Math.min(3720*Splitting*PL,0.4*(yR+yK)));
  yS+=Math.max(0,yL-2000*PL);
  yS=Math.max(0,yS-3456*Children[j]*Splitting);

  //Steuertarif
  yS/=PL*Splitting;
  double tax=0;
  if ((yS>=12096) && (yS<55728)) tax=86.63E-8*yS*yS+0.23813603*yS-2993.256041;
  else if ((yS>=55728) && (yS<120042)) tax=151.91E-8*yS*yS+0.165451253*yS-974.8733263;
  else if (yS>=120042) tax=0.53*yS-22842;
  double y=yL+yR-tax*1.055*Splitting*PL;
 
  if (e==2) return y;
  
  //AL-Vers:
  if ((e==0) && (j<Lr))
  {
   if (UI>24) y+=y(0,MatchOrBenefit,0,0,0,2,j)*(0.6+0.07*Math.min(1,Children[j]));
   // ALGII, falls Assets<Schonvermgen;
   else if ((UI>=0) && (Assets*2<=Math.max(4100,Math.min(13000,Math.floor(Age(j))*200))*Splitting+4100*Children[j]+13000+750*FamSize[j]))
   {
    double ALG1=y(0,MatchOrBenefit,0,0,0,2,j)*(0.6+0.07*Math.min(1,Children[j]));
    double ALG2=Math.max(0,534*12*PL*(1+0.8*(FamSize[j]-Children[j]-1)+0.65*Children[j])-yK);
    double Zuschlag=Math.max(0,ALG1-ALG2);
    if (UI>12) Zuschlag*=0.67;else if (UI>0) Zuschlag*=0.33;else Zuschlag=0;
    Zuschlag=Math.min(Zuschlag,160*Splitting+60*Children[j]);
    y+=ALG2+Zuschlag;
   }
  }

  //Kindergeld
  if (Children[j]>0) y+=220*12*PL*Math.min(2,Children[j]);
  if (Children[j]>2) y+=300*12*PL*Math.min(1,Children[j]-2);
  if (Children[j]>3) y+=350*12*PL*(Children[j]-3);
  
  
  //Soz.Hilfe
  if (Assets<=((Age(j)<60)?2500:4500)+(Splitting-1)*1200+(FamSize[j]-1)*500) y=Math.max(y,534*12*PL*(1+0.8*(FamSize[j]-Children[j]-1)+0.65*Children[j]));

  return y;
 }

 public int MakePeriods(Period[] P,Grid MG)
 {
  Grid ZG=new ZeroGrid();
  Grid AG=new CombinedGrid(new ELogGrid(43,0,-233352,12),new ELogGrid(78,0,2100168,100)); //-4*APW...36*APW
  Grid HG1=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(4,1,1.54,0)); //25 Perioden lang
  Grid HG2=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(6,1,2.05,0)); // die restlichen 20
  Grid BG=new EvenGrid(10,0,4878.5);
  double[] P1={0,1,1.89};Grid PG1=new PLGrid(P1);
  double[] P2={0,0.75,1,1.3,1.89};Grid PG2=new PLGrid(P2);
  double[] P3={0,0.5,0.75,1,1.3,1.6,1.89};Grid PG3=new PLGrid(P3);
  double[] Ue1={0,3.5,7.5,11.5,17.5,23.5};Grid UeG1=new PLGrid(Ue1); 
  double[] Ue2={0,3.5,7.5,11.5,17.5,23.5,35.5};Grid UeG2=new PLGrid(Ue2); 
  double[] Ue3={0,3.5,7.5,11.5,17.5,23.5,35.5,43.5};Grid UeG3=new PLGrid(Ue3); 
  double[] Ue4={0,3.5,7.5,11.5,17.5,23.5,35.5,51.5};Grid UeG4=new PLGrid(Ue4); 
  double[] Ue5={0,3.5,7.5,11.5,17.5,23.5,35.5,51.5,63.5};Grid UeG5=new PLGrid(Ue5);
  double[] Uu1={-1,0,3,6,9,12,15,18,21,24,27,30,33,36};Grid UuG1=new PLGrid(Uu1);
  double[] Uu2={0,3,6,9,12,15,18,21,24,27,30,33,36,39,42};Grid UuG2=new PLGrid(Uu2);
  double[] Uu3={0,3,6,9,12,15,18,21,24,27,30,33,36,39,42,46};Grid UuG3=new PLGrid(Uu3);
  double[] Uu4={0,3,6,9,12,15,18,21,24,27,30,33,36,39,42,46,50};Grid UuG4=new PLGrid(Uu4);
  double[] Uu5={0,3,6,9,12,15,18,21,24,27,30,33,36,39,42,46,50,56};Grid UuG5=new PLGrid(Uu5);

  Grid HG=HG1,PG=PG1,UeG=UeG1,UuG=UuG1;

  P[0]=P[L+1]=new Period(0,0,ZG,ZG,ZG,ZG,ZG,ZG,ZG);
  int cnt=0;
  int Basis=24;
  for (int i=1;i<Lr;i++) 
  {
   if (i==24*25+1) HG=HG2;
   if (i==24*15+1) PG=PG2;
   if (i==24*30+1) PG=PG3;
   if (i==24*22+1) {UeG=UeG2;UuG=UuG2;}
   if (i==24*24+1) {UeG=UeG3;UuG=UuG3;}
   if (i==24*29+1) {UeG=UeG4;UuG=UuG4;}
   if (i==24*34+1) {UeG=UeG5;UuG=UuG5;}
   if (i==Basis)
   { 
    P[i]=new Period(1,0,AG,MG,HG,BG,PG,UeG,UuG);
    if (i<1056) Basis+=24;
    else if (i<1068) Basis+=4;
    else if (i<1076) Basis+=2;
    else Basis++;
   }
   else
    P[i]=new Period(1,Basis,AG,MG,HG,BG,PG,UeG,UuG);
   cnt+=P[i].Size;
  }
  for (int i=Lr;i<=L;i++)
  {
   P[i]=new Period(2,0,AG,ZG,ZG,ZG,PG3,ZG,ZG);
   cnt+=P[i].Size;
  } 
  return cnt;
 }

 double phi,FamSize[],Children[];
 double fs[]=
  {1.469477,1.460009,1.605168,1.638724,1.772256,1.875615,1.810568,1.966422,2.000291,2.07277,2.143389,2.232974,2.387896,2.608009,2.808831,2.89637,2.89637,3.011235,3.11275,3.113931,3.15002,3.021219,3.007174,2.941447,2.999769,2.918441,2.834677,2.828536,2.732589,2.718965,2.80877,2.66286,2.625944,2.538983,2.434001,2.412622,2.385218,2.3481,2.288275,2.18332,2.10425,2.057797,1.990315,1.963784,1.867851,1.791152,1.711168,1.692563,1.619055,1.619637,1.559044,1.544855,1.510234,1.497789,1.455722,1.397622,1.386305,1.380239,1.398441,1.332914,1.294709,1.269522,1.250426,1.193537,1.244331,1.21128,1.207547,1.197409,1.27427,1.272242,1.276638};
 double cn[]=
  {0.1361777,0.1650809,0.2066014,0.2190127,0.2930346,0.3248403,0.3035464,0.4360645,0.4773302,0.516277,0.5954615,0.6829938,0.8019977,0.9494416,1.084804,1.175856,1.199202,1.219951,1.302891,1.272042,1.277275,1.161281,1.070141,0.9528889,0.9065176,0.7995372,0.6956645,0.6073926,0.5349463,0.4847855,0.4695864,0.3561606,0.3150203,0.2539642,0.192064,0.1624788,0.1250561,0.1104801,0.0895124,0.0658358,0.069252,0.0757035,0.0298257,0.0347224,0.0276274,0.0237833,0.0102839,0.0076741,0.0093706,0.0124521,0.0086449,0.0197351,0.0139889,0.0137782,0.0110017,0.0087281,0.0117513,0.0051705,0.0384093,0.0043711,0.0043595,0.0031116,0.0091908,0.0068407,0.0044978,0.0116574,0,0,0,0,0}; 
  
}

class Deutschland
extends Institutions
// Deutschland 1997
// UI: Dauer in Monaten
{
 public Deutschland(int L,int Lr,double q0,double R,double Scale,double phi)
 {
  super(L,Lr,q0,R,Scale);

  this.phi=phi;
  FamSize=new double[L+1];
  Children=new double[L+1];
  for (int i=0;i<L/24;i++) for (int j=1;j<=24;j++)
  {
   FamSize[i*24+j]=(fs[i]*(25-j)+fs[i+1]*(j-1))/24;
   Children[i*24+j]=(cn[i]*(25-j)+cn[i+1]*(j-1))/24;
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
  }
 }

 public void reset(double q0,double R,double Scale,double phi)
 {
  super.reset(q0,R,Scale);
  
  this.phi=phi;
  for (int i=0;i<60;i++) for (int j=0;j<24;j++)
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
 }

 private double Age(int j)
 {
  return 20+((double)j-1)/24;
 }

 public double Brutto(double y)
 {
  if (y>9753.25/2) return y-1556.975/2;
  if (y>7432.275/2) return (y-458.175/2)/1.134;
  return y/1.2085;
 }

 public double nextBenefit(double Match,double HK,double UI,int j)
 {
  // max. 8200 DM/Monat als Benefit-Basis
  // Benefit speichert EK bis Obergrenze
  return Math.min(Scale*Match*(q0+HK),(8200+1556.975)/2);
 }
 
 public double nextPension(double Match,double HK,double Pension,double UI,int e,int j)
 {
  // 0<=Pension<=1.88718117
  if (j>=Lr) return Pension;
  double yL=0;
  if (e==1) yL=Brutto(Scale*Match*(q0+HK));
  else if (UI>=0) yL=Brutto(Match)*0.8;
  return (Pension*(j-1)+Math.min(yL /*"Brutto-EK"*/,8200/2)/(52143/24))/j;
 }
 
 public double nextUI(double Match,double HK,double UI,int e,int j,int en)
 {
// UI (e==1): Anzahl der vor t bereits gearbeiteten Perioden
// Achtung: guenstigere die Altersgrenzen bis April 97; Ab Mai 97 alles +3 Jahre
  if (e==1)
  {
   UI+=0.5;
   if ((Age(j)<44-3) && (UI>24)) UI=24;
   else if ((Age(j)<46.34-3) && (UI>36)) UI=36;
   else if ((Age(j)<51.34-3) && (UI>44)) UI=44;
   else if ((Age(j)<56-3) && (UI>52)) UI=52;
   else if (UI>64) UI=64;
   if (en==1) return UI;
   if (UI<12) return 0;
   UI=Math.floor(UI/2);
   if (Age(j)<45-3) UI=Math.min(UI,12);
   else if (Age(j)<47-3) UI=Math.min(UI,18);
   else if (Age(j)<52-3) UI=Math.min(UI,22);
   else if (Age(j)<57-3) UI=Math.min(UI,26);
   return UI;
  }
  if (en==1) return 0;
  if (UI>0) return UI-0.5;
  return UI;
 }

 private double ErtragsAnteil(int j)
 {
  double Age=(j-1)/24+20;
  if (Age<68) return (92-Age)/100;
  if (Age<81) return (91-Age)/100;
  if (Age<85) return (92-Age)/100;
  if (Age<87) return (93-Age)/100;
  if (Age<90) return (94-Age)/100;
  if (Age<92) return 0.05;
  if (Age<94) return 0.04;
  if (Age<97) return 0.03;
  return 0.02;
 }

 public double y(double Assets,double MatchOrBenefit,double HK,double Pension,double UI,int e,int j)
 {
  double Splitting=Math.min(2,FamSize[j]-Children[j]);
  double PL=1.0/24;
  
  double yL=((e==1) && (j<Lr))?Brutto(Scale*MatchOrBenefit*(q0+HK)):0,yK=Math.max(0,(r+0*0.0284)*Assets/(1+r)),yR=0;
  if (e==2) yL=Brutto(MatchOrBenefit); //zur Berechnung des NettoLohns
  
  // SozVers.
  yL-=Math.min(yL,6150*12*PL)*0.0745+Math.min(yL,8200*12*PL)*0.134;

  //Rente
  if (j>=Lr) 
  {
   if (Pension<0.75) Pension=Math.min(1.5*Pension,0.75); //Rente nach Mindesteinkommen, 262, SGB 6
   yR=Pension*45*40.51*12*PL;
  }

  //Freibetraege
  double yS=0;
  yS=Math.max(0,yK-6100*Splitting*PL)+Math.max(yR*ErtragsAnteil(j)-200,0);
  if (Age(j)>=64) yS-=Math.max(0,Math.min(3720*Splitting*PL,0.4*(yR+yK)));
  yS+=Math.max(0,yL-2000*PL);
  yS=Math.max(0,yS-3456*Children[j]*Splitting);

  //Steuertarif
  yS/=PL*Splitting;
  double tax=0;
  if ((yS>=12096) && (yS<55728)) tax=86.63E-8*yS*yS+0.23813603*yS-2993.256041;
  else if ((yS>=55728) && (yS<120042)) tax=151.91E-8*yS*yS+0.165451253*yS-974.8733263;
  else if (yS>=120042) tax=0.53*yS-22842;
  double y=yL+yR-tax*1.055*Splitting*PL;
 
  if (e==2) return y;
  
  //AL-Vers:
  if ((e==0) && (j<Lr))
  {
   if (UI>0) y+=y(0,MatchOrBenefit,0,0,0,2,j)*(0.6+0.07*Math.min(1,Children[j]));
   //AL-Hilfe nur falls Assets<Schonvermoegen; Abzug von Kap-EK
   else if ((UI==0) && (Assets<=Math.floor(Age(j))*Splitting*1000)) y=Math.max(y(0,MatchOrBenefit,0,0,0,2,j)*(0.53+0.04*Math.min(1,Children[j])),y);
  }

  //Kindergeld
  if (Children[j]>0) y+=220*12*PL*Math.min(2,Children[j]);
  if (Children[j]>2) y+=300*12*PL*Math.min(1,Children[j]-2);
  if (Children[j]>3) y+=350*12*PL*(Children[j]-3);
  
  
  //Soz.Hilfe
  if (Assets<=((Age(j)<60)?2500:4500)+(Splitting-1)*1200+(FamSize[j]-1)*500) y=Math.max(y,534*12*PL*(1+0.8*(FamSize[j]-Children[j]-1)+0.65*Children[j]));

  return y;
 }

 public int MakePeriods(Period[] P,Grid MG)
 {
  Grid ZG=new ZeroGrid();
  Grid AG=new CombinedGrid(new ELogGrid(43,0,-233352,12),new ELogGrid(78,0,2100168,100)); //-4*APW...36*APW
  Grid HG1=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(4,1,1.54,0)); //25 Perioden lang
  Grid HG2=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(6,1,2.05,0)); // die restlichen 20
  Grid BG=new EvenGrid(10,0,4878.5);
  double[] P1={0,1,1.89};Grid PG1=new PLGrid(P1);
  double[] P2={0,0.75,1,1.3,1.89};Grid PG2=new PLGrid(P2);
  double[] P3={0,0.5,0.75,1,1.3,1.6,1.89};Grid PG3=new PLGrid(P3);
  double[] Ue1={0,3.5,7.5,11.5,17.5,23.5};Grid UeG1=new PLGrid(Ue1); 
  double[] Ue2={0,3.5,7.5,11.5,17.5,23.5,35.5};Grid UeG2=new PLGrid(Ue2); 
  double[] Ue3={0,3.5,7.5,11.5,17.5,23.5,35.5,43.5};Grid UeG3=new PLGrid(Ue3); 
  double[] Ue4={0,3.5,7.5,11.5,17.5,23.5,35.5,51.5};Grid UeG4=new PLGrid(Ue4); 
  double[] Ue5={0,3.5,7.5,11.5,17.5,23.5,35.5,51.5,63.5};Grid UeG5=new PLGrid(Ue5);
  double[] Ue6={0,3.5,7.5,11.5,17.5,23.5};Grid UeG6=new PLGrid(Ue6); // ab hier fuer's letzte Jahr
  double[] Ue7={0,3.5,7.5,11.5};Grid UeG7=new PLGrid(Ue7);
  double[] Ue8={0,3.5};Grid UeG8=new PLGrid(Ue8);
  double[] Uu1={-1,0,2,4,6,9,12};Grid UuG1=new PLGrid(Uu1);
  double[] Uu2={0,2,4,6,9,12,18};Grid UuG2=new PLGrid(Uu2);
  double[] Uu3={0,2,4,6,9,12,18,22};Grid UuG3=new PLGrid(Uu3);
  double[] Uu4={0,2,4,6,9,12,18,26};Grid UuG4=new PLGrid(Uu4);
  double[] Uu5={0,2,4,6,9,12,18,26,32};Grid UuG5=new PLGrid(Uu5);
  double[] Uu6={0,2,4,6,9,12};Grid UuG6=new PLGrid(Uu6);
  double[] Uu7={0,2,4,6};Grid UuG7=new PLGrid(Uu7);
  double[] Uu8={0,2};Grid UuG8=new PLGrid(Uu8);

  Grid HG=HG1,PG=PG1,UeG=UeG1,UuG=UuG1;

  P[0]=P[L+1]=new Period(0,0,ZG,ZG,ZG,ZG,ZG,ZG,ZG);
  int cnt=0;
  int Basis=24;
  for (int i=1;i<Lr;i++) 
  {
   if (i==24*25+1) HG=HG2;
   if (i==24*15+1) PG=PG2;
   if (i==24*30+1) PG=PG3;
   if (i==24*22+1) {UeG=UeG2;UuG=UuG2;}
   if (i==24*24+1) {UeG=UeG3;UuG=UuG3;}
   if (i==24*29+1) {UeG=UeG4;UuG=UuG4;}
   if (i==24*34+1) {UeG=UeG5;UuG=UuG5;}
   if (i==24*42+21) {UeG=UeG4;UuG=UuG4;}
   if (i==24*43+5) {UeG=UeG3;UuG=UuG3;}
   if (i==24*43+13) {UeG=UeG2;UuG=UuG2;}
   if (i==24*44+1) {UeG=UeG6;UuG=UuG6;}
   if (i==24*44+13) {UeG=UeG7;UuG=UuG7;}
   if (i==24*44+21) {UeG=UeG8;UuG=UuG8;}
   if (i==Basis)
   { 
    P[i]=new Period(1,0,AG,MG,HG,BG,PG,UeG,UuG);
    if (i<1056) Basis+=24;
    else if (i<1068) Basis+=4;
    else if (i<1076) Basis+=2;
    else Basis++;
   }
   else
    P[i]=new Period(1,Basis,AG,MG,HG,BG,PG,UeG,UuG);
   cnt+=P[i].Size;
  }
  for (int i=Lr;i<=L;i++)
  {
   P[i]=new Period(2,0,AG,ZG,ZG,ZG,PG3,ZG,ZG);
   cnt+=P[i].Size;
  } 
  return cnt;
 }

 double phi,FamSize[],Children[];
 double fs[]=
  {1.469477,1.460009,1.605168,1.638724,1.772256,1.875615,1.810568,1.966422,2.000291,2.07277,2.143389,2.232974,2.387896,2.608009,2.808831,2.89637,2.89637,3.011235,3.11275,3.113931,3.15002,3.021219,3.007174,2.941447,2.999769,2.918441,2.834677,2.828536,2.732589,2.718965,2.80877,2.66286,2.625944,2.538983,2.434001,2.412622,2.385218,2.3481,2.288275,2.18332,2.10425,2.057797,1.990315,1.963784,1.867851,1.791152,1.711168,1.692563,1.619055,1.619637,1.559044,1.544855,1.510234,1.497789,1.455722,1.397622,1.386305,1.380239,1.398441,1.332914,1.294709,1.269522,1.250426,1.193537,1.244331,1.21128,1.207547,1.197409,1.27427,1.272242,1.276638};
 double cn[]=
  {0.1361777,0.1650809,0.2066014,0.2190127,0.2930346,0.3248403,0.3035464,0.4360645,0.4773302,0.516277,0.5954615,0.6829938,0.8019977,0.9494416,1.084804,1.175856,1.199202,1.219951,1.302891,1.272042,1.277275,1.161281,1.070141,0.9528889,0.9065176,0.7995372,0.6956645,0.6073926,0.5349463,0.4847855,0.4695864,0.3561606,0.3150203,0.2539642,0.192064,0.1624788,0.1250561,0.1104801,0.0895124,0.0658358,0.069252,0.0757035,0.0298257,0.0347224,0.0276274,0.0237833,0.0102839,0.0076741,0.0093706,0.0124521,0.0086449,0.0197351,0.0139889,0.0137782,0.0110017,0.0087281,0.0117513,0.0051705,0.0384093,0.0043711,0.0043595,0.0031116,0.0091908,0.0068407,0.0044978,0.0116574,0,0,0,0,0}; 
  
}

class UK
extends Institutions
// UK 1997
{
 public UK(int L,int Lr,double q0,double R,double Scale,double phi)
 {
  super(L,Lr,q0,R,Scale);

  this.phi=phi;
  FamSize=new double[L+1];
  Children=new double[L+1];
  for (int i=0;i<L/24;i++) for (int j=1;j<=24;j++)
  {
   FamSize[i*24+j]=(fs[i]*(25-j)+fs[i+1]*(j-1))/24;
   Children[i*24+j]=(cn[i]*(25-j)+cn[i+1]*(j-1))/24;
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
  }
 }

 public void reset(double q0,double R,double Scale,double phi)
 {
  super.reset(q0,R,Scale);
  
  this.phi=phi;
  for (int i=0;i<60;i++) for (int j=0;j<24;j++)
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
 }

 public double nextBenefit(double Match,double HK,double UI,int j)
 {
  return 0;
 }
 
 public double nextPension(double Match,double HK,double Pension,double UI,int e,int j)
 {
  return 0;
 }
 
 public double Brutto(double y)
 {
  if (y<62*1.03*2.1667) return y;
  if (y<110*1.05*2.1667) return Math.min(y/1.03,110*2.1667);
  if (y<155*1.07*2.1667) return Math.min(y/1.05,155*2.1667);
  if (y<210*1.1*2.1667) return Math.min(y/1.07,210*2.1667);
  return y/1.1;
 }
 
 public double nextUI(double Match,double HK,double UI,int e,int j,int en)
 {
  //Qualifikation: mind. relevantes EK von 50x MindestEK
  if (e==1)
  {
   if (Brutto(Scale*Match*(q0+HK))>=62*2.1667) 
   {
    UI+=Brutto(Scale*Match*(q0+HK))/62;
    if (UI>50) UI=50;
   }
   if (en==0)
   {
    if (UI==50) return 6;
    return 0;
   }
   return UI;
  }
  if (en==0) return Math.max(UI-0.5,0);
  return 0;
 }

 public double firstUI() {return 0;}

 public double y(double Assets,double MatchOrBenefit,double HK,double Pension,double UI,int e,int j)
 {
  double Adults=Math.min(1,FamSize[j]-Children[j]-1);
 	
  double yL=((e==1) && (j<Lr))?Brutto(Scale*MatchOrBenefit*(q0+HK)):0,yK=Math.max(0,Assets*(r+0*0.0281)/(1+r)),yS=0;
  double SStax=0,tax=0;

  //SocSec taxes
  if (yL>=62*2.1667) SStax=0.02*62*2.1667+Math.min((465-62)*2.1667,yL-62*2.1667)*0.1;

  //Taxable SocSec Income
  if (j<Lr)
  {
  // - contribution based jobseeker's allowance
   if ((e==0) && (UI>0)) 
   {
    if (j>24*5) yS=2.1667*(49.15+28*Adults+20.97*Children[j]+10.8*Math.min(Children[j],1));
    else yS=2.1667*(38.9+38.25*Adults+20.97*Children[j]+10.8*Math.min(Children[j],1));
   }
  // - income based jobseeker's allowance
   else if ((e==0) && (UI==0) && ((Assets<8000) || ((j>24*40) && (Assets<12000)))) 
   {
    if (j>24*40) yS=2.1667*Math.max(0,49.15+28*Adults+20.97*Children[j]+10.8*Math.min(Children[j],1)-Math.max(0,Assets-7250-1250*Adults)/250);
    else if (j>24*5) yS=2.1667*Math.max(0,49.15+28*Adults+20.97*Children[j]+10.8*Math.min(Children[j],1)-Math.max(0,Assets-4250-1250*Adults)/250);
    else yS=2.1667*Math.max(0,38.9+38.25*Adults+20.97*Children[j]+10.8*Math.min(Children[j],1)-Math.max(0,Assets-4250-1250*Adults)/250);
   }
  }
  
  //Pension
  if (j>45*24) yS+=(64.7+Adults*38.7)*2.1667;
  
  //Taxes
  double TI=yL+yK+yS,TI2=yK;
  TI-=4045/24;
  if (j>45*24) TI-=1175/24.0;
  if (j>55*24) TI-=180/24.0;
  if (TI>0)
  {
   tax=0.2*Math.min(TI,4100/24);
   TI-=4100/24.0;
  }
  if (TI>0)
  {
   tax+=0.23*Math.max(Math.min(TI-TI2,22000/24.0),0);
   tax+=0.2*(Math.min(TI,22000/24.0)-Math.max(Math.min(TI-TI2,22000/24.0),0));
   TI-=22000/24.0;
  }
  if (TI>0) tax+=0.4*TI;

  // further credits:
  tax-=1830*Adults/24.0*0.15;
  if (j>45*24) tax-=1355*Adults/24.0*0.15;
  if (j>55*24) tax-=40*Adults/24.0*0.15;
  if (tax<0) tax=0;
  
  //child benefits
  yS+=2.1667*(Math.min(1,Children[j])*11.05+Math.max(0,Children[j]-1)*9);

  //Council Tax & benefit
  double ctax=2.1667*12.95;
  if ((Assets<8000) || ((j>24*40) && (Assets<12000)))
  {
   double need;
   if (j>24*5) need=2.1667*(49.15+28*Adults+20.97*Children[j]+10.8*Math.min(Children[j],1));
   else need=2.1667*(38.9+38.25*Adults+20.97*Children[j]+10.8*Math.min(Children[j],1));
   ctax=Math.max(0,Math.min(2.1667*12.95,(Math.max(0,yL-2.1667*5*(1+Adults))+yS-SStax-tax+2.1667*Math.max(0,(Assets-((j>24*40)?6000:3000))/250)-need)/5));
  }

  //family credit:
  if ((j<Lr) && (e==1) && ((Assets<8000) || ((j>24*40) && (Assets<12000))))
   tax-=Math.max(0,(58.2+16.66*Children[j])*2.1667-Math.max(0,yL+yS-SStax-tax-77.15*2.1667+2.1667*Math.max(0,(Assets-((j>24*40)?6000:3000))/250))*0.7);

  return yL+yS-SStax-tax-ctax;
 }

 public int MakePeriods(Period[] P,Grid MG)
 {
  Grid ZG=new ZeroGrid();
  Grid AG=new CombinedGrid(new ELogGrid(43,0,-67040,12),new ELogGrid(78,0,603360,100)); //-4*APW...36*APW
  Grid HG1=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(5,1,1.84,0));
  Grid HG2=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(8,1,2.91,0));
  Grid UeG=new EvenGrid(6,0,50);
  Grid UuG=new EvenGrid(4,0,6);

  Grid HG=HG1;

  P[0]=P[L+1]=new Period(0,0,ZG,ZG,ZG,ZG,ZG,ZG,ZG);
  int cnt=0;
  int Basis=24;
  for (int i=1;i<Lr;i++) 
  {
   if (i==24*25+1) HG=HG2;
   if (i==Basis) 
   {	
    P[i]=new Period(1,0,AG,MG,HG,ZG,ZG,UeG,UuG);
    if (i<1056) Basis+=24;
    else if (i<1068) Basis+=4;
    else if (i<1076) Basis+=2;
    else Basis++;
   }
   else 
    P[i]=new Period(1,Basis,AG,MG,HG,ZG,ZG,UeG,UuG);
   cnt+=P[i].Size;
  }
  for (int i=Lr;i<=L;i++)
  {
   P[i]=new Period(2,0,AG,ZG,ZG,ZG,ZG,ZG,ZG);
   cnt+=P[i].Size;
  }
  return cnt;
 }

 double phi,FamSize[],Children[];
 double fs[]=
  {1.985496,1.975238,2.120441,2.005169,2.109027,2.227369,2.343771,2.344058,2.45445,2.669045,2.688641,2.785546,2.912563,3.085353,3.103468,3.176358,3.239133,3.27449,3.38131,3.367453,3.358546,3.412645,3.298411,3.271034,3.268965,3.203859,3.14573,3.129751,3.004215,2.971515,2.917091,2.732346,2.653705,2.621326,2.468899,2.36725,2.33457,2.35252,2.24384,2.148422,2.065772,2.063153,2.016786,1.894457,1.863157,1.820676,1.769295,1.738888,1.70476,1.653652,1.603439,1.616091,1.546728,1.556033,1.534528,1.511847,1.522371,1.450887,1.443261,1.418112,1.385202,1.330756,1.310689,1.306519,1.30471,1.223761,1.18276,1.22041,1.223361,1.16888,1.14118};
 double cn[]=
  {0.3439325,0.3886463,0.4653672,0.4242553,0.4939128,0.5833062,0.6568484,0.6574574,0.7530054,0.953389,0.9891925,1.0857,1.191507,1.361157,1.358593,1.428835,1.455354,1.464098,1.550461,1.533364,1.49759,1.502965,1.339377,1.233265,1.177506,1.038243,0.9161053,0.7892258,0.6754217,0.6034966,0.5070589,0.3786812,0.2888195,0.2556244,0.2023032,0.1499142,0.1389641,0.1365797,0.0903249,0.0702733,0.0715481,0.0676099,0.0606895,0.0365216,0.0419179,0.037153,0.0166782,0.0219531,0.0274009,0.0163869,0.0119209,0.0160069,0.0147432,0.0054828,0.0088247,0.0026975,0.0176713,0.0114047,0.0065426,0.0122696,0.0049283,0.0053378,0,0,0,0.0010519,0.009202,0,0,0,0}; 
  
}

class US
extends Institutions
// USA 1997
{
 public US(int L,int Lr,double q0,double R,double Scale,double phi)
 {
  super(L,Lr,q0,R,Scale);

  this.phi=phi;
  FamSize=new double[L+1];
  Children=new double[L+1];
  for (int i=0;i<L/24;i++) for (int j=1;j<=24;j++)
  {
   FamSize[i*24+j]=(fs[i]*(25-j)+fs[i+1]*(j-1))/24;
   Children[i*24+j]=(cn[i]*(25-j)+cn[i+1]*(j-1))/24;
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
  }
 }

 public void reset(double q0,double R,double Scale,double phi)
 {
  super.reset(q0,R,Scale);
  
  this.phi=phi;
  for (int i=0;i<60;i++) for (int j=0;j<24;j++)
   CM[i*24+j]=Math.exp(Math.log(FamSize[i*24+j])*phi);
 }

 public double nextBenefit(double Match,double HK,double UI,int j)
 {
  //actual Benefit
  return Math.max(4524/24.0,Math.min(15600/24.0,0.5*Brutto(Scale*Match*(q0+HK))));
 }
 
 public double nextPension(double Match,double HK,double Pension,double UI,int e,int j)
 {
  if (j>45*24) return Pension;
  // AIME (=2x Period pension of Single)
  double Pj=2*((e==1)?Math.min(Brutto(Scale*Match*(q0+HK)),65400/24.0):0);
  if (j<=35*24) return (Pension*(j-1)+Pj)/j;
  if (Pj>Pension) return (Pension*(35*24)+Pj)/(35*24+1);
  return Pension;
 }
 
 public double Brutto(double y)
 {
  if (y>70877.1/24.0) return (y-4528.8/24.0)/1.0145;
  if (y>10700.74/24.0) return (y-474/24.0)/1.0765;
  if (y>7899.5/24.0) return (y-56/24.0)/1.1205;
  return y/1.1285;
 }
 
 public double nextUI(double Match,double HK,double UI,int e,int j,int en)
 {
  if (e==1)
  {
   if ((Brutto(Scale*Match*(q0+HK))>=3084/24.0) && (UI<6)) UI+=0.5;
   if (en==0)
   {
    if (UI==6) return 6;
    return 0;
   }
   return UI;
  }
  if (en==0) return Math.max(UI-0.5,0);
  return 0;
 }

 public double firstUI() {return 0;}

 public double y(double Assets,double MatchOrBenefit,double HK,double Pension,double UI,int e,int j)
 {
  //no local tax
  double Spouse=Math.min(FamSize[j]-Children[j]-1,1);	
 
  double yL=((e==1) && (j<Lr))?Brutto(Scale*MatchOrBenefit*(q0+HK)):0,yK=Math.max(0,(r+0*0.031)*Assets/(1+r)),yU=0,yS=0,SS=0;
  double AGI=0,FItax=0,Stax=0,taxableSS=0,EITC=0;
  double FItaxable=0,Staxable=0;
  double SSI=0,food=0;

  //Payroll tax
  SS=0.062*Math.min(yL,65400/24.0)+0.0145*yL;
  
  if (j>=Lr)
  {
   //soc.sec.
   yS=0.9*Math.min(455,Pension);
   yS+=0.32*Math.max(0,Math.min(2286,Pension-455));
   yS+=0.15*Math.max(0,Pension-2741);
   yS*=0.5*(1+0.5*Spouse);

   //SSI
   if (Assets<(2000+1000*Spouse))
    SSI=Math.max(0,(484+Spouse*242)/2.0-Math.max(0,yS+yK-10));
  }
  //unemp. ins.
  else if ((e==0) && (UI>0)) yU=MatchOrBenefit;
  
  //Fed Inc. Tax
  //and Michigan State tax
  AGI=yL+yK+yU;
  Staxable=AGI-FamSize[j]*2500/24.0;
  if ((yS>0) && (yS/2+AGI>(25000+Spouse*7000)/24.0))
   taxableSS=Math.min(0.85*yS,Math.max(yS/2+AGI-(34000+10000*Spouse)/24.0,0)*0.85+Math.min(yS/2,Math.min((9000+3000)/24.0,0.5*Math.max(yS/2+AGI-(25000+Spouse*7000)/24.0,0)) ));
  AGI+=taxableSS;
  FItaxable=AGI-(4150+2750*Spouse)/24.0;
  if (j>24*45)
  {
   FItaxable-=(1000+600*Spouse)/24.0;
   Staxable-=900*(1+Spouse)/24.0;
  }
  FItaxable-=2650*FamSize[j]/24.0*Math.max(0,1-Math.max(0,AGI*24-121200-Spouse*60600)/125000.0);
  if (FItaxable>0)
  {
   double TT=(24650+Spouse*16550)/24.0;
   FItax=0.15*Math.min(TT,FItaxable);FItaxable-=TT;
   if (FItaxable>0)
   {
    TT=(35100+Spouse*23300)/24.0;FItax+=0.28*Math.min(TT,FItaxable);FItaxable-=TT;
    if (FItaxable>0)
    {
     TT=(64900-Spouse*12750)/24.0;FItax+=0.31*Math.min(TT,FItaxable);FItaxable-=TT;
     if (FItaxable>0)
     {
      TT=(146400-Spouse*27100)/24.0;FItax+=0.36*Math.min(TT,FItaxable);FItaxable-=TT;
      if (FItaxable>0) FItax+=0.396*FItaxable;
     }
    } 
   }
  } 
  if (yU>0) if (AGI<=2*yU) Staxable-=900/24.0;
  Stax=0.044*Math.max(0,Staxable);
  
  //EITC
  if ((yK<2250/24.0) && (yL>0))
  {
   double EIC0,EIC1,EIC2;
   if ((j<=5*24) || (j>45*24)) EIC0=0;
   else EIC0=Math.min(332/24.0,0.0765*AGI)-Math.max(0,Math.min((AGI-5450/24.0)*0.0765,332/24.0));
   EIC1=Math.min(2210/24.0,0.34*AGI)-Math.max(0,Math.min((AGI-11950/24.0)*0.16,2210/24.0));
   EIC2=Math.min(3656/24.0,0.4*AGI)-Math.max(0,Math.min((AGI-11950/24.0)*0.2108,3656/24.0));
   if (Children[j]<=1) EITC=EIC0*(1-Children[j])+EIC1*Children[j];
   else if (Children[j]>1) EITC=EIC1*(1-Math.min(Children[j]-1,1))+EIC2*Math.min(Children[j]-1,1);
  }
    
  //Food stamps (exactly correct for up to 3 persons, ok if <4)
  if ((Assets<2000) || ((Assets<3000) && (j>40*24)))
  {
   double basic,counted;
   basic=yL+yK+yU+yS-FItax-Stax-SS;
   counted=Math.max(basic-0.2*yL-134/2.0-177/2.0*Children[j],0);
   if ((FamSize[j]<2) && (basic<(893+283*(FamSize[j]-1))/2.0) && (counted<(671+234*(FamSize[j]-1))/2.0)) food=Math.max(0,125/2.0-0.3*counted);
   else if ((FamSize[j]*0<3) && (basic<(1176+303*(FamSize[j]-2))/2.0) && (counted<(905+233*(FamSize[j]-2))/2.0)) food=Math.max(0,230/2.0-0.3*counted);
  }   

  return yL+yU+yS+food-SS-FItax-Stax+EITC+SSI;
 }

 public int MakePeriods(Period[] P,Grid MG)
 {
  Grid ZG=new ZeroGrid();
  Grid AG=new CombinedGrid(new ELogGrid(43,0,-114336,12),new ELogGrid(78,0,1029024,100)); //-4*APW...36*APW
  Grid HG1=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(5,1,1.85,0)); //23 Perioden lang
  Grid HG2=new CombinedGrid(new EvenGrid(7,0,1),new LogGrid(9,1,3.32,0)); // die restlichen 22
  Grid BG=new EvenGrid(6,188.5,650);
  Grid PG1=new EvenGrid(3,0,5450);
  Grid PG2=new EvenGrid(5,0,5450);
  Grid PG3=new EvenGrid(9,0,5450);
  Grid UG=new EvenGrid(4,0,6);

  Grid HG=HG1,PG=PG1;

  int cnt=0;
  int Basis=24;
  P[0]=P[L+1]=new Period(0,0,ZG,ZG,ZG,ZG,ZG,ZG,ZG);
  for (int i=1;i<Lr;i++) 
  {
   if (i==24*23+1) HG=HG2;
   if (i==24*15+1) PG=PG2;
   if (i==24*30+1) PG=PG3;
   if (i==Basis) 
   {	
    P[i]=new Period(1,0,AG,MG,HG,BG,PG,UG,UG);
    if (i<1056) Basis+=24;
    else if (i<1068) Basis+=4;
    else if (i<1076) Basis+=2;
    else Basis++;
   }
   else 
    P[i]=new Period(1,Basis,AG,MG,HG,BG,PG,UG,UG);
   cnt+=P[i].Size;
  }
  for (int i=Lr;i<=L;i++)
  {
   P[i]=new Period(2,0,AG,ZG,ZG,ZG,PG,ZG,ZG);
   cnt+=P[i].Size;
  }
  return cnt;
 }

 double phi,FamSize[],Children[];
 double fs[]=
  {1.875342,1.948713,1.82825,1.799823,1.904865,1.955105,2.080133,2.085318,2.238999,2.328083,2.44759,2.521041,2.726534,2.806953,2.989861,3.076663,3.07061,3.179005,3.109714,2.994171,3.024195,3.000719,2.941247,2.910616,2.960295,2.860008,2.869957,2.833571,2.711429,2.598353,2.550589,2.437344,2.383241,2.327384,2.236204,2.133012,2.144219,2.126836,2.12439,2.123333,2.116677,2.072742,1.995117,1.976793,1.940439,1.949167,1.878473,1.868272,1.901043,1.847554,1.794773,1.75487,1.776353,1.736906,1.762271,1.687131,1.630891,1.587733,1.569849,1.503675,1.465886,1.500762,1.475022,1.402472,1.39519,1.295989,1.294879,1.311057,1.352917,1.209096,1.415462};
 double cn[]=
  {0.470242,0.517566,0.447822,0.4330432,0.481953,0.5147713,0.595273,0.6158588,0.7225585,0.8305319,0.9349168,0.9685141,1.154981,1.221325,1.342863,1.393512,1.387165,1.461266,1.393254,1.271097,1.263912,1.226633,1.140716,1.044452,1.01211,0.9018111,0.8495852,0.7459612,0.6380427,0.5398549,0.4597526,0.3885322,0.3274153,0.2894415,0.2081096,0.1763081,0.1728856,0.1381417,0.1517108,0.1010706,0.1018906,0.1006871,0.0715019,0.0622521,0.0776425,0.070011,0.0816301,0.06214,0.0713894,0.0498848,0.0444782,0.0443683,0.0494928,0.0436891,0.0515947,0.0335454,0.0308722,0.0124626,0.0251226,0.0152833,0.0107336,0.0187258,0.0189497,0.0151641,0.0147921,0.0064061,0.0160397,0.0139362,0.0129067,0,0.0635865}; 
  
}

//-------------------------------------------------------------------
//		LifeCycle
//-------------------------------------------------------------------

class LCOptimiser
{
 public LCOptimiser(int AssetMax,int MatchMax,int BenefitMax,int HKMax,int PensionMax,int UIeMax,int UIuMax)
 {
  ValueE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax+1];
  ValueU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax+1];
  AssetE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax];
  AssetU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax];
  IValue=new double[PensionMax][HKMax][MatchMax][AssetMax];
  LValueE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax+1];
  LValueU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax+1];
  LAssetE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax];
  LAssetU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax];
  LIValue=new double[PensionMax][HKMax][MatchMax][AssetMax];

  EValue=new double[AssetMax];
  EValueO=new double[AssetMax];
  mPos=new double[AssetMax];
  EIndex=new int[AssetMax];

  IndexBeu=new int[2];IndexHee=new int[2];IndexHeu=new int[2];IndexHu=new int[2];IndexPe=new int[2];IndexPu=new int[2];IndexUee=new int[2];IndexUeu=new int[2];IndexUu=new int[2];
  WeightsBeu=new double[2];WeightsHee=new double[2];WeightsHeu=new double[2];WeightsHu=new double[2];WeightsPe=new double[2];WeightsPu=new double[2];WeightsUee=new double[2];WeightsUeu=new double[2];WeightsUu=new double[2];
 
  Size=8*2*AssetMax*HKMax*PensionMax*(MatchMax*(1+2*UIeMax)+2*BenefitMax*UIuMax);
 }

 public void res(int AssetMax,int MatchMax,int BenefitMax,int HKMax,int PensionMax,int UIeMax,int UIuMax)
 {
  ValueE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax+1];
  ValueU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax+1];
  AssetE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax];
  AssetU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax];
  IValue=new double[PensionMax][HKMax][MatchMax][AssetMax];
  LValueE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax+1];
  LValueU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax+1];
  LAssetE=new double[PensionMax][UIeMax][HKMax][MatchMax][AssetMax];
  LAssetU=new double[PensionMax][UIuMax][HKMax][BenefitMax][AssetMax];
  LIValue=new double[PensionMax][HKMax][MatchMax][AssetMax];

  EValue=new double[AssetMax];
  EValueO=new double[AssetMax];
  mPos=new double[AssetMax];
  EIndex=new int[AssetMax];

  IndexBeu=new int[2];IndexHee=new int[2];IndexHeu=new int[2];IndexHu=new int[2];IndexPe=new int[2];IndexPu=new int[2];IndexUee=new int[2];IndexUeu=new int[2];IndexUu=new int[2];
  WeightsBeu=new double[2];WeightsHee=new double[2];WeightsHeu=new double[2];WeightsHu=new double[2];WeightsPe=new double[2];WeightsPu=new double[2];WeightsUee=new double[2];WeightsUeu=new double[2];WeightsUu=new double[2];
 
  Size=8*2*AssetMax*HKMax*PensionMax*(MatchMax*(1+2*UIeMax)+2*BenefitMax*UIuMax);
 }

 public final void change()
 {
  double L1[][][][][],L2[][][][];
  L1=LValueE;LValueE=ValueE;ValueE=L1;
  L1=LAssetE;LAssetE=AssetE;AssetE=L1;
  L1=LValueU;LValueU=ValueU;ValueU=L1;
  L1=LAssetU;LAssetU=AssetU;AssetU=L1;
  L2=LIValue;LIValue=IValue;IValue=L2;
 }
 	
 public final double getEValue(int An)
 {
  if (Type==1)
  {
   double EValue1=0,EValue2=0;
   for (int k=0;k<CountUee;k++) for (int j=0;j<CountPe;j++) for (int i=0;i<CountHee;i++)
    EValue1+=LValueE[IndexPe[j]][IndexUee[k]][IndexHee[i]][M][An]*WeightsPe[j]*WeightsHee[i]*WeightsUee[k];
   for (int k=0;k<CountUeu;k++) for (int h=0;h<CountBeu;h++) for (int j=0;j<CountPe;j++) for (int i=0;i<CountHeu;i++)
    EValue2+=LValueU[IndexPe[j]][IndexUeu[k]][IndexHeu[i]][IndexBeu[h]][An]*WeightsPe[j]*WeightsHeu[i]*WeightsUeu[k]*WeightsBeu[h];
   return (1-jd)*EValue1+jd*EValue2;
  }
  if (Type==0)
  {
   double Alternative=0,ValueMin=0,ValueMax=0,V,IValue=0;
   int max=Mmax,min=0,n;
   for (int j=0;j<CountPu;j++) for (int i=0;i<CountHu;i++)
   {
    ValueMax+=LValueE[IndexPu[j]][U][IndexHu[i]][max][An]*WeightsPu[j]*WeightsHu[i];
    ValueMin+=LValueE[IndexPu[j]][U][IndexHu[i]][0][An]*WeightsPu[j]*WeightsHu[i];
    for (int k=0;k<CountUu;k++)
     Alternative+=LValueU[IndexPu[j]][IndexUu[k]][IndexHu[i]][M][An]*WeightsPu[j]*WeightsHu[i]*WeightsUu[k];
   }
   if (Alternative>=ValueMax) return Alternative;
   if (Alternative<=ValueMin) max=0;
   else while (max-min>1)
   {
    n=(max+min)>>1;
    V=0;
    for (int j=0;j<CountPu;j++) for (int i=0;i<CountHu;i++)
     V+=LValueE[IndexPu[j]][U][IndexHu[i]][n][An]*WeightsPu[j]*WeightsHu[i];
    if (V<Alternative) min=n;else max=n;
   }
   
   for (int j=0;j<CountPu;j++) for (int i=0;i<CountHu;i++)
    IValue+=LIValue[IndexPu[j]][IndexHu[i]][max][An]*WeightsPu[j]*WeightsHu[i];
   return Alternative*IWAgg[max]+IValue;
  }
  return LValueE[P][0][0][0][An];
}
 	
 public final double getMin()
 {
  if (Type==1)
  {
   double min1=0,min2=0;
   for (int k=0;k<CountUee;k++) for (int j=0;j<CountPe;j++) for (int i=0;i<CountHee;i++)
    min1+=LValueE[IndexPe[j]][IndexUee[k]][IndexHee[i]][M][Am]*WeightsPe[j]*WeightsHee[i]*WeightsUee[k];
   for (int k=0;k<CountUeu;k++) for (int h=0;h<CountBeu;h++) for (int j=0;j<CountPe;j++) for (int i=0;i<CountHeu;i++)
    min2+=LValueU[IndexPe[j]][IndexUeu[k]][IndexHeu[i]][IndexBeu[h]][Am]*WeightsPe[j]*WeightsHeu[i]*WeightsUeu[k]*WeightsBeu[h];
   return Math.max(min1,min2);
  }
  if (Type==0)
  {
   double min=0;
   for (int k=0;k<CountUu;k++) for (int j=0;j<CountPu;j++) for (int i=0;i<CountHu;i++)
     min+=LValueU[IndexPu[j]][IndexUu[k]][IndexHu[i]][M][Am]*WeightsPu[j]*WeightsHu[i]*WeightsUu[k];
   return min;
  }
  return LValueE[P][0][0][0][Am];
}
 	
 boolean StableGrid;
 
 int Type,P,M,U,Mmax,Am;
 
 double jd,IWAgg[];
 
 int    CountBeu,CountHee,CountHeu,CountHu,CountPe,CountPu,CountUee,CountUeu,CountUu;
 int    IndexBeu[],IndexHee[],IndexHeu[],IndexHu[],IndexPe[],IndexPu[],IndexUee[],IndexUeu[],IndexUu[];
 double WeightsBeu[],WeightsHee[],WeightsHeu[],WeightsHu[],WeightsPe[],WeightsPu[],WeightsUee[],WeightsUeu[],WeightsUu[];
 
 double ValueE[][][][][],ValueU[][][][][],AssetE[][][][][],AssetU[][][][][],IValue[][][][];
 double LValueE[][][][][],LValueU[][][][][],LAssetE[][][][][],LAssetU[][][][][],LIValue[][][][];
 
 double EValue[],mPos[],EValueO[],mPosO[];
 int    EIndex[],EAnz;
 
 int Size;
}

class LifeCycle
{
 public LifeCycle(int L,int Lr,ModelParameters Params,Institutions Inst)
 {
  this.L=L;this.Lr=Lr;
  q0=Params.P[0];de=Params.P[1];du=Params.P[2];df=Params.P[3];jd=Params.P[4];
  jc=Params.P[5];jv=Params.P[6];R=Params.P[7];beta=Params.P[8];gamma=Params.P[9];
  this.Inst=Inst;
  
  Periods=new Period[L+2];

  Grid MG=new LinWeightsGrid(10,10/(24*(1-Math.exp(-jc))+5),jc,jv,2,1.5);

  IW=new IntegrationWeights(MG,jc,jv);
  IWAgg=new double[MG.numPoints];
  IWAgg[MG.numPoints-1]=1-IW.Weights[MG.numPoints-1];
  for (int i=MG.numPoints-2;i>=0;i--) IWAgg[i]=IWAgg[i+1]-IW.Weights[i];
  
  System.out.println("Groesse: Perioden:   "+Inst.MakePeriods(Periods,MG)/(1024*1024)+" MByte");

  empU=new int[L+1];
  for (int j=1;j<=L;j++) empU[j]=Periods[j].UIEGrid.getIndex(Inst.empUI);
  CMgamma=new double[L+2];
  for (int i=1;i<=L;i++) CMgamma[i]=Math.pow(Inst.CM[i],gamma);

  int AssetMax=1,MatchMax=1,BenefitMax=1,HKMax=1,UIeMax=1,UIuMax=1,PensionMax=1;
  for (int j=1;j<=L;j++)
  {
   AssetMax=Math.max(Periods[j].AssetGrid.numPoints,AssetMax);
   MatchMax=Math.max(Periods[j].MatchGrid.numPoints,MatchMax);
   BenefitMax=Math.max(Periods[j].BenefitGrid.numPoints,BenefitMax);
   HKMax=Math.max(Periods[j].HKGrid.numPoints,HKMax);
   UIeMax=Math.max(Periods[j].UIEGrid.numPoints,UIeMax);
   UIuMax=Math.max(Periods[j].UIUGrid.numPoints,UIuMax);
   PensionMax=Math.max(Periods[j].PensionGrid.numPoints,PensionMax);
  }
  LCO=new LCOptimiser(AssetMax,MatchMax,BenefitMax,HKMax,PensionMax,UIeMax,UIuMax);
  LCO.IWAgg=IWAgg;
  LCO.jd=jd;
  System.out.println("         Optimierer: "+LCO.Size/(1024*1024)+" MByte");
 }

 public void reset(ModelParameters Params)
 {
  q0=Params.P[0];de=Params.P[1];du=Params.P[2];df=Params.P[3];jd=Params.P[4];
  jc=Params.P[5];jv=Params.P[6];R=Params.P[7];beta=Params.P[8];gamma=Params.P[9];

  Grid MG=new LinWeightsGrid(10,10/(24*(1-Math.exp(-jc))+5),jc,jv,2,1.5);

  IW=new IntegrationWeights(MG,jc,jv);
  IWAgg[MG.numPoints-1]=1-IW.Weights[MG.numPoints-1];
  for (int i=MG.numPoints-2;i>=0;i--) IWAgg[i]=IWAgg[i+1]-IW.Weights[i];

  Inst.reset(q0,R,Params.P[10]);

  for (int i=1;i<=L;i++) CMgamma[i]=Math.pow(Inst.CM[i],gamma);
  for (int i=1;i<=L;i++) Periods[i].reset(MG);

  LCO.jd=jd;
 }

 public final double u(double c,int j)
 {
  if (c<0) return Double.NEGATIVE_INFINITY;
  else if (gamma==1) return CMgamma[j]*Math.log(c);
  else return CMgamma[j]*(Math.exp(Math.log(c)*(1-gamma))-1)/(1-gamma);
 }

 private final double uprime(double c,int j)
 {
  if (c<0) return Double.POSITIVE_INFINITY;
  return CMgamma[j]*Math.exp(-Math.log(c)*gamma);
 }

 private final double invuprime(double dEV,int j)
 {
  if (dEV<=0) return Double.POSITIVE_INFINITY;
  return Inst.CM[j]*Math.exp(-Math.log(R*beta*dEV)/gamma);
 }


 private int binarySearch(double[] A,int max,double key)
 {
  if (A[0]>=key) return 0;
  if (A[--max]<=key) return max;
  int m,min=0;
  while (max-min>1)
  {
   m=(max+min)>>1;
   if (A[m]>key) max=m;else min=m;
  }
  if (A[max]-key>key-A[min]) return min;
  return max;
 }

 private double oA,oV;

 private final void computeOptAsset(int j,int A,double X,LCOptimiser LCO)
 {
  if (LCO.EIndex[A+1]-LCO.EIndex[A]==1)
  {
   double c=invuprime((LCO.EValue[A+1]-LCO.EValue[A])/(LCO.mPos[A+1]-LCO.mPos[A]),j);
   oA=R*(X-c);
   oV=u(c,j)+beta*((oA-LCO.mPos[A])/(LCO.mPos[A+1]-LCO.mPos[A])*(LCO.EValue[A+1]-LCO.EValue[A])+LCO.EValue[A]);
  }
  else
  {
   double ml,mr,or,c;
   int a=LCO.EIndex[A];
   oV=Double.NEGATIVE_INFINITY;
   ml=LCO.mPosO[a];
   while ((LCO.mPosO[a]<X*R) && (a<LCO.EIndex[A+1]))
   {
    c=invuprime((LCO.EValueO[a+1]-LCO.EValueO[a])/(LCO.mPosO[a+1]-LCO.mPosO[a]),j);
    mr=R*(X-c);
    if ((ml>=LCO.mPosO[a]) && (mr<=LCO.mPosO[a]))
    {
     or=u(X-LCO.mPosO[a]/R,j)+beta*LCO.EValueO[a];
     if (or>oV) {oV=or;oA=LCO.mPosO[a];}
    }
    else if ((mr<LCO.mPosO[a+1]) && (mr>LCO.mPosO[a]))
    {
     or=u(c,j)+beta*((mr-LCO.mPosO[a])/(LCO.mPosO[a+1]-LCO.mPosO[a])*(LCO.EValueO[a+1]-LCO.EValueO[a])+LCO.EValueO[a]);
     if (or>oV) {oV=or;oA=LCO.mPosO[a];}
    }
    ml=mr;
    a++;
   }
   if (ml>=LCO.mPos[A+1])
   {
    or=u(X-LCO.mPos[A+1]/R,j)+beta*LCO.EValue[A+1];
    if (or>oV) {oV=or;oA=LCO.mPos[A+1];}
   }
  }
 }

 private final void AssetChoice(int j,int MB,int H,int P,int U,int e,LCOptimiser LCO)
 {
  int nbp=j+1;if (Periods[j+1].Basis>0) nbp=Periods[j+1].Basis;
  int tp=j;if (Periods[j].Basis>0) tp=Periods[j].Basis;
  double Assets,MatchBenefit=(e==1)?Periods[tp].MatchGrid.Values[MB]:Periods[tp].BenefitGrid.Values[MB],HK=Periods[tp].HKGrid.Values[H],Pension=Periods[tp].PensionGrid.Values[P],UI=((e==1)?Periods[tp].UIEGrid.Values[U]:Periods[tp].UIUGrid.Values[U]);
  double cAssets,cValue;

  double upm,upl,upr;
  double y,AssetMin=LCO.getMin(); 
   
  int A,An;
  
  An=Periods[nbp].AssetGrid.getIndex(AssetMin);if (Periods[nbp].AssetGrid.Values[An]>AssetMin) An--;
  
  for (A=0;A<=An;A++) LCO.EValueO[A]=Double.NEGATIVE_INFINITY;
  for (A=An+1;A<Periods[nbp].AssetGrid.numPoints;A++) LCO.EValueO[A]=LCO.getEValue(A);
  LCO.mPosO=Periods[nbp].AssetGrid.Values;

  //Create concave value function
  LCO.EAnz=0;
  if (An<0) An=0;
  for (A=An;A<Periods[nbp].AssetGrid.numPoints;A++)
  {
   while ((LCO.EAnz>1) && ((LCO.EValue[LCO.EAnz-1]-LCO.EValue[LCO.EAnz-2])*(Periods[nbp].AssetGrid.Values[A]-LCO.mPos[LCO.EAnz-1])<(LCO.EValueO[A]-LCO.EValue[LCO.EAnz-1])*(LCO.mPos[LCO.EAnz-1]-LCO.mPos[LCO.EAnz-2]))) LCO.EAnz--;
   LCO.EValue[LCO.EAnz]=LCO.EValueO[A];
   LCO.mPos[LCO.EAnz]=Periods[nbp].AssetGrid.Values[A];
   LCO.EIndex[LCO.EAnz]=A;
   LCO.EAnz++;
  }

  for (A=0;A<Periods[tp].AssetGrid.numPoints;A++)
  {
   Assets=Periods[tp].AssetGrid.Values[A];
   y=Inst.y(Assets,MatchBenefit,HK,Pension,UI,e,j);

   if ((Assets+y)*R>=AssetMin)
   {	
    if (LCO.StableGrid)
    {
     if (e==0) cAssets=LCO.LAssetU[P][U][H][MB][A];
     else cAssets=LCO.LAssetE[P][U][H][MB][A];
    }
    else cAssets=Assets+y;
    An=binarySearch(LCO.mPos,LCO.EAnz,cAssets);

    while (LCO.EValue[An]==Double.NEGATIVE_INFINITY) An++;
    upm=uprime(Assets-LCO.mPos[An]/R+y,j);

    if ((An<LCO.EAnz-1) && (R*beta*(LCO.EValue[An+1]-LCO.EValue[An])>upm*(LCO.mPos[An+1]-LCO.mPos[An])))
    {
     upr=uprime(Assets-LCO.mPos[An+1]/R+y,j);
     while ((An<LCO.EAnz-2) && (R*beta*(LCO.EValue[An+2]-LCO.EValue[An+1])>upr*(LCO.mPos[An+2]-LCO.mPos[An+1])))
      {An++;upm=upr;upr=uprime(Assets-LCO.mPos[An+1]/R+y,j);}
 
     if (R*beta*(LCO.EValue[An+1]-LCO.EValue[An])<upr*(LCO.mPos[An+1]-LCO.mPos[An]))
     {
      computeOptAsset(j,An,Assets+y,LCO);cAssets=oA;cValue=oV;
     }
     else
     {
      cAssets=LCO.mPos[An+1];
      cValue=u(Assets-cAssets/R+y,j)+beta*LCO.EValue[An+1];
     }
    }
    else if ((An>0) && ((upm==Double.POSITIVE_INFINITY) || (R*beta*(LCO.EValue[An]-LCO.EValue[An-1])<upm*(LCO.mPos[An]-LCO.mPos[An-1]))))
    {
     upl=uprime(Assets-LCO.mPos[An-1]/R+y,j);
     while ((An>1) && (LCO.EValue[An-1]>Double.NEGATIVE_INFINITY) && (R*beta*(LCO.EValue[An-1]-LCO.EValue[An-2])<=upl*(LCO.mPos[An-1]-LCO.mPos[An-2])))
      {An--;upm=upl;upl=uprime(Assets-LCO.mPos[An-1]/R+y,j);}
 
     if (LCO.EValue[An-1]==Double.NEGATIVE_INFINITY)
     {
      if ((An>=LCO.EAnz-1) || (LCO.EValue[An]>=LCO.EValue[An+1]))
      {
       cAssets=AssetMin;
       cValue=u(Assets-cAssets/R+y,j)+beta*LCO.EValue[An];
      }
      else
      {
       double m0=AssetMin,mm=LCO.mPos[An],mr=LCO.mPos[An+1],eps,c;
       double EVm=LCO.EValue[An],EVr=LCO.EValue[An+1];
      
       eps=Math.exp(-Math.log(R*beta*(EVr-EVm)/((mr-mm)*CMgamma[j]))/gamma);
       c=eps*(R*(Assets+y)-m0)/(mm-m0+eps*R);
       cAssets=(Assets+y-c)*R;     
       if (gamma==1) cValue=Math.log((cAssets-m0)/(mm-m0))*(mm-m0)*(EVr-EVm)/(mr-mm)+EVm;
       else cValue=(Math.exp(Math.log((cAssets-m0)/(mm-m0))*(1-gamma))-1)*(mm-m0)*(EVr-EVm)/((mr-mm)*(1-gamma))+EVm;
       cValue=u(c,j)+beta*cValue;
      } 
     }  
     else if (R*beta*(LCO.EValue[An]-LCO.EValue[An-1])>upl*(LCO.mPos[An]-LCO.mPos[An-1]))
     {
      computeOptAsset(j,An-1,Assets+y,LCO);cAssets=oA;cValue=oV;
     }
     else
     {
      cAssets=LCO.mPos[An-1];
      cValue=u(Assets-cAssets/R+y,j)+beta*LCO.EValue[An-1];
     }
    }
    else
    {
     cAssets=LCO.mPos[An];
     cValue=u(Assets-cAssets/R+y,j)+beta*LCO.EValue[An];
    }
 
   }
   else
   {
    cAssets=AssetMin;
    cValue=Double.NEGATIVE_INFINITY;
   }
          	 
   if (Periods[j].Basis>0)
   {
    if (Periods[j].Type==2) {LCO.ValueE[P][0][0][0][A]=cValue;LCO.AssetE[P][0][0][0][A]=cAssets;}
    else if (e==1) {LCO.ValueE[P][U][H][MB][A]=cValue;LCO.AssetE[P][U][H][MB][A]=cAssets;}
    else {LCO.ValueU[P][U][H][MB][A]=cValue;LCO.AssetU[P][U][H][MB][A]=cAssets;}
   }
   else
   {   
    if (Periods[j].Type==2) {Periods[j].ConsR[P][A]=(float)(Assets+y-cAssets/R);LCO.ValueE[P][0][0][0][A]=cValue;LCO.AssetE[P][0][0][0][A]=cAssets;}
    else if (e==1) {if (U==LCO.U) Periods[j].ValueE[P][H][MB][A]=(float)((MB==0)?0:cValue-LCO.ValueE[P][U][H][0][A]);Periods[j].ConsE[U][P][H][MB][A]=(float)(Assets+y-cAssets/R);LCO.ValueE[P][U][H][MB][A]=cValue;LCO.AssetE[P][U][H][MB][A]=cAssets;}
    else {Periods[j].ValueU[U][P][MB][H][A]=(float)(cValue-LCO.ValueE[P][LCO.U][H][0][A]);Periods[j].ConsU[U][P][MB][H][A]=(float)(Assets+y-cAssets/R);LCO.ValueU[P][U][H][MB][A]=cValue;LCO.AssetU[P][U][H][MB][A]=cAssets;}
   }
   
  } 
 }

 private final void nextMin(int j,int MB,int H,int P,int U,int e,LCOptimiser LCO)
 {
   int nbp=j+1;if (Periods[j+1].Basis>0) nbp=Periods[j+1].Basis;
   int tp=j;if (Periods[j].Basis>0) tp=Periods[j].Basis;
   double MatchBenefit=(e==1)?Periods[tp].MatchGrid.Values[MB]:Periods[tp].BenefitGrid.Values[MB],HK=Periods[tp].HKGrid.Values[H],Pension=Periods[tp].PensionGrid.Values[P],UI=((e==1)?Periods[tp].UIEGrid.Values[U]:Periods[tp].UIUGrid.Values[U]);

   double Save=LCO.getMin()/R,Amin=Save-Inst.y(0,MatchBenefit,HK,Pension,UI,e,j),diff;
   do
   {
    diff=Save-Inst.y(Amin,MatchBenefit,HK,Pension,UI,e,j)-Amin;
    Amin+=diff;
   } while (Math.abs(diff)>1);
   if (e==1) LCO.ValueE[P][U][H][MB][Periods[j].AssetGrid.numPoints]=Amin;
   else LCO.ValueU[P][U][H][MB][Periods[j].AssetGrid.numPoints]=Amin;
 }

 public final void optimise()
 {
  int A,MB,H,U,P;
  int tp,np;
  double HK,Match,Benefit,Pension,UI;
 
  LCO.ValueE[0][0][0][0][0]=LCO.AssetE[0][0][0][0][0]=0;LCO.ValueE[0][0][0][0][1]=-0.001;
  
  System.out.print("    optimiere: Alt");
  LCO.Type=2;

  for (int j=L;j>=Lr;j--)
  {
   tp=j;np=j+1;
   if ((Periods[tp].AssetGrid==Periods[np].AssetGrid) && (Periods[tp].MatchGrid==Periods[np].MatchGrid) && (Periods[tp].BenefitGrid==Periods[np].BenefitGrid) && (Periods[tp].HKGrid==Periods[np].HKGrid) && (Periods[tp].PensionGrid==Periods[np].PensionGrid) && (Periods[tp].UIEGrid==Periods[np].UIEGrid) && (Periods[tp].UIUGrid==Periods[np].UIUGrid)) LCO.StableGrid=true;else LCO.StableGrid=false;
   LCO.change();
   LCO.Am=Periods[np].AssetGrid.numPoints;
   for (P=0;P<Periods[j].PensionGrid.numPoints;P++)
   {
    LCO.P=(j<L)?P:0;
    AssetChoice(j,0,0,P,0,1,LCO);
    nextMin(j,0,0,P,0,1,LCO);
   }

  }
  for (P=0;P<Periods[Lr].PensionGrid.numPoints;P++)
  {
   for (A=0;A<Periods[Lr].AssetGrid.numPoints;A++)
   {
    LCO.ValueU[P][0][0][0][A]=LCO.ValueE[P][0][0][0][A];
    LCO.AssetU[P][0][0][0][A]=LCO.AssetE[P][0][0][0][A];
   }
   LCO.ValueU[P][0][0][0][Periods[Lr].AssetGrid.numPoints]=LCO.ValueE[P][0][0][0][Periods[Lr].AssetGrid.numPoints];
  }

  System.out.print(", Jung");
  for (int j=Lr-1;j>0;j--)
  {
   if (j%24==0) System.out.print(".");
   tp=j;if (Periods[j].Basis>0) tp=Periods[j].Basis;
   np=j+1;if (Periods[j+1].Basis>0) np=Periods[j+1].Basis;
   LCO.change();
   if ((Periods[tp].AssetGrid==Periods[np].AssetGrid) && (Periods[tp].MatchGrid==Periods[np].MatchGrid) && (Periods[tp].BenefitGrid==Periods[np].BenefitGrid) && (Periods[tp].HKGrid==Periods[np].HKGrid) && (Periods[tp].PensionGrid==Periods[np].PensionGrid) && (Periods[tp].UIEGrid==Periods[np].UIEGrid) && (Periods[tp].UIUGrid==Periods[np].UIUGrid)) LCO.StableGrid=true;else LCO.StableGrid=false;
   LCO.U=Periods[tp].UIEGrid.getIndex(Inst.empUI);
   LCO.Mmax=Periods[np].MatchGrid.numPoints-1;
   LCO.Am=Periods[np].AssetGrid.numPoints;
   for (H=0;H<Periods[tp].HKGrid.numPoints;H++)
   {
    HK=Periods[tp].HKGrid.Values[H];
    LCO.CountHee=Periods[np].HKGrid.getInterpolation(HK*(1+de),LCO.IndexHee,LCO.WeightsHee);
    LCO.CountHeu=Periods[np].HKGrid.getInterpolation(HK*(1+de)*(1-df),LCO.IndexHeu,LCO.WeightsHeu);
    LCO.CountHu=Periods[np].HKGrid.getInterpolation(HK*(1-du),LCO.IndexHu,LCO.WeightsHu);
    for (P=0;P<Periods[tp].PensionGrid.numPoints;P++)
    {
     Pension=Periods[tp].PensionGrid.Values[P];
     LCO.Type=1;
     for (U=0;U<Periods[tp].UIEGrid.numPoints;U++)
     {
      UI=Periods[tp].UIEGrid.Values[U];
      for (MB=0;MB<Periods[tp].MatchGrid.numPoints;MB++)
      {
       Match=Periods[tp].MatchGrid.Values[MB];
       LCO.M=(j+1<Lr)?MB:0;
       LCO.CountBeu=Periods[np].BenefitGrid.getInterpolation(Inst.nextBenefit(Match,HK,UI,j),LCO.IndexBeu,LCO.WeightsBeu);
       LCO.CountPe=Periods[np].PensionGrid.getInterpolation(Inst.nextPension(Match,HK,Pension,UI,1,j),LCO.IndexPe,LCO.WeightsPe);
       LCO.CountUee=Periods[np].UIEGrid.getInterpolation(Inst.nextUI(Match,HK,UI,1,j,1),LCO.IndexUee,LCO.WeightsUee);
       LCO.CountUeu=Periods[np].UIUGrid.getInterpolation(Inst.nextUI(Match,HK,UI,1,j,0),LCO.IndexUeu,LCO.WeightsUeu);
       AssetChoice(j,MB,H,P,U,1,LCO);
       nextMin(j,MB,H,P,U,1,LCO);
      }
     }
     LCO.Type=0;
     for (U=0;U<Periods[tp].UIUGrid.numPoints;U++)
     {
      UI=Periods[tp].UIUGrid.Values[U];
      for (MB=0;MB<Periods[tp].BenefitGrid.numPoints;MB++)
      {
       Benefit=Periods[tp].BenefitGrid.Values[MB];
       LCO.M=(j+1<Lr)?MB:0;
       LCO.CountPu=Periods[np].PensionGrid.getInterpolation(Inst.nextPension(Benefit,0,Pension,UI,0,j),LCO.IndexPu,LCO.WeightsPu);
       LCO.CountUu=Periods[np].UIUGrid.getInterpolation(Inst.nextUI(Benefit,HK,UI,0,j,0),LCO.IndexUu,LCO.WeightsUu);
       AssetChoice(j,MB,H,P,U,0,LCO);
       nextMin(j,MB,H,P,U,0,LCO);
      }      
     }

     for (A=0;A<Periods[tp].AssetGrid.numPoints;A++)
     {     
      LCO.IValue[P][H][Periods[tp].MatchGrid.numPoints-1][A]=LCO.ValueE[P][LCO.U][H][Periods[tp].MatchGrid.numPoints-1][A]*IW.Weights[Periods[tp].MatchGrid.numPoints-1];
      for (MB=Periods[tp].MatchGrid.numPoints-2;MB>=0;MB--) 
       LCO.IValue[P][H][MB][A]=LCO.IValue[P][H][MB+1][A]+LCO.ValueE[P][LCO.U][H][MB][A]*IW.Weights[MB];
     }
     
    }	
   }
  } 
  System.out.println("fertig");
 }

 private final int getInterpolation(int j,int[] Index,double[] Weights)
 {
  if (Periods[j].Basis==0) {Index[0]=j;Weights[0]=1;return 1;}
  if (Periods[j].LastFull==-1)
  {
   int LF=j-1;
   while ((LF>0) && (Periods[LF].Basis>0)) LF--;
   Periods[j].LastFull=LF;
  }
  if (Periods[j].LastFull==0) {Index[0]=Periods[j].Basis;Weights[0]=1;return 1;}
  Index[0]=Periods[j].LastFull;Index[1]=Periods[j].Basis;
  Weights[0]=(double)(Periods[j].Basis-j)/(Periods[j].Basis-Periods[j].LastFull);
  Weights[1]=(double)(j-Periods[j].LastFull)/(Periods[j].Basis-Periods[j].LastFull);
  return 2;
 }
 
 public final double Value(int Period,double Assets,double MatchBenefit,double HK,double Pension,double UI,int e) 
 {
  if (Periods[Period].Type!=1)
  {
   if ((Periods[Period].Type==0) && (Assets<0)) return Double.NEGATIVE_INFINITY; // For non-negative bequest
   return 0;
  }

  Count0=getInterpolation(Period,Index0,Weights0);

  IntValue=0;
  for (int ji=0;ji<Count0;ji++)
  {
   int j=Index0[ji];
   Count1=Periods[j].AssetGrid.getInterpolation(Assets,Index1,Weights1);
   Count3=Periods[j].HKGrid.getInterpolation(HK,Index3,Weights3);
   Count4=Periods[j].PensionGrid.getInterpolation(Pension,Index4,Weights4);
   if (e==1)
   {
    Count2=Periods[j].MatchGrid.getInterpolation(MatchBenefit,Index2,Weights2);
    for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int m=0;m<Count2;m++) for (int h=0;h<Count3;h++)
     IntValue+=Periods[j].ValueE[Index4[p]][Index3[h]][Index2[m]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights0[ji];
   }
   else 
   {
    Count2=Periods[j].BenefitGrid.getInterpolation(MatchBenefit,Index2,Weights2);
    Count5=Periods[j].UIUGrid.getInterpolation(UI,Index5,Weights5);
    for (int u=0;u<Count5;u++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int m=0;m<Count2;m++) for (int h=0;h<Count3;h++)
     IntValue+=Periods[j].ValueU[Index5[u]][Index4[p]][Index2[m]][Index3[h]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u]*Weights0[ji];
   }
  }
  return IntValue;
 }

 public final double Asset(int Period,double Assets,double MatchBenefit,double HK,double Pension,double UI,int e) 
 {
  if (Periods[Period].Type==0) return Double.NEGATIVE_INFINITY;
  
  Count0=getInterpolation(Period,Index0,Weights0);

  IntValue=0;
  for (int ji=0;ji<Count0;ji++)
  {
   int j=Index0[ji];
   Count1=Periods[j].AssetGrid.getInterpolation(Assets,Index1,Weights1);
   Count4=Periods[j].PensionGrid.getInterpolation(Pension,Index4,Weights4);
   if (Periods[j].Type==1)
   {
    Count3=Periods[j].HKGrid.getInterpolation(HK,Index3,Weights3);
    if (e==1)
    {
     Count2=Periods[j].MatchGrid.getInterpolation(MatchBenefit,Index2,Weights2);
     Count5=Periods[j].UIEGrid.getInterpolation(UI,Index5,Weights5);
     for (int u=0;u<Count5;u++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int m=0;m<Count2;m++) for (int h=0;h<Count3;h++) 
      IntValue+=Periods[j].ConsE[Index5[u]][Index4[p]][Index3[h]][Index2[m]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u]*Weights0[ji];
    }
    else
    {
     Count2=Periods[j].BenefitGrid.getInterpolation(MatchBenefit,Index2,Weights2);
     Count5=Periods[j].UIUGrid.getInterpolation(UI,Index5,Weights5);
     for (int u=0;u<Count5;u++) for (int m=0;m<Count2;m++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int h=0;h<Count3;h++) 
      IntValue+=Periods[j].ConsU[Index5[u]][Index4[p]][Index2[m]][Index3[h]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u]*Weights0[ji];
    }
   }
   else
    for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++)
     IntValue+=Periods[j].ConsR[Index4[p]][Index1[a]]*Weights1[a]*Weights4[p]*Weights0[ji];
  } 

  return R*(Assets+Inst.y(Assets,MatchBenefit,HK,Pension,UI,e,Period)-IntValue);
 }

 public final double Consumption(int Period,double Assets,double MatchBenefit,double HK,double Pension,double UI,int e) 
 {
  if (Periods[Period].Type==0) return Double.NEGATIVE_INFINITY;
  
  Count0=getInterpolation(Period,Index0,Weights0);

  double IV0=0,IV1=0;

   int j=Index0[0],k=(Count0>1)?Index0[1]:0;
   Count1=Periods[j].AssetGrid.getInterpolation(Assets,Index1,Weights1);
   Count4=Periods[j].PensionGrid.getInterpolation(Pension,Index4,Weights4);
   if (Periods[j].Type==1)
   {
    Count3=Periods[j].HKGrid.getInterpolation(HK,Index3,Weights3);
    if (e==1)
    {
     Count2=Periods[j].MatchGrid.getInterpolation(MatchBenefit,Index2,Weights2);
     Count5=Periods[j].UIEGrid.getInterpolation(UI,Index5,Weights5);
     if ((Count0>1) && (Periods[j].AssetGrid==Periods[k].AssetGrid) && (Periods[j].PensionGrid==Periods[k].PensionGrid) && (Periods[j].HKGrid==Periods[k].HKGrid) && (Periods[j].MatchGrid==Periods[k].MatchGrid) && (Periods[j].UIEGrid==Periods[k].UIEGrid))
     {
      double W;
      for (int u=0;u<Count5;u++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int m=0;m<Count2;m++) for (int h=0;h<Count3;h++)
      {
       W=Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u];
       IV0+=Periods[j].ConsE[Index5[u]][Index4[p]][Index3[h]][Index2[m]][Index1[a]]*W;
       IV1+=Periods[k].ConsE[Index5[u]][Index4[p]][Index3[h]][Index2[m]][Index1[a]]*W;
      }
      return IV0*Weights0[0]+IV1*Weights0[1];
     }
     else
     {
      for (int u=0;u<Count5;u++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int m=0;m<Count2;m++) for (int h=0;h<Count3;h++) 
       IV0+=Periods[j].ConsE[Index5[u]][Index4[p]][Index3[h]][Index2[m]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u];
     }
    }
    else
    {
     Count2=Periods[j].BenefitGrid.getInterpolation(MatchBenefit,Index2,Weights2);
     Count5=Periods[j].UIUGrid.getInterpolation(UI,Index5,Weights5);
/*     if ((Count0>1) && (Periods[j].AssetGrid==Periods[k].AssetGrid) && (Periods[j].PensionGrid==Periods[k].PensionGrid) && (Periods[j].HKGrid==Periods[k].HKGrid) && (Periods[j].BenefitGrid==Periods[k].BenefitGrid) && (Periods[j].UIUGrid==Periods[k].UIUGrid))
     {
      double W;
      for (int u=0;u<Count5;u++) for (int m=0;m<Count2;m++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int h=0;h<Count3;h++) 
      {
       W=Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u];
       IV0+=Periods[j].ConsU[Index5[u]][Index4[p]][Index2[m]][Index3[h]][Index1[a]]*W;
       IV1+=Periods[k].ConsU[Index5[u]][Index4[p]][Index2[m]][Index3[h]][Index1[a]]*W;
      } 
      return IV0*Weights0[0]+IV1*Weights0[1];
     }
     else*/
     {
      for (int u=0;u<Count5;u++) for (int m=0;m<Count2;m++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int h=0;h<Count3;h++) 
       IV0+=Periods[j].ConsU[Index5[u]][Index4[p]][Index2[m]][Index3[h]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u];
     }
    }
   }
   else
    for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++)
     IV0+=Periods[j].ConsR[Index4[p]][Index1[a]]*Weights1[a]*Weights4[p];

  if (Count0>1)
  {
   j=Index0[1];
   if (Periods[j].AssetGrid!=Periods[Index0[0]].AssetGrid) Count1=Periods[j].AssetGrid.getInterpolation(Assets,Index1,Weights1);
   if (Periods[j].PensionGrid!=Periods[Index0[0]].PensionGrid) Count4=Periods[j].PensionGrid.getInterpolation(Pension,Index4,Weights4);
   if (Periods[j].Type==1)
   {
    if (Periods[j].HKGrid!=Periods[Index0[0]].HKGrid) Count3=Periods[j].HKGrid.getInterpolation(HK,Index3,Weights3);
    if (e==1)
    {
     if (Periods[j].MatchGrid!=Periods[Index0[0]].MatchGrid) Count2=Periods[j].MatchGrid.getInterpolation(MatchBenefit,Index2,Weights2);
     if (Periods[j].UIEGrid!=Periods[Index0[0]].UIEGrid) Count5=Periods[j].UIEGrid.getInterpolation(UI,Index5,Weights5);
     for (int u=0;u<Count5;u++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int m=0;m<Count2;m++) for (int h=0;h<Count3;h++) 
      IV1+=Periods[j].ConsE[Index5[u]][Index4[p]][Index3[h]][Index2[m]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u];
    }
    else
    {
     if (Periods[j].BenefitGrid!=Periods[Index0[0]].BenefitGrid) Count2=Periods[j].BenefitGrid.getInterpolation(MatchBenefit,Index2,Weights2);
     if (Periods[j].UIUGrid!=Periods[Index0[0]].UIUGrid) Count5=Periods[j].UIUGrid.getInterpolation(UI,Index5,Weights5);
     for (int u=0;u<Count5;u++) for (int m=0;m<Count2;m++) for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++) for (int h=0;h<Count3;h++) 
      IV1+=Periods[j].ConsU[Index5[u]][Index4[p]][Index2[m]][Index3[h]][Index1[a]]*Weights1[a]*Weights2[m]*Weights3[h]*Weights4[p]*Weights5[u];
    }
   }
   else
    for (int p=0;p<Count4;p++) for (int a=0;a<Count1;a++)
     IV1+=Periods[j].ConsR[Index4[p]][Index1[a]]*Weights1[a]*Weights4[p];

   return IV0*Weights0[0]+IV1*Weights0[1];
  }

  return IV0;
 }

 public final boolean Accept(int j,double Assets,double Benefits,double HK,double Pension,double UI,double Matchn)
 {
  if (Periods[j+1].Type!=1) return false;
  double Assetsn=Asset(j,Assets,Benefits,HK,Pension,UI,0),HKn=HK*(1-du),Pensionn=Inst.nextPension(Benefits,HK,Pension,UI,0,j);
  if (Value(j+1,Assetsn,Matchn,HKn,Pensionn,Inst.nextUI(Benefits,HK,UI,0,j,1),1)>Value(j+1,Assetsn,Benefits,HKn,Pensionn,Inst.nextUI(Benefits,HK,UI,0,j,0),0)) return true;
  else return false;
 }
 
// zum Interpolieren 
 private int Count0,Count1,Count2,Count3,Count4,Count5;
 private int[] Index0=new int[2],Index1=new int[2],Index2=new int[2],Index3=new int[2],Index4=new int[2],Index5=new int[2];
 private double[] Weights0=new double[2],Weights1=new double[2],Weights2=new double[2],Weights3=new double[2],Weights4=new double[2],Weights5=new double[2];
 private double IntValue;
  

 public int    L,Lr;
 public double q0,de,du,df,jd,jc,jv,R;  
 public double beta,gamma;

 public Period Periods[];
 public Institutions Inst;
 
 private IntegrationWeights IW;
 private LCOptimiser LCO;

 private double[] IWAgg;
 private int[] empU;
 private double[] CMgamma;
}