#include <oxstd.h>
#include <oxdraw.h>

// This array contains the names of the countries in the order they appear in the .xls files
static decl asCountries = {
"Belgium",			//  0
"Bulgaria",			//  1
"Czech Republic",	//  2
"Denmark",			//  3
"Germany",			//  4
"Estonia",			//  5
"Ireland",			//  6
"Greece",			//  7
"Spain",			//	8
"France",			//  9
"Italy",			// 10
"Cyprus",			// 11
"Latvia",			// 12
"Lithuania",		// 13
"Luxembourg",		// 14
"Hungary",			// 15
"Malta",			// 16
"Netherlands",		// 17
"Austria",			// 18
"Poland",			// 19
"Portugal",			// 20
"Romania",			// 21
"Slovenia",			// 22
"Slovakia",			// 23
"Finland",			// 24
"Sweden",			// 25
"United Kingdom",	// 26
"Iceland",			// 27
"Norway",			// 28
"Switzerland",		// 29
"Croatia",			// 30
"Turkey"};			// 31

// This is used to be able to use the names NONE, CONST, TREND in the ADF and KPSS tests instead of the option number 0, 1, 2
enum
{
	NONE=0, CONST=1, TREND=2
};


// Definition of the functions ------------------------------------------------------------------------

adf(const vY, const iDet, const iLag)
// Performs ADF test on the vector vY, with deterministic component iDet (= NONE, CONST, TREND)
// and iLag lags of the differenced variable vY.
// It returns the ADF "tau" test statistic.
{
	decl cT = rows(vY);
	decl cK = 1+iDet+iLag;
	decl mX = new matrix[cT][cK];
	decl cNdx = 0;
	mX[][cNdx++] = lag0(vY,1);
	if (iDet>0)
	{
		mX[][cNdx++] = ones(cT,1);
		if (iDet>1)
		{
			mX[][cNdx++] = range(1,cT)';
		}
	}
	decl vDY = diff0(vY,1);
	for (decl i=1; i<=iLag; ++i)
	{
		mX[][cNdx++] = lag0(vDY,i);
	}
	decl vBeta, mInvXX;
	decl iFirst = 1+iLag;
	olsc(vDY[iFirst:], mX[iFirst:][], &vBeta, &mInvXX);
	decl dSig2 = meanc( (vDY[iFirst:]-mX[iFirst:][]*vBeta).^2 )*cT/(cT-cK);
	return vBeta[0]/sqrt(dSig2*mInvXX[0][0]);
}

bartlett(const vX, const iBW)
// Computes the Bartlett kernel on the vector vX with bandwidth iBW.
// It returns a vector of the same dimension ov vX
{
	return (fabs(vX).>iBW).? 0 .: (1-fabs(vX)./(iBW+1));
}

kpss(const mY, const iDet, const iBW)
// Performs the KPSS test on the time series in the columns of mY
// with deterministic component iDet (=NONE, CONST, TREND) and
// Bartlett kernel with bandwidth iBW.
// It returns a row vector with as many columns as mY
// with the KPSS test statistics
{
	decl mZ, cT=rows(mY);
	switch_single (iDet)
	{
		case 1: mZ = mY - meanc(mY);
		case 2:
			decl vBeta, mX = ones(cT,1)~(range(1,cT)');
			olsc(mY, mX, &vBeta);
			mZ = mY - mX*vBeta;
		default:
			mZ = mY;
	}
	decl vEta = meanc(cumsum(mZ,1).^2)/cT;
	decl vLVar;
	if (iBW == 0)
	{
		vLVar = varc(mZ);
	}
	else
	{
		decl mAcf = acf(mZ, iBW);
		vLVar = varc(mZ).*(1+bartlett(range(1,iBW),iBW)*mAcf[1:iBW][]);
	}
	return vEta./vLVar;
}

sim_har(const nr, const nc, const mu, const sd)
// Simulates nc time series of nr observations with
// mean-zero, variance-one AR(1) processes with
// AR(1) coefficient drawn from a normal variate
// with mean mu and st.dev. sd but truncated to the
// range [-.99,.99].
// It returns a nr x mc matrix with the simulated
// AR(1) in the columns
{
	decl mAR = new matrix[nr][nc];
	decl vrho = rann(1,nc)*sd+mu;
	vrho = (vrho .> 1).? 0.99 .: vrho;
	vrho = (vrho .< -1).? -0.99 .: vrho;
	parallel for (decl i=0; i<nc; ++i)	 // if your version of Ox < 7, remove "parallel" before for (...)
	{
		mAR[][i] = cumsum(rann(nr,1)*sqrt(1-vrho[i]),vrho[i]);
	}
	return mAR;
}

sim_adf_kpss(const vMean, const vMeanDelta, const mCh0, const mCh1, const vW0, const vW1,
             const cT, const cS, const iDet, const iLag, const iBW, const dMeanPhi0, const dSdPhi0, const dMeanPhi1, const dSdPhi1)
// It computes the ADF and KPSS statistics for real exchange rates time series generated as in
// equation (8) and the one followinf immediatly in the article.
// 
// vMean (1 x K) drifts of the multivariate RW
// vMeanDelta (1 x K) mean difference vector between log-prices in country 1 and 0
// mCh0, mCh1 (K x K) Choleski of covariance matrices of base prices and of price differences between countries
// vW0, vW1 (1 x K)	weights of the two countries
// cT (scalar int) length of time series
// cS (scalar int) number of simulations
// iDet (scalar int) 0 = none, 1 = constant, 2 = linear trend
// iLag (scalar int) number of lags of differences in ADF test
// iBW (scalar int)	bandwidth for KPSS
// dMeanPhi0 (scalar double) mean of  AR(1) coefficient of the ARIMA(1,1,0) process generating prices in country 0
// dSdPhi0 (scalar double) st.dev. of AR(1) coefficient of the ARIMA(1,1,0) process generating prices in country 0
// dMeanPhi1 (scalar double) mean of  AR(1) coefficient of the ARIMA(1) process generating price differences in the 2 countries
// dSdPhi1 (scalar double) st.dev. of AR(1) coefficient of the ARIMA(1) process generating price differences in the 2 countries
{
	decl cN = rows(mCh0);
	decl mX0, mX1, vY, vBeta;
	decl mTest = new matrix[3][cS];
	for (decl i=0; i<cS; ++i)
	{
		mX0 = cumsum(vMean+sim_har(cT,cN,dMeanPhi0,dSdPhi0)*mCh0',1);
		mX1 = mX0 + vMeanDelta + sim_har(cT,cN,dMeanPhi1,dSdPhi1)*mCh1';
		vY  = log(exp(mX1)*vW1)-log(exp(mX0)*vW0);
		mTest[0][i] =  adf(vY, iDet, iLag);
		mTest[1][i] = kpss(vY, iDet, iBW);
		olsc(vY[1:cT-1], ones(cT-1,1)~vY[0:cT-2],&vBeta);
		mTest[2][i] = vBeta[1];
	}
	return mTest;
}


// Main program: it creates printed output and .xls files --------------------------------------------------------------

main()
{
	decl asVarNames;
	decl mW=loadmat("ItemWeights2005ox.xls",&asVarNames)[][1:];	// 2005 CPI weights	for each country listed above
		 mW = mW./sumr(mW)*100;									// weights are normalized to sum exately to 100
	decl mP0=log(loadmat("YearlyFR.xls"));		  	// yearly log-CPIs for France
	decl mP1=log(loadmat("YearlyBE.xls"));		  	// yearly log-CPIs for Belgium and some Deutschland when a Belgian time series is missing
	decl cYears = rows(mP0);						// number of years in the sample

	// France
	decl mDP0   = mP0[1:][] - mP0[:cYears-2][];	   // first diff. of Franch log-CPIs (~ sector inflation rates)
	decl vMean0 = meanc(mDP0);					   // mean rate of inflation for each sector CPI (France)
	decl mCov0  = variance(mDP0*100);
	mCov0 += diagonalize(mCov0)*0.1;               // inflating a bit the variance a bit to guarantee p.d. of the covariance matrix
	decl mCh0	= choleski(mCov0);				   // Choleski of covariance matrix of Franch inflation rate
	// Belgium-France Difference
	decl mP10    = mP1 - mP0;					   // first diff. of Belgian log-CPIs (~ sector inflation rates)
	decl vMean10 = meanc(mP10);					   // mean rate of inflation for each sector CPI (Belgium)
	decl mCov10  = variance(mP10);
	mCov10 += diagonalize(mCov10)*0.1;             // inflating a bit the variance a bit to guarantee p.d. of the covariance matrix
	decl mCh10   = choleski(mCov10);			   // Choleski of covariance matrix of Belgian inflation rate

	// *** PART TO PLAY WITH ****************************************************************************************
	decl cT = 30;	  		            // Length of (yearly) time series
	decl iBase = 9;				        // index of base country (9 = France, see above for the list)
	decl iComp = 21;                    // index of comparison country (21 = Romania, see above for the list)
	decl iBW = int(12*(cT/100)^0.25);   // bandwidth for KPSS
	decl iLag = 4;					    // max lag of ADF
	decl cSims = 10000;					// number of simulations
	// **************************************************************************************************************
	
	decl cCountries = sizeof(asCountries);	// number of listed countries (see above for list)
	decl cN = rows(mCov0);		            // number of COICOP items
	decl vW0 = mW[iBase][]';	            // CPI weights in the base country
	decl vW1 = mW[iComp][]';	            // CPI weights in the comparison country
	decl vWdist, mTest;						// variables to be used in the for loop below
	decl mTab = new matrix[cCountries][6];	// container of test results

	// This loop performs the simulations for the ADF and KPSS statistic using the base
	// country weights vs. all the other countries' weights
	for (decl i=0; i< cCountries; ++i)
	{
		println("Simulating with weights: ",asCountries[iBase],"-",asCountries[i]);
		vW1 = mW[i][]';	          // weights of the i-th comparison country
		vWdist = fabs(vW1 - vW0); // absolute differences of the weights

	mTab[i][3] = double( sqrt((vWdist'vWdist)/rows(vWdist)) );	// Root Mean Squared Distance of weights
	mTab[i][4] = double( meanc(vWdist) );						// Mean Absolute Difference of weights 
	mTab[i][5] = double( maxc(vWdist) );						// Max Difference among the weights
	
	// time series are simulated and ADF and KPSS sttaistics generated
	mTest = sim_adf_kpss(vMean0, vMean10, mCh0, mCh10, vW0, vW1, cT, cSims, CONST, iLag, iBW, 0,0,0,0);	// changing the last four parametrs you change the AR persistence
	mTab[i][0] = double(meanr(mTest[0][].< -2.8737)*100); // Percentage of rejections for the ADF
	mTab[i][1] = double(meanr(mTest[1][].> 0.463)*100);	  // Percentage of rejections for the KPSS
	mTab[i][2] = double(meanr(deletec(mTest[2][])));	  // Mean estimated AR(1) coeff. on the simulated log-RER time series
	}
	savemat(sprint("Rej",cT,".xls"),mTab,{"ADF","KPSS","AR1","RMSD","MAD","MaxD"});	// saving all results to a .xls file

	// Here we simulate to see how the persistence changes when the weights and the time series length change
	decl vT = <20,30,50,100,150,200,250,300>;                // grid of time series length
	decl mAK = new matrix[columns(vT)][2];					 // will contain the percentage of rejections of the two tests for each sample size
	decl vProb = <0.1,0.2,0.3,0.4,0.5,0.6,0.7,0.8,0.9>;		 // percentiles to save of the distribution of the AR(1) coefficients
	decl mQuantPhi = new matrix[columns(vProb)][columns(vT)];// will contain the percentiles of the AR(1) coefficients for each sample size

	// This loop performs the simulations under different time series lengths
	for (decl i=0; i<columns(vT); ++i)
	{
		println("Simulating with time series length ",vT[i]);
		mTest = sim_adf_kpss(vMean0, vMean10, mCh0, mCh10, vW0, vW1, vT[i], cSims, CONST, iLag, iBW, 0.1, 0.3, 0.6, 0.2); // changing the last four parametrs you change the AR persistence
		mAK[i][0] = meanr(mTest[0][].< -2.8737)*100;			 // rejections of ADF
		mAK[i][1] = meanr(mTest[1][].> 0.463)*100;				 // rejections of KPSS
		mQuantPhi[][i] = quantiler(deletec(mTest[2][]), vProb)'; // percentiles of sample AR(1) coefficients
	}

	// Print and save results of last simulations
	println("%c",{"T","ADF","KPSS"},vT'~mAK);
	println("%c",{"T","10%","20%","30%","40%","50%","60%","70%","80%","90%"},vT'~mQuantPhi');
	savemat("FarWeightAR.xls",vT'~mAK,{"T","ADF","KPSS"});
	savemat("FarQuantPhiAR.xls",vT'~mQuantPhi',{"T","10%","20%","30%","40%","50%","60%","70%","80%","90%"});
}