clear 
close all
clc

% =========================================================================
% DESCRIPTION
% This script runs the pseudo-out-of-sample forecasting exercises for the
% trivariate systems.

% -------------------------------------------------------------------------
% USER-DEFINED VARIABLES

% Switch for running bootstraps
run_bootstrap=0;

% Switch for using great moderation data (0 => in-sample period begins in
% Jan. 1959, 1 => in-sample period begins in Jan. 1984)
great_mod=0;

% Switch for whether to use the MSW sample (forecasts for 1979:01+h to 2002:12) 
% or the 2003-2016 sample (forecasts for 2002:12+h to 2016:12). This is 
% only needed for bootstrapping.
msw_sample=0;

% Total number of computers used to create the bootstrap results (i.e. the
% number of sections the bootstraps are broken into)
ncomp=6;

% Current computer number (i.e. the number of the section of the bootstrap
% that this script will use); must be no more than ncomp.
comp=1;

% Bootstrap variables
blocksize=40; % block size
nbs=299; % number of bootstrapped samples

% Vintage of FRED-MD used (YYYY-MM)
vintage='2017-06';

% Series in each of the three groups
%   group1 = real series
%   group2 = nominal series
%   group3 = financial series
group1={'RPI','INDPRO','CE16OV','UNRATE','AWHMAN','DPCERA3M086SBEA'};
group2={'CES3000000008','WPSFD49207','OILPRICEx','CPIAUCSL','PCEPI'};
group3={'M1SL','FEDFUNDS','GS10','TWEXMMTH','S&P 500'};

% Minimum number of observations that each model must be estimated with for
% the forecasts for a certain forecast date to be considered
nobs_min=120;

% Seed numbers for each bootstrap
if great_mod==1
    seeds=1000000*(0:nbs)';
else
    seeds=1000000*(0:nbs)'+500000;
end

% Number of workers for parallel computing
nworkers=12; 

% Folder containng data
datafolder='Data';

% Folder containing auxiliary functions
auxfcn='FcnsFcast';

% Folder to where .mat files with forecats are saved
outfolder='RawOutput';

% =========================================================================
% PART 1: SETUP

addpath(auxfcn);

% -------------------------------------------------------------------------
% DEFINE IN-SAMPLE AND OUT-OF-SAMPLE PERIODS

% Check switch is defined properly
assert(great_mod+msw_sample<2);

% Define in-sample start and out-of-sample periods
%   sample_start = first in-sample period 
%   OOS_start = last in-sample period for the first forecast produced (so
%       the first period being forecast is OOS_start+h where h is the 
%       forecasting horizon)
%   OOSF_end = last period to be forecast
switch great_mod
    case 0
        sample_start='1/31/1959';
        OOS_start='1/31/1979';
    case 1
        sample_start='1/31/1984';
        OOS_start='12/31/2002';
        
    otherwise
        error('The switch great_mod is misspecified.');
end

switch msw_sample
    case 0
        OOSF_end='12/31/2016';
    case 1
        if run_bootstrap==1
            OOSF_end='12/31/2002';
        else
            OOSF_end='12/31/2016';
        end
    otherwise
        error('The switch msw_sample is misspecified.');
end
        
% -------------------------------------------------------------------------
% BOOTSTRAP INDICES
% Number of the first and last bootstrap sample to produced. Note that the 
% index of 0 refers to the actual sample.

switch run_bootstrap
    
    % If not running bootstraps, just use the actual sample
    case 0
        
        % Conditional exercises performed
        cond_options={'none','full1'};
        
        % Horizons
        horizons=[3,12,24];
        
        % Set the bootstrap indices to 0 for the actual sample
        bs_i1=0;
        bs_i2=0;
        
        % Methods used to determine lag order
        %   number - lag order is fixed at that number
        %   'BIC' - lag order chosen by the BIC
        %   'AIC' - lag order chosen by the AIC
        lags={4,12,'BIC','AIC'};
        
        
    % If running bootstraps, get index numbers for bootstrap samples to be
    % created based on which computer number is being used.
    case 1
        
        % Conditioning exercises performed; only consider conditioning on
        % the full path 
        cond_options={'full1'};
        
        % Horizons
        horizons=[3,12,24];
        
        % Make sure computer number is less than or equal to the total 
        % number of computers being used for the code
        assert(comp<=ncomp);

        % Determine first and last bootstrap sample to use based on the 
        % given computer number and total number of computers
        bs_i1=(comp-1)*ceil((nbs+1)/ncomp);
        if comp<ncomp
            bs_i2=comp*ceil((nbs+1)/ncomp)-1;
        else
            bs_i2=nbs;
        end
        
        % Methods used to determine lag order
        %   number - lag order is fixed at that number
        %   'BIC' - lag order chosen by the BIC
        %   'AIC' - lag order chosen by the AIC
        lags={2,4,12};
  
        
    % Otherwise there's an error
    otherwise
        error('The switch run_bootstrap is misspecified.');     
end

% =========================================================================
% PART 2: GET DATA

% -------------------------------------------------------------------------
% LOAD FRED-MD VINTAGE

% Import from csv
raw=importdata([datafolder,'\',vintage,'.csv']);

% Data
fredmd=raw.data(2:end,:);

% Series names
series=raw.textdata(1,2:end);

% Transformation codes
tcode=raw.data(1,:);

% Dates
dates=datenum(raw.textdata(3:end,1));

% -------------------------------------------------------------------------
% ADJUST/LABEL DATA

% Number of series
N1=length(group1);
N2=length(group2);
N3=length(group3);

% Adjust all dates to be the final day of the month
sample_start=eom(sample_start);
OOS_start=eom(OOS_start);
OOSF_end=eom(OOSF_end);
dates=eom(dates);

% Remove excess dates
fredmd=fredmd(dates>=sample_start & dates<=datenum(OOSF_end),:);
dates=dates(dates>=sample_start & dates<=datenum(OOSF_end));

% Forecast dates
assert(sample_start<OOS_start);
fdates=dates(dates>datenum(OOS_start));

% Transform data
[fredmd_t1,fredmd_t2,~]=transform(fredmd,tcode,1);
    
% Identify where series from each group lie in fredmd
id1=cellfun(@(x)find(strcmp(x,series),1),group1);
id2=cellfun(@(x)find(strcmp(x,series),1),group2);
id3=cellfun(@(x)find(strcmp(x,series),1),group3);

% Get tcode for the three groups
tcode1=tcode(id1);
tcode2=tcode(id2);
tcode3=tcode(id3);

% Get series for the three groups after the first transformation (level or
% log) is applied
fredmd1_t1=fredmd_t1(:,id1);
fredmd2_t1=fredmd_t1(:,id2);
fredmd3_t1=fredmd_t1(:,id3);

% Get series for the three groups after the first transformation (level or
% log) and second transformation (level, first difference, or second
% difference) are applied
fredmd1_t2=fredmd_t2(:,id1);
fredmd2_t2=fredmd_t2(:,id2);
fredmd3_t2=fredmd_t2(:,id3);

% Check that last row is non-missing
assert(sum(isnan(fredmd1_t1(end,:)))==0);
assert(sum(isnan(fredmd2_t1(end,:)))==0);
assert(sum(isnan(fredmd3_t1(end,:)))==0);


% =========================================================================
% PART 3: RUN FORECASTING EXERCISES

% -------------------------------------------------------------------------
% IDENTIFY SYSTEMS
% Identify each system of variables for which forecasts are to be
% generated.

% Get dimensions
nF=sum(dates<=datenum(OOSF_end))-sum(dates<=datenum(OOS_start));
H=length(horizons);
L=length(lags);
C=length(cond_options);
K=6;

% Check number of forecast dates equals number of forecasts
assert(nF==length(fdates));

% Get all possible systems (i.e. triples of series, with one from each
% group)
triples=NaN(N1*N2*N3,3);
r=1;
for i=1:N1
    for j=1:N2
        for k=1:N3
            triples(r,:)=[i,j,k];
            r=r+1;
        end
    end
end

% Break up all triples into equal-sized blocks, one for each worker
triples_i=NaN(ceil(N1*N2*N3/nworkers),3,nworkers);
removed=0;
for i=1:nworkers
    n=floor((N1*N2*N3-removed)/(nworkers+1-i));
    triples_i(1:n,:,i)=triples((removed+1):(removed+n),:);
    removed=removed+n;
end

% -------------------------------------------------------------------------
% PREALLOCATE MEMORY

% IMS forecasts
IMSF_temp3=NaN(nF,H,L,C,K,N1,N2,N3,bs_i2-bs_i1+1,nworkers);

% DMS forecasts
DMSF_temp3=IMSF_temp3;

% Actuals
act_temp3=IMSF_temp3;

% Number of lags selected
P_IMS_temp3=IMSF_temp3;
P_DMS_temp3=IMSF_temp3;

% IMS forecasts before undoing differencing
IMSF_diff_temp3=IMSF_temp3;

% IMS forecasts before undoing differencing, unconditional only
IMSF_diff_u_temp3=NaN(nF,H,L,1,K,N1,N2,N3,bs_i2-bs_i1+1,nworkers);

% Actuals before undoing differencing
act_diff_temp3=IMSF_temp3;

% Expectation of difference between conditional vs. uncondtional squared 
% forecast error
Ecvu_temp3=IMSF_temp3;

% -------------------------------------------------------------------------
% PERFORM FORECASTING EXERCISES



% Loop through bootstrapped samples; a boostrap index of 0 (i.e. bs=0)
% indicates the actual sample
for bs=bs_i1:bs_i2
    
    if run_bootstrap==1
        disp(['BOOTSTRAP ITERATION ',num2str(bs-bs_i1+1),' OF ',num2str(bs_i2-bs_i1+1)]);
    else
        disp('RUNNING FORECASTING EXERCISES');
    end
    
    rng(seeds(bs+1));
    
    % -------------------------------------------------------------------------
    % SAVE INPUT VALUES
    % Save values to structure 'input' to be passed along as an input into
    % the function that performs the forecasting exercises
    input.nF=nF;
    input.tcode1=tcode1;
    input.tcode2=tcode2;
    input.tcode3=tcode3;
    input.horizons=horizons;
    input.dates=dates;
    input.cond_options=cond_options;
    input.lags=lags;
    input.OOS_start=OOS_start;
    input.OOSF_end=OOSF_end;
    input.nobs_min=nobs_min;
    input.fredmd1_t1=fredmd1_t1;
    input.fredmd2_t1=fredmd2_t1;
    input.fredmd3_t1=fredmd3_t1;
    input.fredmd1_t2=fredmd1_t2; 
    input.fredmd2_t2=fredmd2_t2;
    input.fredmd3_t2=fredmd3_t2;
    input.triples=triples_i;
    input.group3=group3;
    input.blocksize=blocksize;
    
    % Switch for bootstrapping sample (only run if bootsrap index is
    % greater than 0)
    if bs==0
        run_bs=0;
    else
        run_bs=1;
    end
    
    % ---------------------------------------------------------------------
    % RUN FORECASTING EXERCISES
    
    % Preallocate memory for temporary arrays containing results for the
    % given bootstrap sample
    IMSF_temp2=NaN(nF,H,L,C,K,N1,N2,N3,nworkers);
    DMSF_temp2=IMSF_temp2;
    act_temp2=IMSF_temp2;
    P_IMS_temp2=IMSF_temp2;
    P_DMS_temp2=IMSF_temp2;

    IMSF_diff_temp2=IMSF_temp2;
    IMSF_diff_u_temp2=NaN(nF,H,L,1,K,N1,N2,N3,nworkers);
    act_diff_temp2=IMSF_temp2;

    Ecvu_temp2=IMSF_temp2;
    
    % Loop through each set of group pairings
    parfor w=1:nworkers
        
        % Call function that performs forecasting exercises 
        [IMSF_temp,DMSF_temp,act_temp,Pvar_all_temp,Pdms_all_temp,IMSF_diff_temp,IMSF_diff_u_temp,act_diff_temp,Ecvu_temp]=trivar_fcast(input,w,run_bs);

        % Save output
        IMSF_temp2(:,:,:,:,:,:,:,:,w)=IMSF_temp;
        DMSF_temp2(:,:,:,:,:,:,:,:,w)=DMSF_temp;
        act_temp2(:,:,:,:,:,:,:,:,w)=act_temp;
        P_IMS_temp2(:,:,:,:,:,:,:,:,w)=Pvar_all_temp;
        P_DMS_temp2(:,:,:,:,:,:,:,:,w)=Pdms_all_temp;

        IMSF_diff_temp2(:,:,:,:,:,:,:,:,w)=IMSF_diff_temp;
        IMSF_diff_u_temp2(:,:,:,:,:,:,:,:,w)=IMSF_diff_u_temp;
        act_diff_temp2(:,:,:,:,:,:,:,:,w)=act_diff_temp;

        Ecvu_temp2(:,:,:,:,:,:,:,:,w)=Ecvu_temp;

        
    end
    
    % Clear temp files
    clear IMSF_temp DMSF_temp act_temp Pvar_all_temp Pdms_all_temp...
          IMSF_diff_temp IMSF_diff_u_temp act_diff_temp Ecvu_temp...
          ARDL_IMS_f_temp P_ARDL_IMS_temp locF_temp;
    
    % ---------------------------------------------------------------------
    % SAVE RESULTS FOR BOOTSTRAP SAMPLE TO ARRAYS FOR ALL RESULTS
    
    for w=1:nworkers
        IMSF_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=IMSF_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear IMSF_temp2;
    
    for w=1:nworkers
        DMSF_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=DMSF_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear DMSF_temp2
    
    for w=1:nworkers
        act_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=act_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear act_temp2;
    
    for w=1:nworkers
        P_IMS_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=P_IMS_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear P_IMS_temp2;
    
    for w=1:nworkers
        P_DMS_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=P_DMS_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear P_DMS_temp2;
    
    for w=1:nworkers
        IMSF_diff_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=IMSF_diff_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear IMSF_diff_temp2;
    
    for w=1:nworkers
        IMSF_diff_u_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=IMSF_diff_u_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear IMSF_diff_u_temp2;
    
    for w=1:nworkers
        act_diff_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=act_diff_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear act_diff_temp2;

    for w=1:nworkers
        Ecvu_temp3(:,:,:,:,:,:,:,:,bs-bs_i1+1,w)=Ecvu_temp2(:,:,:,:,:,:,:,:,w);
    end
    clear Ecvu_temp2;

    
end

% =========================================================================
% PART 3: SAVE OUTPUT


disp('SAVING RESULTS');

% -------------------------------------------------------------------------
% PREALLOCATE MEMORY FOR NEW ARRAYS

IMSF=NaN(nF,H,L,C,K,N1,N2,N3,bs_i2-bs_i1+1);
DMSF=IMSF;
act=IMSF;
P_IMS=IMSF;
P_DMS=IMSF;

IMSF_diff=IMSF;
IMSF_diff_u=NaN(nF,H,L,1,K,N1,N2,N3,bs_i2-bs_i1+1);
act_diff=IMSF;

Ecvu=IMSF;

% -------------------------------------------------------------------------
% REARRANGE RESULTS FROM OLD ARRAYS INTO NEW ONES

for w=1:nworkers
    triples_w=triples_i(:,:,w);
    triples_w=triples_w(~isnan(triples_w(:,1)),:);
    
    for grp=1:size(triples_w,1)
    
        i1=triples_w(grp,1);
        i2=triples_w(grp,2);
        i3=triples_w(grp,3);
        
        for j=1:(bs_i2-bs_i1+1)
            IMSF(:,:,:,:,:,i1,i2,i3,j)=IMSF_temp3(:,:,:,:,:,i1,i2,i3,j,w);
            DMSF(:,:,:,:,:,i1,i2,i3,j)=DMSF_temp3(:,:,:,:,:,i1,i2,i3,j,w);
            act(:,:,:,:,:,i1,i2,i3,j)=act_temp3(:,:,:,:,:,i1,i2,i3,j,w);

            P_IMS(:,:,:,:,:,i1,i2,i3,j)=P_IMS_temp3(:,:,:,:,:,i1,i2,i3,j,w);
            P_DMS(:,:,:,:,:,i1,i2,i3,j)=P_DMS_temp3(:,:,:,:,:,i1,i2,i3,j,w);

            IMSF_diff(:,:,:,:,:,i1,i2,i3,j)=IMSF_diff_temp3(:,:,:,:,:,i1,i2,i3,j,w);
            IMSF_diff_u(:,:,:,:,:,i1,i2,i3,j)=IMSF_diff_u_temp3(:,:,:,:,:,i1,i2,i3,j,w);
            act_diff(:,:,:,:,:,i1,i2,i3,j)=act_diff_temp3(:,:,:,:,:,i1,i2,i3,j,w);

            Ecvu(:,:,:,:,:,i1,i2,i3,j)=Ecvu_temp3(:,:,:,:,:,i1,i2,i3,j,w);
            
        end
    end
end

% -------------------------------------------------------------------------
% SAVE OUTPUT TO STRUCTURE


S.IMSF=IMSF;
S.DMSF=DMSF;
S.act=act;
S.P_IMS=P_IMS;
S.P_DMS=P_DMS;
S.IMSF_diff=IMSF_diff;
S.IMSF_diff_u=IMSF_diff_u;
S.act_diff=act_diff;
S.Ecvu=Ecvu;

S.fdates=fdates;
S.horizons=horizons;
S.cond_options=cond_options;
S.sample_start=sample_start;
S.lags=lags;

S.bs_i1=bs_i1;
S.bs_i2=bs_i2;

% -------------------------------------------------------------------------
% SAVE STRUCTURE AS .MAT FILE



% .mat file name
switch sample_start
    case eom('1/31/1959')
        yr=59;
    case eom('1/31/1984')
        yr=84;
end

switch msw_sample
    case 0
        sample='03-16';
    case 1
        sample='MSW';
end

matfilename=[outfolder,'\trivar_start',num2str(yr)];
if run_bootstrap==1
    matfilename=[matfilename,'_',sample,'_comp',num2str(comp),'of',num2str(ncomp)];
end

save(matfilename,'S','-v7.3');


