clear
close all
clc

% =========================================================================
% DESCRIPTION
% This script loads and organizes the trivariate bootstrap results.
%
% -------------------------------------------------------------------------
% USER-DEFINED VARIABLES

% Switch determining whether to demean bootstrapped t-stats
demean=1;

% Total number of computers used to create the bootstrap results (i.e. the
% number of sections the bootstraps are broken into)
ncomp=6;

% Significance level for hypothesis testing
sig_lev=0.05;

% Folder containing auxiliary functions for organizing data
auxfcn='FcnsOrg';

% Folder containing LeSage Toolbox functions
lesage='FcnsLeSage';

% Folder containing .mat files with raw ouputs from the forecasting
% exercises
rawoutput='RawOutput';

% Name of Excel file and sheet to which Table 8 is written.
outfile='FinalOutput\Tables.xlsx';
outsheet='Table9';

% =========================================================================

addpath(auxfcn);
addpath(lesage);

% Preallocate memory for Table 3 and Table 6 from the paper
T9=NaN(3*3*4,3*3*2);

% Establish sample number
sn=0;

% Loop through great moderation data
for great_mod=0:1
    
sn=sn+1;

disp(['Sample-Split ',num2str(sn)]);
    
% -------------------------------------------------------------------------
% DEFINE IN-SAMPLE AND OUT-OF-SAMPLE PERIODS
switch great_mod
    case 0
        sample_start='1/31/1959';
        OOS_start='1/31/1979';
        msw_sample=1;
    case 1
        sample_start='1/31/1984';
        OOS_start='12/31/2002';
        msw_sample=0;
    otherwise
        error('The switch great_mod is misspecified.');
end

switch msw_sample
    case 0
        OOSF_end='12/31/2016';
    case 1
        OOSF_end='12/31/2002';
    otherwise
        error('The switch msw_sample is misspecified.');
end

% =========================================================================

% Get .mat file name base (either 'trivar_start59' or 'trivar_start84'
% depending on whether or not using great moderation dataset)
switch great_mod
    case 0
        yr=59;
    case 1
        yr=84;
end
switch msw_sample
    case 0
        sample='03-16';
    case 1
        sample='MSW';
end
filename_base=[rawoutput,'\trivar_start',num2str(yr),'_',sample];

fprintf('  Loading data \n');

% Combine results from separate saved files
for comp=1:ncomp
    
    fprintf(['    File ',num2str(comp),' of ',num2str(ncomp),' \n']);
    
    % Get .mat file name 
    filename=[filename_base,'_comp',num2str(comp),'of',num2str(ncomp)];
    
    % Load .mat file
    load(filename);
    
    % Combine arrays
    if comp==1
        IMSF_diff=S.IMSF_diff;
        IMSF_diff_u=S.IMSF_diff_u;
        act_diff=S.act_diff;
        Ecvu=S.Ecvu;
    else
        nbs1=size(IMSF_diff,9)+1;
        nbs2=size(IMSF_diff,9)+size(S.IMSF_diff,9);
        IMSF_diff(:,:,:,:,:,:,:,:,nbs1:nbs2)=S.IMSF_diff;
        IMSF_diff_u(:,:,:,:,:,:,:,:,nbs1:nbs2)=S.IMSF_diff_u;
        act_diff(:,:,:,:,:,:,:,:,nbs1:nbs2)=S.act_diff;
        Ecvu(:,:,:,:,:,:,:,:,nbs1:nbs2)=S.Ecvu;
    end
    
    fdates=S.fdates;
    horizons=S.horizons;
    cond_options=S.cond_options;
    sample_start=S.sample_start;
    
    clear S;
    
end

%%
fprintf('  Organizing results \n');

[T,H,L,C,K,N1,N2,N3,nbs]=size(IMSF_diff);
nbs=nbs-1;


OOS_start=eom(OOS_start);
OOSF_end=eom(OOSF_end);
  

% Note:
%   model 1 = VAR-based IMS

% Forecasts (before undoing transformation; 'd' stands for differenced)
F1d=IMSF_diff;

% Unconditional forecasts (before undoing transformation)
UF1d=IMSF_diff_u;

% Actual values (before undoing transformation)
Ad=act_diff;

% Value of E for MSE-test
E=Ecvu(end,:,:,:,:,:,:,:,:);

% Get statistics
[ETtstat_a,ETtstat_b]=efficiency_test(UF1d,Ad,0); % bias and efficiency test
MSEtstat=MSEtest(F1d,UF1d,E,Ad,cond_options,0); % MSE test

% Checks
assert(size(ETtstat_a,1)==1);
assert(size(ETtstat_b,1)==1);
assert(size(MSEtstat,1)==1);

% Preallocate memory for critical values (upper and lower)
critL_ETa=NaN(1,H,L,C,K,N1,N2,N3);
critU_ETa=NaN(1,H,L,C,K,N1,N2,N3);

critL_ETb=NaN(1,H,L,C,K,N1,N2,N3);
critU_ETb=NaN(1,H,L,C,K,N1,N2,N3);

critL_MSET=NaN(1,H,L,C,K,N1,N2,N3);
critU_MSET=NaN(1,H,L,C,K,N1,N2,N3);

% Get critical values
for h=1:H
  for l=1:L
    for c=1:C      
      for k=1:K
        for i1=1:N1
          for i2=1:N2
            for i3=1:N3
              
              % Efficiency test only involves unconditional forecasts so
              % ignore results from conditioning on individuals series
              % because those are all missing for the unconditional results
              if k<=(K/2)
                
                ETa_i=ETtstat_a(1,h,l,c,k,i1,i2,i3,2:end);
                ETa_i_col=ETa_i(:);
                if demean==1
                    ETa_i_col=ETa_i_col-mean(ETa_i_col);
                end
                [Li,Ui]=getCritVal(ETa_i_col,sig_lev);
                critL_ETa(1,h,l,c,k,i1,i2,i3)=Li;
                critU_ETa(1,h,l,c,k,i1,i2,i3)=Ui;
                
                
                ETb_i=ETtstat_b(1,h,l,c,k,i1,i2,i3,2:end);
                ETb_i_col=ETb_i(:);
                if demean==1
                    ETb_i_col=ETb_i_col-mean(ETb_i_col);
                end
                [Li,Ui]=getCritVal(ETb_i_col,sig_lev);
                critL_ETb(1,h,l,c,k,i1,i2,i3)=Li;
                critU_ETb(1,h,l,c,k,i1,i2,i3)=Ui;
                
              end
              
              MSET_i=MSEtstat(1,h,l,c,k,i1,i2,i3,2:end);
              MSET_i_col=MSET_i(:);
              if demean==1
                MSET_i_col=MSET_i_col-mean(MSET_i_col);
              end
              [Li,Ui]=getCritVal(MSET_i_col,sig_lev);
              critL_MSET(1,h,l,c,k,i1,i2,i3)=Li;
              critU_MSET(1,h,l,c,k,i1,i2,i3)=Ui;
              
            end
          end
        end
      end
    end
  end
end
   
% Identify rejections
rejETa=ETtstat_a(:,:,:,:,:,:,:,:,1)<critL_ETa | ...
        ETtstat_a(:,:,:,:,:,:,:,:,1)>critU_ETa;
    
rejETb=ETtstat_b(:,:,:,:,:,:,:,:,1)<critL_ETb | ...
        ETtstat_b(:,:,:,:,:,:,:,:,1)>critU_ETb;
    
rejMSET=MSEtstat(:,:,:,:,:,:,:,:,1)<critL_MSET | ...
        MSEtstat(:,:,:,:,:,:,:,:,1)>critU_MSET;

% Organize results
[T5_ETa,T5g_ETa]=getT5_tstat_trivar(ETtstat_a(:,:,:,:,:,:,:,:,1),rejETa,[],{'none'});
[T5_ETb,T5g_ETb]=getT5_tstat_trivar(ETtstat_b(:,:,:,:,:,:,:,:,1),rejETb,[],{'none'});
[T5_MSET,T5g_MSET]=getT5_tstat_trivar(MSEtstat(:,:,:,:,:,:,:,:,1),rejMSET,[],cond_options);

for l=1:L
    for h_idx=1:H
        T9(1+(l-1)*3,h_idx+(sn-1)*length(horizons)*3)=T5_ETa(1+(l-1)*8,h_idx,:);
        T9(2+(l-1)*3,h_idx+(sn-1)*length(horizons)*3)=T5_ETa(4+(l-1)*8,h_idx,:);
        T9(3+(l-1)*3,h_idx+(sn-1)*length(horizons)*3)=100*T5_ETa(7+(l-1)*8,h_idx,:)/T5_ETa(l*8,h_idx,:);
        
        T9(1+(l-1)*3,h_idx+length(horizons)+(sn-1)*length(horizons)*3)=T5_ETb(1+(l-1)*8,h_idx,:);
        T9(2+(l-1)*3,h_idx+length(horizons)+(sn-1)*length(horizons)*3)=T5_ETb(4+(l-1)*8,h_idx,:);
        T9(3+(l-1)*3,h_idx+length(horizons)+(sn-1)*length(horizons)*3)=100*T5_ETb(7+(l-1)*8,h_idx,:)/T5_ETb(l*8,h_idx,:);
            
        T9(1+(l-1)*3,h_idx+2*length(horizons)+(sn-1)*length(horizons)*3)=T5_MSET(1+(l-1)*8,h_idx,strcmp(cond_options,'full1'));
        T9(2+(l-1)*3,h_idx+2*length(horizons)+(sn-1)*length(horizons)*3)=T5_MSET(4+(l-1)*8,h_idx,strcmp(cond_options,'full1'));
        T9(3+(l-1)*3,h_idx+2*length(horizons)+(sn-1)*length(horizons)*3)=100*T5_MSET(7+(l-1)*8,h_idx,:)/T5_MSET(l*8,h_idx,strcmp(cond_options,'full1'));
        
        for g=1:3
        
            T9(1+(l-1)*3+g*L*3,h_idx+(sn-1)*length(horizons)*3)=T5g_ETa(1+(l-1)*8,h_idx,:,g);
            T9(2+(l-1)*3+g*L*3,h_idx+(sn-1)*length(horizons)*3)=T5g_ETa(4+(l-1)*8,h_idx,:,g);
            T9(3+(l-1)*3+g*L*3,h_idx+(sn-1)*length(horizons)*3)=100*T5g_ETa(7+(l-1)*8,h_idx,:,g)/T5g_ETa(l*8,h_idx,:,g);

            T9(1+(l-1)*3+g*L*3,h_idx+length(horizons)+(sn-1)*length(horizons)*3)=T5g_ETb(1+(l-1)*8,h_idx,:,g);
            T9(2+(l-1)*3+g*L*3,h_idx+length(horizons)+(sn-1)*length(horizons)*3)=T5g_ETb(4+(l-1)*8,h_idx,:,g);
            T9(3+(l-1)*3+g*L*3,h_idx+length(horizons)+(sn-1)*length(horizons)*3)=100*T5g_ETb(7+(l-1)*8,h_idx,:,g)/T5g_ETb(l*8,h_idx,:,g);

            T9(1+(l-1)*3+g*L*3,h_idx+2*length(horizons)+(sn-1)*length(horizons)*3)=T5g_MSET(1+(l-1)*8,h_idx,strcmp(cond_options,'full1'),g);
            T9(2+(l-1)*3+g*L*3,h_idx+2*length(horizons)+(sn-1)*length(horizons)*3)=T5g_MSET(4+(l-1)*8,h_idx,strcmp(cond_options,'full1'),g);
            T9(3+(l-1)*3+g*L*3,h_idx+2*length(horizons)+(sn-1)*length(horizons)*3)=100*T5g_MSET(7+(l-1)*8,h_idx,:,g)/T5g_MSET(l*8,h_idx,strcmp(cond_options,'full1'),g);
        end
        
    end
end


end

disp('Formatting Table 9');

Table9=cell(size(T9,1)+3,size(T9,2)+2);
Table9(4:end,3:end)=num2cell(T9);
lag_labels=cell(L*3,1);
lag_labels{1}='AR(2)';
lag_labels{4}='AR(4)';
lag_labels{7}='AR(12)';
Table9(4:end,1)=repmat(lag_labels,4,1);
Table9(4:end,2)=repmat({'Mean';'Median';'Rejected'},4*L,1);
Table9(3,3:end)=repmat(num2cell(horizons),1,6);
test_labels=cell(1,3*length(horizons));
test_labels{1}='Bias';
test_labels{1+length(horizons)}='Efficiency';
test_labels{1+2*length(horizons)}='Equal MSE';
Table9(2,3:end)=repmat(test_labels,1,2);
Table9{1,3}='Full/ MSW';
Table9{1,3+3*length(horizons)}='Great Moderation/ 2003-2016';
Table9{1,1}='Data/ Sample:';
Table9{2,1}='Test:';
Table9{3,1}='Forecast Horizon:';
Table9=[Table9(1:3,:);...
        cell(1,size(Table9,2));...
        Table9(4:(3+3*L),:);...
        cell(1,size(Table9,2));...
        Table9((4+3*L):(3+2*3*L),:);...
        cell(1,size(Table9,2));...
        Table9((4+2*3*L):(3+3*3*L),:);...
        cell(1,size(Table9,2));...
        Table9((4+3*3*L):end,:)];
Table9{4,1}='(A) All';
Table9{5+3*L,1}='(B) Real';
Table9{6+2*3*L,1}='(C) Nominal';
Table9{7+3*3*L,1}='(D) Financial';

% -------------------------------------------------------------------------
% WRITE RESULTS TO EXCEL

disp('Exporting tables to Excel');
xlswrite(outfile,Table9,outsheet,'B2');

disp('Finished');


