clear
close all
clc

% =========================================================================
% DESCRIPTION
% This script organizes and writes the trivariate output to Excel (but not
% the bootstrapped results).
%
% -------------------------------------------------------------------------
% USER-DEFINED VARIABLES

% Folder containing auxiliary functions for organizing data
auxfcn='FcnsOrg';

% Folder containing LeSage Toolbox functions
lesage='FcnsLeSage';

% Folder containing .mat files with raw ouputs from the forecasting
% exercises
rawoutput='RawOutput';

% Name of Excel file and sheet to which Table 8 is written.
outfile='FinalOutput\Tables.xlsx';
outsheet4='Table4';
outsheet5='Table5';
outsheet7='Table7';

% =========================================================================
% SETUP

addpath(auxfcn);
addpath(lesage);

% Preallocate memory for Table 3 and Table 6 from the paper
T4_long=NaN(16*4,3*3);
T7_long=NaN(16*4,3*3);

% Establish sample number
sn=0;

% Counter for number of structures
counter=0;

% Loop through great moderation data
for great_mod=0:1
    
counter=counter+1;
disp(['Structure ',num2str(counter)]);

% -------------------------------------------------------------------------
% DEFINE IN-SAMPLE AND OUT-OF-SAMPLE PERIODS

% Out-of-sample start and end dates
switch great_mod
    case 0
        OOS_start=[eom('1/31/1979');eom('12/31/2002')];
        OOSF_end=[eom('12/31/2002');eom('12/31/2016')];
    case 1
        OOS_start=eom('12/31/2002');
        OOSF_end=eom('12/31/2016');
    otherwise
        error('The switch great_mod is misspecified.');
end

% Number of samples
n_samples=length(OOS_start);

% -------------------------------------------------------------------------
% LOAD RELEVANT .MAT FILE

disp('  Loading Data');

switch great_mod
    case 0
        load([rawoutput,'\trivar_start59.mat']);
    case 1
        load([rawoutput,'\trivar_start84.mat']);
    otherwise
        error('The switch great_mod is misspecified.');
end

% -------------------------------------------------------------------------
% CHECKS

% Bootstrap indices are 0
assert(S.bs_i1==0);
assert(S.bs_i2==0);

% -------------------------------------------------------------------------
% RENAME VARIABLES FROM OUTPUT STRUCTURE
% Makes it easier to identify them within the code

% Forecasts, actuals, lags, and Ecvu
IMSF=S.IMSF; % IMS forecasts 
DMSF=S.DMSF; % DMS forecasts
act=S.act; % Actuals

% User-defined variables
fdates=S.fdates; % Forecast dates
horizons=S.horizons; % Horizons
cond_options=S.cond_options; % Conditional options
sample_start=S.sample_start; % Sample start
lags=S.lags;

% Clear structure
clear S;

% =========================================================================
% ORGANIZE AND SAVE RESULTS FOR EACH SAMPLE-SPLIT CONSIDERED

% Loop through sample splits being considered
for i=1:n_samples
    
    disp(['  Organizing Output for Sample-Split ',num2str(i),' of ',num2str(n_samples)]);

    % Increase sample number by 1
    sn=sn+1;
    
    % ---------------------------------------------------------------------
    % KEEP DESIRED OUT-OF-SAMPLE FORECASTS
    % Methods:
    %   Method 1 = VAR-based IMS
    %   Method 2 = ARDL-based DMS
    % Variable names
    %   F# - forecasts for method #
    %   A - actuals
    
    disp('    Removing unnecessary forecasts');

    % Remove observations to reflect the desired sample split
    F1=adjust_sample_split(IMSF,fdates,horizons,OOS_start(i),OOSF_end(i));
    F2=adjust_sample_split(DMSF,fdates,horizons,OOS_start(i),OOSF_end(i));
    A=adjust_sample_split(act,fdates,horizons,OOS_start(i),OOSF_end(i));
    
    % ---------------------------------------------------------------------
    % CALCULATE DESIRED STATISTICS
    
    disp('    Calculating desired statistics');
    
    % MSE ratios, t-stats from MSE pairwise comparisons, and dummies
    % identifying which comparisons are statistically significant in favor
    % of each model
    [MSEr_2v1,MSEr_1vB,MSEr_2vB,PCtstat_2v1,better_1v2,better_2v1]=get_stats(F1,F2,A,horizons);
    
    % ---------------------------------------------------------------------
    % DISTRIBUTIONS
    
    disp('    Calculating moments of each statistic''s distribution');
    
    % Distribution of MSE ratios between models
    [T5_MSEr_2v1,T5g_MSEr_2v1]=getT5_trivar(MSEr_2v1,cond_options);
    
    % Distribution of t-stats from tests of equal MSEs
    [T5_PC_2v1,T5g_PC_2v1]=getT5_tstat_trivar(PCtstat_2v1,better_1v2,better_2v1,cond_options);

    % Distribution of MSE ratios of models to base model (first lag structure
    % of model 1)
    [T6_MSEr,T6g_MSEr]=getT6_trivar(MSEr_1vB,MSEr_2vB,cond_options);
    
    % ---------------------------------------------------------------------
    % ORGANIZE OUTPUT MATRICES
    
    disp('  Formatting results into output tables');
    
    % Table 4, long format
    for l=1:length(lags)
        for h_idx=1:length(horizons)
            
            T4_long(1+(l-1)*4,h_idx+(sn-1)*length(horizons))=T5_MSEr_2v1(1+(l-1)*6,h_idx,strcmp(cond_options,'full1'));
            T4_long(2+(l-1)*4,h_idx+(sn-1)*length(horizons))=T5_MSEr_2v1(4+(l-1)*6,h_idx,strcmp(cond_options,'full1'));
            
            T4_long(3+(l-1)*4,h_idx+(sn-1)*length(horizons))=100*T5_PC_2v1(7+(l-1)*9,h_idx,strcmp(cond_options,'full1'))/...
                                                       T5_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'full1'));
            T4_long(4+(l-1)*4,h_idx+(sn-1)*length(horizons))=100*T5_PC_2v1(8+(l-1)*9,h_idx,strcmp(cond_options,'full1'))/...
                                                       T5_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'full1'));                                        
            
            for g=1:3
                T4_long(1+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_2v1(1+(l-1)*6,h_idx,strcmp(cond_options,'full1'),g);
                T4_long(2+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_2v1(4+(l-1)*6,h_idx,strcmp(cond_options,'full1'),g);
                
                T4_long(3+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=100*T5g_PC_2v1(7+(l-1)*9,h_idx,strcmp(cond_options,'full1'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'full1'),g);
                T4_long(4+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=100*T5g_PC_2v1(8+(l-1)*9,h_idx,strcmp(cond_options,'full1'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'full1'),g);
            end
        end
    end
    
    % Table 5, long format
    if sn==1 
        T5=[T6_MSEr(:,:,strcmp(cond_options,'full1'));...
                 T6g_MSEr(:,:,strcmp(cond_options,'full1'),1);...
                 T6g_MSEr(:,:,strcmp(cond_options,'full1'),2);...
                 T6g_MSEr(:,:,strcmp(cond_options,'full1'),3)];
    elseif sn==3
        T5=[T5, [T6_MSEr(:,:,strcmp(cond_options,'full1'));...
                 T6g_MSEr(:,:,strcmp(cond_options,'full1'),1);...
                 T6g_MSEr(:,:,strcmp(cond_options,'full1'),2);...
                 T6g_MSEr(:,:,strcmp(cond_options,'full1'),3)]];
    end
    
    % Table 7, long format
    for l=1:length(lags)
        for h_idx=1:length(horizons)
            
            T7_long(1+(l-1)*4,h_idx+(sn-1)*length(horizons))=T5_MSEr_2v1(1+(l-1)*6,h_idx,strcmp(cond_options,'none'));
            T7_long(2+(l-1)*4,h_idx+(sn-1)*length(horizons))=T5_MSEr_2v1(4+(l-1)*6,h_idx,strcmp(cond_options,'none'));
            
            T7_long(3+(l-1)*4,h_idx+(sn-1)*length(horizons))=100*T5_PC_2v1(7+(l-1)*9,h_idx,strcmp(cond_options,'none'))/...
                                                       T5_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'none'));
            T7_long(4+(l-1)*4,h_idx+(sn-1)*length(horizons))=100*T5_PC_2v1(8+(l-1)*9,h_idx,strcmp(cond_options,'none'))/...
                                                       T5_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'none'));                                        
            
            for g=1:3
                T7_long(1+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_2v1(1+(l-1)*6,h_idx,strcmp(cond_options,'none'),g);
                T7_long(2+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_2v1(4+(l-1)*6,h_idx,strcmp(cond_options,'none'),g);
                
                T7_long(3+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=100*T5g_PC_2v1(7+(l-1)*9,h_idx,strcmp(cond_options,'none'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'none'),g);
                T7_long(4+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(horizons))=100*T5g_PC_2v1(8+(l-1)*9,h_idx,strcmp(cond_options,'none'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,h_idx,strcmp(cond_options,'none'),g);
            end
        end
    end


end

end

% -------------------------------------------------------------------------
% FORMAT TABLES

disp('Formatting tables');

T4=[[T4_long(1:(4*length(lags)),:),T4_long((4*length(lags)+1):(2*4*length(lags)),:)];...
       [T4_long((2*4*length(lags)+1):(3*4*length(lags)),:),T4_long((3*4*length(lags)+1):(4*4*length(lags)),:)]];
   
T7=[[T7_long(1:(4*length(lags)),:),T7_long((4*length(lags)+1):(2*4*length(lags)),:)];...
       [T7_long((2*4*length(lags)+1):(3*4*length(lags)),:),T7_long((3*4*length(lags)+1):(4*4*length(lags)),:)]];

Table4=cell(size(T4,1)+3,size(T4,2)+2);
Table4(4:end,3:end)=num2cell(T4);
for i=1:length(lags)
    if isnumeric(lags{i})
        lags_i=['AR(',num2str(lags{i}),')'];
    else
        lags_i=lags{i};
    end
    Table4{4+4*(i-1),1}=lags_i;
    Table4{4+4*length(lags)+4*(i-1),1}=lags_i;
end
Table4(4:end,2)=repmat({'Mean';'Median';'VAR Better';'ARDL Better'},2*length(lags),1);
Table4{1,1}='Data:';
Table4{2,1}='Sample:';
Table4{3,1}='Forecast Horizon:';
Table4(3,3:end)=num2cell(repmat(horizons,1,2*sn));
for i=1:2
    Table4{1,3+(i-1)*length(horizons)*3}='Full';
    Table4{1,3+length(horizons)+(i-1)*length(horizons)*3}='Full';
    Table4{1,3+2*length(horizons)+(i-1)*length(horizons)*3}='Great Moderation';
    
    Table4{2,3+(i-1)*length(horizons)*3}='MSW';
    Table4{2,3+length(horizons)+(i-1)*length(horizons)*3}='2003-2016';
    Table4{2,3+2*length(horizons)+(i-1)*length(horizons)*3}='2003-2016';  
    
end
Table4=[Table4(1:3,:);cell(1,size(Table4,2));Table4(4:(3+4*length(lags)),:);cell(1,size(Table4,2));Table4((4+4*length(lags)):end,:)];
Table4{4,3}='(A) All variables';
Table4{4,3+length(horizons)*sn}='(B) Real variables';
Table4{5+length(lags)*4,3}='(C) Nominal variables';
Table4{5+length(lags)*4,3+length(horizons)*sn}='(D) Financial variables';


Table5=cell(size(T5,1)+3,size(T5,2)+2);
Table5(4:end,3:end)=num2cell(T5);
for i=1:size(Table5,1)
    for j=1:size(Table5,2)
        if isnan(Table5{i,j})
            Table5{i,j}=[];
        end
    end
end
for i=1:length(horizons)
    for j=1:4
        Table5{4+(i-1)*3+(j-1)*length(horizons)*3,1}=horizons(i);
    end
end
Table5(4:end,2)=repmat({'Mean';'Median';'Fraction best'},4*length(horizons),1);
lag_labels=cell(1,length(lags)+1);
for i=1:length(lags)
    if isnumeric(lags{i})
        lag_labels{i}=['AR(',num2str(lags{i}),')'];
    else
        lag_labels{i}=lags{i};
    end
end
lag_labels{end}='Sum';
Table5(3,3:end)=repmat(lag_labels,1,2*2);
method_labels=cell(1,10);
method_labels{1}='Iterated forecasts';
method_labels{6}='Direct forecasts';
Table5(2,3:end)=repmat(method_labels,1,2);
Table5{1,3}='Full Data, MSW Sample';
Table5{1,3+10}='Great Moderation Data, 2003-2016 Sample';
Table5{2,1}='Horizon';
Table5{2,2}='Summary variable';
Table5=[Table5(1:3,:);...
        cell(1,size(Table5,2));...
        Table5(4:(3+3*length(horizons)),:);...
        cell(1,size(Table5,2));...
        Table5((4+3*length(horizons)):(3+2*3*length(horizons)),:);...
        cell(1,size(Table5,2));...
        Table5((4+2*3*length(horizons)):(3+3*3*length(horizons)),:);...
        cell(1,size(Table5,2));...
        Table5((4+3*3*length(horizons)):end,:)];
Table5{4,1}='(A) All';
Table5{5+3*length(horizons),1}='(B) Real';
Table5{6+2*3*length(horizons),1}='(C) Nominal';
Table5{7+3*3*length(horizons),1}='(D) Financial';




Table7=cell(size(T7,1)+3,size(T7,2)+2);
Table7(4:end,3:end)=num2cell(T7);
for i=1:length(lags)
    if isnumeric(lags{i})
        lags_i=['AR(',num2str(lags{i}),')'];
    else
        lags_i=lags{i};
    end
    Table7{4+4*(i-1),1}=lags_i;
    Table7{4+4*length(lags)+4*(i-1),1}=lags_i;
end
Table7(4:end,2)=repmat({'Mean';'Median';'VAR Better';'ARDL Better'},2*length(lags),1);
Table7{1,1}='Data:';
Table7{2,1}='Sample:';
Table7{3,1}='Forecast Horizon:';
Table7(3,3:end)=num2cell(repmat(horizons,1,2*sn));
for i=1:2
    Table7{1,3+(i-1)*length(horizons)*3}='Full';
    Table7{1,3+length(horizons)+(i-1)*length(horizons)*3}='Full';
    Table7{1,3+2*length(horizons)+(i-1)*length(horizons)*3}='Great Moderation';
    
    Table7{2,3+(i-1)*length(horizons)*3}='MSW';
    Table7{2,3+length(horizons)+(i-1)*length(horizons)*3}='2003-2016';
    Table7{2,3+2*length(horizons)+(i-1)*length(horizons)*3}='2003-2016';  
    
end
Table7=[Table7(1:3,:);cell(1,size(Table7,2));Table7(4:(3+4*length(lags)),:);cell(1,size(Table7,2));Table7((4+4*length(lags)):end,:)];
Table7{4,3}='(A) All variables';
Table7{4,3+length(horizons)*sn}='(B) Real variables';
Table7{5+length(lags)*4,3}='(C) Nominal variables';
Table7{5+length(lags)*4,3+length(horizons)*sn}='(D) Financial variables';
   
   
% -------------------------------------------------------------------------
% WRITE RESULTS TO EXCEL

disp('Exporting tables to Excel');
xlswrite(outfile,Table4,outsheet4,'B2');
xlswrite(outfile,Table5,outsheet5,'B2');
xlswrite(outfile,Table7,outsheet7,'B2');

disp('Finished');
