clear
close all
clc

% =========================================================================
% DESCRIPTION
% This script organizes and writes the large models output (Table 8 from
% the paper) to Excel.
%
% -------------------------------------------------------------------------
% USER DEFINED VARIABLES

% Folder containing auxiliary functions for organizing data
auxfcn='FcnsOrg';

% Folder containing LeSage Toolbox functions
lesage='FcnsLeSage';

% Folder containing .mat files with raw ouputs from the forecasting
% exercises
rawoutput='RawOutput';

% Name of Excel file and sheet to which Table 8 is written.
outfile='FinalOutput\Tables.xlsx';
outsheet='Table8';

% =========================================================================
% SETUP

addpath(auxfcn);
addpath(lesage);

% Establish sample number
sn=0;

% Preallocate memory for Table 8 from paper
T8_long=NaN(4*3*6,2*3);


% Counter for number of structures
counter=0;

for great_mod=0:1
 
counter=counter+1;
disp(['Structure ',num2str(counter)]);
    
% -------------------------------------------------------------------------
% DEFINE IN-SAMPLE AND OUT-OF-SAMPLE PERIODS


switch great_mod
    case 0
        OOS_start=eom('1/31/1979');
        OOSF_end=eom('12/31/2002');
    case 1
        OOS_start=eom('12/31/2002');
        OOSF_end=eom('12/31/2016');
    otherwise
        error('The switch great_mod is misspecified.');
end

% Number of samples
n_samples=length(OOS_start);



% -------------------------------------------------------------------------
% LOAD RELEVANT .MAT FILE

disp('  Loading Data');

switch great_mod
    case 0
        load([rawoutput,'\largevar_start59.mat']);
    case 1
        load([rawoutput,'\largevar_start84.mat']);
    otherwise
        error('The switch great_mod is misspecified.');
end

% -------------------------------------------------------------------------
% RENAME VARIABLES FROM OUTPUT STRUCTURE
% Makes it easier to identify them within the code

fdates=S.fdates;
horizons=S.horizons;
cond_options=S.cond_options;
sample_start=S.sample_start;
lags=S.lags;

IMSF3=S.IMSF3;
DMSF3=S.DMSF3;
act3=S.act3;
s3=S.s3;

IMSF4=S.IMSF4;
DMSF4=S.DMSF4;
act4=S.act4;
s4=S.s4;

IMSF5=S.IMSF5;
DMSF5=S.DMSF5;
act5=S.act5;
s5=S.s5;

IMSF6=S.IMSF6;
DMSF6=S.DMSF6;
act6=S.act6;
s6=S.s6;

% Clear structure
clear S;




% =========================================================================
% ORGANIZE AND SAVE RESULTS FOR EACH SAMPLE-SPLIT CONSIDERED

% Loop through sample splits being considered
for i=1:n_samples
    
    disp(['  Organizing Output for Sample-Split ',num2str(i),' of ',num2str(n_samples)]);

    % Increase sample number by 1
    sn=sn+1;
    
    % ---------------------------------------------------------------------
    % KEEP DESIRED OUT-OF-SAMPLE FORECASTS
    % Methods:
    %   Method 1 = VAR-based IMS
    %   Method 2 = ARDL-based DMS
    %   Method 3 = ARDL-based IMS
    %   Method 4 = location model
    % Variable names
    %   F# - forecasts for method #
    %   A - actuals
    
    disp('    Removing unnecessary forecasts');

    % Remove observations to reflect the desired sample split
    F1_s3=adjust_sample_split(IMSF3,fdates,horizons,OOS_start(i),OOSF_end(i));
    F2_s3=adjust_sample_split(DMSF3,fdates,horizons,OOS_start(i),OOSF_end(i));
    A_s3=adjust_sample_split(act3,fdates,horizons,OOS_start(i),OOSF_end(i));
    
    F1_s4=adjust_sample_split(IMSF4,fdates,horizons,OOS_start(i),OOSF_end(i));
    F2_s4=adjust_sample_split(DMSF4,fdates,horizons,OOS_start(i),OOSF_end(i));
    A_s4=adjust_sample_split(act4,fdates,horizons,OOS_start(i),OOSF_end(i));
    
    F1_s5=adjust_sample_split(IMSF5,fdates,horizons,OOS_start(i),OOSF_end(i));
    F2_s5=adjust_sample_split(DMSF5,fdates,horizons,OOS_start(i),OOSF_end(i));
    A_s5=adjust_sample_split(act5,fdates,horizons,OOS_start(i),OOSF_end(i));
    
    F1_s6=adjust_sample_split(IMSF6,fdates,horizons,OOS_start(i),OOSF_end(i));
    F2_s6=adjust_sample_split(DMSF6,fdates,horizons,OOS_start(i),OOSF_end(i));
    A_s6=adjust_sample_split(act6,fdates,horizons,OOS_start(i),OOSF_end(i));
    
    % ---------------------------------------------------------------------
    % CALCULATE DESIRED STATISTICS
    
    disp('    Calculating relative MSEs');
    
    % MSE ratios for each system size   
    [MSEr_s3_2v1,~,~,~,~,~]=get_stats(F1_s3,F2_s3,A_s3,horizons);
    [MSEr_s4_2v1,~,~,~,~,~]=get_stats(F1_s4,F2_s4,A_s4,horizons);
    [MSEr_s5_2v1,~,~,~,~,~]=get_stats(F1_s5,F2_s5,A_s5,horizons);
    [MSEr_s6_2v1,~,~,~,~,~]=get_stats(F1_s6,F2_s6,A_s6,horizons);

    
    % ---------------------------------------------------------------------
    % DISTRIBUTIONS
    
    disp('    Calculating moments of relative MSEs distribution');
    
    % Distribution of MSE ratios between models
    [T5_MSEr_s3_2v1,T5g_MSEr_s3_2v1]=getT5_largevar(MSEr_s3_2v1,cond_options,s3);
    [T5_MSEr_s4_2v1,T5g_MSEr_s4_2v1]=getT5_largevar(MSEr_s4_2v1,cond_options,s4);
    [T5_MSEr_s5_2v1,T5g_MSEr_s5_2v1]=getT5_largevar(MSEr_s5_2v1,cond_options,s5);
    [T5_MSEr_s6_2v1,T5g_MSEr_s6_2v1]=getT5_largevar(MSEr_s6_2v1,cond_options,s6);
    
    % ---------------------------------------------------------------------
    % ORGANIZE OUTPUT MATRICES
    
    disp('    Formatting results into output table');
    
    % Fill in Table 8 from the paper
    for l=1:length(lags)
        for h_idx=1:length(horizons)        
            for g=1:6
            
                T8_long(l+(g-1)*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_s3_2v1(2+(l-1)*2,h_idx,strcmp(cond_options,'none'),g);
                T8_long(l+length(lags)+(g-1)*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_s4_2v1(l*2,h_idx,strcmp(cond_options,'none'),g);
                T8_long(l+2*length(lags)+(g-1)*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_s5_2v1(l*2,h_idx,strcmp(cond_options,'none'),g);
                T8_long(l+3*length(lags)+(g-1)*length(lags)*4,h_idx+(sn-1)*length(horizons))=T5g_MSEr_s6_2v1(l*2,h_idx,strcmp(cond_options,'none'),g);
            
            end
        end
    end
            
    
    


end

end

% -------------------------------------------------------------------------
% FORMAT TABLE 8

disp('Formatting Table 8');

T8=[[T8_long(1:(4*length(lags)),:),T8_long((4*length(lags)+1):(2*4*length(lags)),:),T8_long((2*4*length(lags)+1):(3*4*length(lags)),:)];...
       [T8_long((3*4*length(lags)+1):(4*4*length(lags)),:),T8_long((4*4*length(lags)+1):(5*4*length(lags)),:),T8_long((5*4*length(lags)+1):(6*4*length(lags)),:)]];

Table8=cell(size(T8,1)+3,size(T8,2)+2);
Table8(4:end,3:end)=num2cell(T8);
sys_size={'Trivariate','4-Variate','5-Variate','6-Variate'};
for i=1:4
    Table8{4+3*(i-1),1}=sys_size{i};
    Table8{4+4*length(lags)+3*(i-1),1}=sys_size{i};
end
lags_label=cell(length(lags),1);
for i=1:length(lags)
    if isnumeric(lags{i})
        lags_label{i}=['AR(',num2str(lags{i}),')'];
    else
        lags_label{i}=lags{i};
    end
end
Table8(4:end,2)=repmat(lags_label,2*4,1);
Table8{1,1}='Data:';
Table8{2,1}='Sample:';
Table8{3,1}='Forecast Horizon:';
Table8(3,3:end)=num2cell(repmat(horizons,1,3*sn));
for i=1:3
    Table8{1,3+(i-1)*length(horizons)*2}='Full';
    Table8{1,3+length(horizons)+(i-1)*length(horizons)*2}='Great Moderation';
    
    Table8{2,3+(i-1)*length(horizons)*2}='MSW';
    Table8{2,3+length(horizons)+(i-1)*length(horizons)*2}='2003-2016'; 
    
end
Table8=[Table8(1:3,:);cell(1,size(Table8,2));Table8(4:(3+4*length(lags)),:);cell(1,size(Table8,2));Table8((4+4*length(lags)):end,:)];
Table8{4,3}='(A) Industrial Production';
Table8{4,3+length(horizons)*sn}='(B) Employment';
Table8{4,3+2*length(horizons)*sn}='(C) CPI';
Table8{5+length(lags)*4,3}='(D) PCEPI';
Table8{5+length(lags)*4,3+length(horizons)*sn}='(E) Federal Funds Rate';
Table8{5+length(lags)*4,3+2*length(horizons)*sn}='(F) 10-Year Treasury Yield';

% -------------------------------------------------------------------------
% WRITE RESULTS TO EXCEL

disp('Exporting Table 8 to Excel');
xlswrite(outfile,Table8,outsheet,'B2');

disp('Finished');

