clear
close all
clc

% =========================================================================
% DESCRIPTION
% This script organizes and writes the bivariate output to Excel.
%
% -------------------------------------------------------------------------
% USER-DEFINED VARIABLES


% Subsets of results to be tabulated
%   ycell = groups of series being forecast
%   xcell = groups of series as the other variable in the system
% E.g. if ycell{i}=[1,2,3,4] and xcell{i}=5, then the ith subset of results
% will be tabulated for forecasts of series in groups 1, 2, 3, 4 where the
% other variable in the system is from group 5.
ycell={[1,2,3,4];[1,2,3,4];5};
xcell={[1,2,3,4];5;[1,2,3,4]};

% Folder containing auxiliary functions for organizing data
auxfcn='FcnsOrg';

% Folder containing LeSage Toolbox functions
lesage='FcnsLeSage';

% Folder containing .mat files with raw ouputs from the forecasting
% exercises
rawoutput='RawOutput';

% Name of Excel file and sheet to which Table 8 is written.
outfile='FinalOutput\Tables.xlsx';
outsheet1='Table1';
outsheet3='Table3';
outsheet6='Table6';

% Horizons considered for Table 1, Table 3, and Table 6 in the paper
h1=[3,6,12,24];
h3=[3,12,24];
h6=h3;



% =========================================================================
% SETUP

addpath(auxfcn);
addpath(lesage);

% Preallocate memory for Table 3 and Table 6 from the paper
T3_long=NaN(16*4,3*length(h3));
T6_long=NaN(16*4,3*length(h3));

% Establish sample number
sn=0;

% Counter for number of structures
counter=0;

% Loop through great moderation data
for great_mod=0:1
    
counter=counter+1;
disp(['Structure ',num2str(counter)]);

% -------------------------------------------------------------------------
% DEFINE IN-SAMPLE AND OUT-OF-SAMPLE PERIODS

% Out-of-sample start and end dates
switch great_mod
    case 0
        OOS_start=[eom('1/31/1979');eom('12/31/2002')];
        OOSF_end=[eom('12/31/2002');eom('12/31/2016')];
    case 1
        OOS_start=eom('12/31/2002');
        OOSF_end=eom('12/31/2016');
    otherwise
        error('The switch great_mod is misspecified.');
end

% Number of samples
n_samples=length(OOS_start);

% -------------------------------------------------------------------------
% LOAD RELEVANT .MAT FILE

disp('  Loading Data');

switch great_mod
    case 0
        load([rawoutput,'\bivar_start59.mat']);
    case 1
        load([rawoutput,'\bivar_start84.mat']);
    otherwise
        error('The switch great_mod is misspecified.');
end

% -------------------------------------------------------------------------
% RENAME VARIABLES FROM OUTPUT STRUCTURE
% Makes it easier to identify them within the code

% Forecasts, actuals, lags, and Ecvu
IMSF=S.IMSF; % IMS forecasts 
DMSF=S.DMSF; % DMS forecasts
act=S.act; % Actuals

% User-defined variables
fdates=S.fdates; % Forecast dates
horizons=S.horizons; % Horizons
cond_options=S.cond_options; % Conditional options
sample_start=S.sample_start; % Sample start
lags=S.lags; % lags

% Group pairs
grp_pairs=S.grp_pairs;

% Clear structure
clear S;

% =========================================================================
% ORGANIZE AND SAVE RESULTS FOR EACH SAMPLE-SPLIT CONSIDERED

% Loop through sample splits being considered
for i=1:n_samples
    
    disp(['  Organizing Output for Sample-Split ',num2str(i),' of ',num2str(n_samples)]);
    
    % Increase sample number by 1
    sn=sn+1;


    % ---------------------------------------------------------------------
    % KEEP DESIRED OUT-OF-SAMPLE FORECASTS
    % Methods:
    %   Method 1 = VAR-based IMS
    %   Method 2 = ARDL-based DMS
    % Variable names
    %   F# - forecasts for method #
    %   A - actuals

    disp('    Removing unnecessary forecasts');

    % Remove observations to reflect the desired sample split
    F1=adjust_sample_split(IMSF,fdates,horizons,OOS_start(i),OOSF_end(i));
    F2=adjust_sample_split(DMSF,fdates,horizons,OOS_start(i),OOSF_end(i));
    A=adjust_sample_split(act,fdates,horizons,OOS_start(i),OOSF_end(i));
    
    % ---------------------------------------------------------------------
    % CALCULATE DESIRED STATISTICS
    
    disp('    Calculating desired statistics');
    
    % MSE ratios, t-stats from MSE pairwise comparisons, and dummies
    % identifying which comparisons are statistically significant in favor
    % of each model
    [MSEr_2v1,~,~,PCtstat_2v1,better_1v2,better_2v1]=get_stats(F1,F2,A,horizons);
    
    % ---------------------------------------------------------------------
    % DISTRIBUTIONS
    
    disp('    Calculating moments of each statistic''s distribution');
    
    % Distribution of MSE ratios between models
    [T5_MSEr_2v1,T5g_MSEr_2v1]=getT5_bivar(MSEr_2v1,grp_pairs,ycell,xcell);

    % Distribution of t-stats from test of equal MSEs
    [T5_PC_2v1,T5g_PC_2v1]=getT5_tstat_bivar(PCtstat_2v1,better_1v2,better_2v1,grp_pairs,ycell,xcell);
    
    % ---------------------------------------------------------------------
    % ORGANIZE OUTPUT MATRICES
    
    disp('    Formatting results into output tables');
    
    % Table 1
    if sn==1
        T1=NaN(size(T5_MSEr_2v1,1),length(h1));
        for h_idx=1:length(h1)
            T1(:,h_idx)=T5_MSEr_2v1(:,horizons==h1(h_idx));
        end
    end
    
    
    % Table 3
    for l=1:length(lags)
        for h_idx=1:length(h3)
            
            T3_long(1+(l-1)*4,h_idx+(sn-1)*length(h3))=T5_MSEr_2v1(1+(l-1)*6,horizons==h3(h_idx),strcmp(cond_options,'full'));
            T3_long(2+(l-1)*4,h_idx+(sn-1)*length(h3))=T5_MSEr_2v1(4+(l-1)*6,horizons==h3(h_idx),strcmp(cond_options,'full'));
            
            T3_long(3+(l-1)*4,h_idx+(sn-1)*length(h3))=100*T5_PC_2v1(7+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'))/...
                                                       T5_PC_2v1(9+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'));
            T3_long(4+(l-1)*4,h_idx+(sn-1)*length(h3))=100*T5_PC_2v1(8+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'))/...
                                                       T5_PC_2v1(9+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'));                                        
            
            for g=1:3
                T3_long(1+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h3))=T5g_MSEr_2v1(1+(l-1)*6,horizons==h3(h_idx),strcmp(cond_options,'full'),g);
                T3_long(2+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h3))=T5g_MSEr_2v1(4+(l-1)*6,horizons==h3(h_idx),strcmp(cond_options,'full'),g);
                
                T3_long(3+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h3))=100*T5g_PC_2v1(7+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'),g);
                T3_long(4+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h3))=100*T5g_PC_2v1(8+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,horizons==h3(h_idx),strcmp(cond_options,'full'),g);
            end
        end
    end
    
    % Table 6
    for l=1:length(lags)
        for h_idx=1:length(h6)
            
            T6_long(1+(l-1)*4,h_idx+(sn-1)*length(h6))=T5_MSEr_2v1(1+(l-1)*6,horizons==h6(h_idx),strcmp(cond_options,'none'));
            T6_long(2+(l-1)*4,h_idx+(sn-1)*length(h6))=T5_MSEr_2v1(4+(l-1)*6,horizons==h6(h_idx),strcmp(cond_options,'none'));
            
            T6_long(3+(l-1)*4,h_idx+(sn-1)*length(h6))=100*T5_PC_2v1(7+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'))/...
                                                       T5_PC_2v1(9+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'));
            T6_long(4+(l-1)*4,h_idx+(sn-1)*length(h6))=100*T5_PC_2v1(8+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'))/...
                                                       T5_PC_2v1(9+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'));                                        
            
            for g=1:3
                T6_long(1+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h6))=T5g_MSEr_2v1(1+(l-1)*6,horizons==h6(h_idx),strcmp(cond_options,'none'),g);
                T6_long(2+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h6))=T5g_MSEr_2v1(4+(l-1)*6,horizons==h6(h_idx),strcmp(cond_options,'none'),g);
                
                T6_long(3+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h6))=100*T5g_PC_2v1(7+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'),g);
                T6_long(4+(l-1)*4+g*length(lags)*4,h_idx+(sn-1)*length(h6))=100*T5g_PC_2v1(8+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'),g)/...
                                                           T5g_PC_2v1(9+(l-1)*9,horizons==h6(h_idx),strcmp(cond_options,'none'),g);
            end
        end
    end
    




end

end

% -------------------------------------------------------------------------
% FORMAT TABLES

disp('Formatting tables');

T3=[[T3_long(1:(4*length(lags)),:),T3_long((4*length(lags)+1):(2*4*length(lags)),:)];...
       [T3_long((2*4*length(lags)+1):(3*4*length(lags)),:),T3_long((3*4*length(lags)+1):(4*4*length(lags)),:)]];
   
T6=[[T6_long(1:(4*length(lags)),:),T6_long((4*length(lags)+1):(2*4*length(lags)),:)];...
       [T6_long((2*4*length(lags)+1):(3*4*length(lags)),:),T6_long((3*4*length(lags)+1):(4*4*length(lags)),:)]];

   
Table1=cell(size(T1,1)+2,size(T1,2)+2);
Table1(3:end,3:end)=num2cell(T1);
for i=1:length(lags)
    if isnumeric(lags{i})
        lags_i=['AR(',num2str(lags{i}),')'];
    else
        lags_i=lags{i};
    end
    Table1{3+6*(i-1),1}=lags_i;
end  
Table1(3:end,2)=repmat({'Mean';'0.10';'0.25';'0.50';'0.75';'0.90'},length(lags),1); 
Table1{1,1}='Model';
Table1{1,2}='Mean/percentile';
Table1{1,3}='Forecast horizon';
Table1(2,3:end)=num2cell(horizons);


   
Table3=cell(size(T3,1)+3,size(T3,2)+2);
Table3(4:end,3:end)=num2cell(T3);
for i=1:length(lags)
    if isnumeric(lags{i})
        lags_i=['AR(',num2str(lags{i}),')'];
    else
        lags_i=lags{i};
    end
    Table3{4+4*(i-1),1}=lags_i;
    Table3{4+4*length(lags)+4*(i-1),1}=lags_i;
end
Table3(4:end,2)=repmat({'Mean';'Median';'VAR Better';'ARDL Better'},2*length(lags),1);
Table3{1,1}='Data:';
Table3{2,1}='Sample:';
Table3{3,1}='Forecast Horizon:';
Table3(3,3:end)=num2cell(repmat(h3,1,2*sn));
for i=1:2
    Table3{1,3+(i-1)*length(h3)*3}='Full';
    Table3{1,3+length(h3)+(i-1)*length(h3)*3}='Full';
    Table3{1,3+2*length(h3)+(i-1)*length(h3)*3}='Great Moderation';
    
    Table3{2,3+(i-1)*length(h3)*3}='MSW';
    Table3{2,3+length(h3)+(i-1)*length(h3)*3}='2003-2016';
    Table3{2,3+2*length(h3)+(i-1)*length(h3)*3}='2003-2016';  
    
end
Table3=[Table3(1:3,:);cell(1,size(Table3,2));Table3(4:(3+4*length(lags)),:);cell(1,size(Table3,2));Table3((4+4*length(lags)):end,:)];
Table3{4,3}='(A) All variables';
Table3{4,3+length(h3)*sn}='(B) Pairs not including PWM';
Table3{5+length(lags)*4,3}='(C) Non-PWM variables in pairs with PWM variable';
Table3{5+length(lags)*4,3+length(h3)*sn}='(D) PWM variables';



Table6=cell(size(T6,1)+3,size(T6,2)+2);
Table6(4:end,3:end)=num2cell(T6);
for i=1:length(lags)
    if isnumeric(lags{i})
        lags_i=['AR(',num2str(lags{i}),')'];
    else
        lags_i=lags{i};
    end
    Table6{4+4*(i-1),1}=lags_i;
    Table6{4+4*length(lags)+4*(i-1),1}=lags_i;
end
Table6(4:end,2)=repmat({'Mean';'Median';'VAR Better';'ARDL Better'},2*length(lags),1);
Table6{1,1}='Data:';
Table6{2,1}='Sample:';
Table6{3,1}='Forecast Horizon:';
Table6(3,3:end)=num2cell(repmat(h6,1,2*sn));
for i=1:2
    Table6{1,3+(i-1)*length(h6)*3}='Full';
    Table6{1,3+length(h6)+(i-1)*length(h6)*3}='Full';
    Table6{1,3+2*length(h6)+(i-1)*length(h6)*3}='Great Moderation';
    
    Table6{2,3+(i-1)*length(h6)*3}='MSW';
    Table6{2,3+length(h6)+(i-1)*length(h6)*3}='2003-2016';
    Table6{2,3+2*length(h6)+(i-1)*length(h6)*3}='2003-2016';  
    
end
Table6=[Table6(1:3,:);cell(1,size(Table6,2));Table6(4:(3+4*length(lags)),:);cell(1,size(Table6,2));Table6((4+4*length(lags)):end,:)];
Table6{4,3}='(A) All variables';
Table6{4,3+length(h6)*sn}='(B) Pairs not including PWM';
Table6{5+length(lags)*4,3}='(C) Non-PWM variables in pairs with PWM variable';
Table6{5+length(lags)*4,3+length(h6)*sn}='(D) PWM variables';
 


% -------------------------------------------------------------------------
% WRITE RESULTS TO EXCEL

disp('Exporting tables to Excel');
xlswrite(outfile,Table1,outsheet1,'B2');
xlswrite(outfile,Table3,outsheet3,'B2');
xlswrite(outfile,Table6,outsheet6,'B2');

disp('Finished');