clear 
close all
clc

% =========================================================================
% DESCRIPTION
% This script runs the pseudo-out-of-sample forecasting exercises for the
% small subset of larger systems (trivariate, 4-variate, 5-variate, and
% 6-variate).

% -------------------------------------------------------------------------
% USER-DEFINED VARIABLES

% Switch for using great moderation data (0 => in-sample period begins in
% Jan. 1959, 1 => in-sample period begins in Jan. 1984)
great_mod=0;

% Vintage of FRED-MD used (YYYY-MM)
vintage='2017-06';

% Series in each of the three groups
%   group1 = real series
%   group2 = nominal series
%   group3 = financial series
group1={'INDPRO','CE16OV'};
group2={'CPIAUCSL','PCEPI'};
group3={'FEDFUNDS','GS10'};

% Minimum number of observations that each model must be estimated with for
% the forecasts for a certain forecast date to be considered
nobs_min=120;

% Conditional exercises performed
cond_options={'none'};

% Horizons
horizons=[3,12,24];

% Methods used to determine lag order
%   number - lag order is fixed at that number
%   'BIC' - lag order chosen by the BIC
%   'AIC' - lag order chosen by the AIC
lags={4,12,'BIC'};

% Folder containng data
datafolder='Data';

% Folder containing auxiliary functions
auxfcn='FcnsFcast';

% Folder to where .mat files with forecats are saved
outfolder='RawOutput';

% =========================================================================
% SETUP

addpath(auxfcn);

assert(great_mod==0 || great_mod==1);

% Define in-sample start and out-of-sample periods
%   sample_start = first in-sample period 
%   OOS_start = last in-sample period for the first forecast produced (so
%       the first period being forecast is OOS_start+h where h is the 
%       forecasting horizon)
%   OOSF_end = last period to be forecast
switch great_mod
    case 0
        sample_start='1/31/1959';
        OOS_start='1/31/1979';
    case 1
        sample_start='1/31/1984';
        OOS_start='12/31/2002';
        
    otherwise
        error('The switch great_mod is misspecified.');
end
OOSF_end='12/31/2016';
 
        
% =========================================================================
% GET DATA

% -------------------------------------------------------------------------
% LOAD FRED-MD VINTAGE

% Import from csv
raw=importdata([datafolder,'\',vintage,'.csv']);

% Data
fredmd=raw.data(2:end,:);

% Series names
series=raw.textdata(1,2:end);

% Transformation codes
tcode=raw.data(1,:);

% Dates
dates=datenum(raw.textdata(3:end,1));

% -------------------------------------------------------------------------
% ADJUST/LABEL DATA

N1=length(group1);
N2=length(group2);
N3=length(group3);

% Adjust all dates to be the final day of the month
sample_start=eom(sample_start);
OOS_start=eom(OOS_start);
OOSF_end=eom(OOSF_end);
dates=eom(dates);

% Remove excess dates
fredmd=fredmd(dates>=sample_start & dates<=datenum(OOSF_end),:);
dates=dates(dates>=sample_start & dates<=datenum(OOSF_end));

% Forecast dates
assert(sample_start<OOS_start);
fdates=dates(dates>datenum(OOS_start));

% Transform data
[fredmd_t1,fredmd_t2,~]=transform(fredmd,tcode,1);

% Combine groups
group=[group1,group2,group3];
    
% Identify where series from each group lie in fredmd
id=cellfun(@(x)find(strcmp(x,series),1),group);

% Get tcode for the three groups
tcode=tcode(id);

% Get series for the three groups after the first transformation (level or
% log) is applied
fredmd_t1=fredmd_t1(:,id);

% Get series for the three groups after the first transformation (level or
% log) and second transformation (level, first difference, or second
% difference) are applied
fredmd_t2=fredmd_t2(:,id);

% Check that last row is non-missing
assert(sum(isnan(fredmd_t1(end,:)))==0);

% -------------------------------------------------------------------------
% IDENTIFY ALL POSSIBLE SYSTEMS
% All 3-variable, 4-variable, 5-variable, and 6-variable systems; series
% are identified by numbers 1-6, with 1 being the first series in group 1
% and 6 being the last series in group 3.

% All possible combinations
s3=nchoosek(1:(N1+N2+N3),3);
s4=nchoosek(1:(N1+N2+N3),4);
s5=nchoosek(1:(N1+N2+N3),5);
s6=nchoosek(1:(N1+N2+N3),6);

% Identifiy which group (1,2, or 3) that each series belongs to
s3_group=NaN(size(s3));
s3_group(s3<=N1)=1;
s3_group(s3>N1 & s3<=(N1+N2))=2;
s3_group(s3>(N1+N2))=3;

s4_group=NaN(size(s4));
s4_group(s4<=N1)=1;
s4_group(s4>N1 & s4<=(N1+N2))=2;
s4_group(s4>(N1+N2))=3;

s5_group=NaN(size(s5));
s5_group(s5<=N1)=1;
s5_group(s5>N1 & s5<=(N1+N2))=2;
s5_group(s5>(N1+N2))=3;

s6_group=NaN(size(s6));
s6_group(s6<=N1)=1;
s6_group(s6>N1 & s6<=(N1+N2))=2;
s6_group(s6>(N1+N2))=3;

% Identify systems that meet our restirction of having at least one series
% from each group
keep3=(sum(s3_group==1,2)==1 & sum(s3_group==2,2)==1 & sum(s3_group==3,2)==1);
keep4=(sum(s4_group==1,2)>=1 & sum(s4_group==2,2)>=1 & sum(s4_group==3,2)>=1);
keep5=(sum(s5_group==1,2)>=1 & sum(s5_group==2,2)>=1 & sum(s5_group==3,2)>=1);
keep6=(sum(s6_group==1,2)>=1 & sum(s6_group==2,2)>=1 & sum(s6_group==3,2)>=1);

s3=s3(keep3,:);
s4=s4(keep4,:);
s5=s5(keep5,:);
s6=s6(keep6,:);

% -------------------------------------------------------------------------
% PREALLOCATE MEMORY

% Get dimensions
nF=sum(dates<=datenum(OOSF_end))-sum(dates<=datenum(OOS_start)); % number of forecasts
H=length(horizons); % number of horizons
L=length(lags); % number of lag-selection methods
C=length(cond_options); % number of conditioning exercises

% Check number of forecast dates equals number of forecasts
assert(nF==length(fdates));

% Number of systems of each size
Nsys3=size(s3,1); % trivariate
Nsys4=size(s4,1); % 4-variate
Nsys5=size(s5,1); % 5-variate
Nsys6=size(s6,1); % 6-variate

% IMS forecasts
%   Dimensions:
%       1 = period being forecast
%       2 = horizon
%       3 = lag-selection method
%       4 = conditioning exercise
%       5 = variable in system is being forecast
%       6 = system 
IMSF3=NaN(nF,H,L,C,3,Nsys3); % trivariate
IMSF4=NaN(nF,H,L,C,4,Nsys4); % 4-variate
IMSF5=NaN(nF,H,L,C,5,Nsys5); % 5-variate
IMSF6=NaN(nF,H,L,C,6,Nsys6); % 6-variate

% DMS forecats
DMSF3=IMSF3;
DMSF4=IMSF4;
DMSF5=IMSF5;
DMSF6=IMSF6;

% Realized values
act3=IMSF3;
act4=IMSF4;
act5=IMSF5;
act6=IMSF6;

% Number of lags, IMS models
P_IMS3=IMSF3;
P_IMS4=IMSF4;
P_IMS5=IMSF5;
P_IMS6=IMSF6;

% Number of lags, DMS models
P_DMS3=IMSF3;
P_DMS4=IMSF4;
P_DMS5=IMSF5;
P_DMS6=IMSF6;

% =========================================================================
% PART 2: RUN FORECASTING EXERCISES

% -------------------------------------------------------------------------
% SAVE INPUT VALUES

% Input to function that produces the forecasts
input.nF=nF; 
input.tcode=tcode;
input.horizons=horizons;
input.dates=dates;
input.cond_options=cond_options;
input.lags=lags;
input.OOS_start=OOS_start;
input.OOSF_end=OOSF_end;
input.nobs_min=nobs_min;
input.fredmd_t1=fredmd_t1;
input.fredmd_t2=fredmd_t2; 

% -------------------------------------------------------------------------
% TRIVARIATE SYSTEMS

disp('TRIVARIATE SYSTEM EXERCISES');
parfor i=1:Nsys3 
    
    disp(['System ',num2str(i),' of ',num2str(Nsys3)]);
    
    [IMSF3_temp,DMSF3_temp,act3_temp,P_IMS3_temp,P_DMS3_temp]=largevar_fcast(input,s3(i,:));
    
    IMSF3(:,:,:,:,:,i)=IMSF3_temp;
    DMSF3(:,:,:,:,:,i)=DMSF3_temp;
    act3(:,:,:,:,:,i)=act3_temp;
    P_IMS3(:,:,:,:,:,i)=P_IMS3_temp;
    P_DMS3(:,:,:,:,:,i)=P_DMS3_temp;
    
end

% -------------------------------------------------------------------------
% 4-VARIABLE SYSTEMS

disp('4-VARIABLE SYSTEM EXERCISES');
parfor i=1:Nsys4 
    
    disp(['System ',num2str(i),' of ',num2str(Nsys4)]);
    
    [IMSF4_temp,DMSF4_temp,act4_temp,P_IMS4_temp,P_DMS4_temp]=largevar_fcast(input,s4(i,:));
    
    IMSF4(:,:,:,:,:,i)=IMSF4_temp;
    DMSF4(:,:,:,:,:,i)=DMSF4_temp;
    act4(:,:,:,:,:,i)=act4_temp;
    P_IMS4(:,:,:,:,:,i)=P_IMS4_temp;
    P_DMS4(:,:,:,:,:,i)=P_DMS4_temp;
    
end

% -------------------------------------------------------------------------
% 5-VARIABLE SYSTEMS

disp('5-VARIABLE SYSTEM EXERCISES');
parfor i=1:Nsys5 
    
    disp(['System ',num2str(i),' of ',num2str(Nsys5)]);
    
    [IMSF5_temp,DMSF5_temp,act5_temp,P_IMS5_temp,P_DMS5_temp]=largevar_fcast(input,s5(i,:));
    
    IMSF5(:,:,:,:,:,i)=IMSF5_temp;
    DMSF5(:,:,:,:,:,i)=DMSF5_temp;
    act5(:,:,:,:,:,i)=act5_temp;
    P_IMS5(:,:,:,:,:,i)=P_IMS5_temp;
    P_DMS5(:,:,:,:,:,i)=P_DMS5_temp;
    
end

% -------------------------------------------------------------------------
% 6-VARIABLE SYSTEMS

disp('6-VARIABLE SYSTEM EXERCISES');
assert(Nsys6==1);
for i=1:Nsys6 
    
    disp(['System ',num2str(i),' of ',num2str(Nsys6)]);
    
    [IMSF6_temp,DMSF6_temp,act6_temp,P_IMS6_temp,P_DMS6_temp]=largevar_fcast(input,s6(i,:));
    
    IMSF6(:,:,:,:,:,i)=IMSF6_temp;
    DMSF6(:,:,:,:,:,i)=DMSF6_temp;
    act6(:,:,:,:,:,i)=act6_temp;
    P_IMS6(:,:,:,:,:,i)=P_IMS6_temp;
    P_DMS6(:,:,:,:,:,i)=P_DMS6_temp;
    
end


% =========================================================================
% PART 3: SAVE OUTPUT

disp('SAVING RESULTS');

% Save output to structure

% General variables
S.fdates=fdates; 
S.horizons=horizons; 
S.cond_options=cond_options; 
S.sample_start=sample_start;
S.lags=lags;

% Trivariate results
S.IMSF3=IMSF3;
S.DMSF3=DMSF3;
S.act3=act3;
S.P_IMS3=P_IMS3;
S.P_DMS3=P_DMS3;
S.s3=s3;

% 4-variate results
S.IMSF4=IMSF4;
S.DMSF4=DMSF4;
S.act4=act4;
S.P_IMS4=P_IMS4;
S.P_DMS4=P_DMS4;
S.s4=s4;

% 5-variate results
S.IMSF5=IMSF5;
S.DMSF5=DMSF5;
S.act5=act5;
S.P_IMS5=P_IMS5;
S.P_DMS5=P_DMS5;
S.s5=s5;

% 6-variate results
S.IMSF6=IMSF6;
S.DMSF6=DMSF6;
S.act6=act6;
S.P_IMS6=P_IMS6;
S.P_DMS6=P_DMS6;
S.s6=s6;

% Name of .mat file
switch sample_start
    case eom('1/31/1959')
        yr=59;
    case eom('1/31/1984')
        yr=84;
end
matfilename=[outfolder,'/largevar_start',num2str(yr)];

% Save
save(matfilename,'S','-v7.3');


