clear 
close all
clc

% =========================================================================
% DESCRIPTION
% This script creates Figure 1 from the paper. This figure contains four
% subplots, each plotting DMS percent improvement in BIC vs IMS percent
% improvement in BIC for a given series group, horizon, and number of lags
% in the model. The percent improvement in BIC is calculated as
% (x1-x2)/|x1| where x1 is the BIC in the first sample (Jan. 1959 - Dec.
% 1983) and x2 is the BIC in the second sample (Jan. 1984 - Dec. 2008).

% -------------------------------------------------------------------------
% USER-DEFINED VARIABLES

% Vintage of FRED-MD used (YYYY-MM)
vintage='2017-06';

% Series in each of the three groups
%   group1 = real series
%   group2 = nominal series
%   group3 = financial series
group1={'RPI','INDPRO','CE16OV','UNRATE','AWHMAN','DPCERA3M086SBEA'};
group2={'CES3000000008','WPSFD49207','OILPRICEx','CPIAUCSL','PCEPI'};
group3={'M1SL','FEDFUNDS','GS10','S&P 500'};

% Maximum number of lags considered
pmax=12;

% Forecast horizons
horizons=[3,12,24];

% Lags
lags={2,4,12};

% Folder containng data
datafolder='Data';

% Folder containing auxiliary functions for performing forecasts 
auxfcn1='FcnsFcast';

% Folder containing auxiliary functions for organizing data
auxfcn2='FcnsOrg';

% Name of output file
outfile='FinalOutput\Figure1';

% =========================================================================
% SETUP

addpath(auxfcn1);
addpath(auxfcn2);

% Get dimensions
H=length(horizons);
L=length(lags);
K=3;
N1=length(group1);
N2=length(group2);
N3=length(group3);

% Preallocate memory
BIC_IMS=NaN(1,H,L,1,K,N1,N2,N3,2);
BIC_DMS=NaN(1,H,L,1,K,N1,N2,N3,2);

% Two runs, one for pre-GM data, one for post-GM data
for run=1:2
    
switch run
    case 1
        sample_start='1/31/1959';
        sample_end='12/31/1983';

    case 2
        sample_start='1/31/1984';
        sample_end='12/31/2008';
end



% =========================================================================
% GET DATA

% -------------------------------------------------------------------------
% LOAD FRED-MD VINTAGE

% Import from csv
raw=importdata([datafolder,'\',vintage,'.csv']);

% Data
fredmd=raw.data(2:end,:);

% Series names
series=raw.textdata(1,2:end);

% Transformation codes
tcode=raw.data(1,:);

% Dates
dates=datenum(raw.textdata(3:end,1));

% -------------------------------------------------------------------------
% ADJUST/LABEL DATA

% Adjust all dates to be the final day of the month
sample_start=eom(sample_start);
sample_end=eom(sample_end);
dates=eom(dates);

% Remove excess dates
fredmd=fredmd(dates>=sample_start & dates<=datenum(sample_end),:);
dates=dates(dates>=sample_start & dates<=datenum(sample_end));

% Transform data
[fredmd_t1,fredmd_t2,~]=transform(fredmd,tcode,1);

% Identify where series from each group lie in fredmd
id1=cellfun(@(x)find(strcmp(x,series),1),group1);
id2=cellfun(@(x)find(strcmp(x,series),1),group2);
id3=cellfun(@(x)find(strcmp(x,series),1),group3);

% Get tcode for the three groups
tcode1=tcode(id1);
tcode2=tcode(id2);
tcode3=tcode(id3);

% Get series for the three groups after the first transformation (level or
% log) is applied
fredmd1_t1=fredmd_t1(:,id1);
fredmd2_t1=fredmd_t1(:,id2);
fredmd3_t1=fredmd_t1(:,id3);

% Get series for the three groups after the first transformation (level or
% log) and second transformation (level, first difference, or second
% difference) are applied
fredmd1_t2=fredmd_t2(:,id1);
fredmd2_t2=fredmd_t2(:,id2);
fredmd3_t2=fredmd_t2(:,id3);

% Check that last row is non-missing
assert(sum(isnan(fredmd1_t1(end,:)))==0);
assert(sum(isnan(fredmd2_t1(end,:)))==0);
assert(sum(isnan(fredmd3_t1(end,:)))==0);


% -------------------------------------------------------------------------
counter=0;
for i1=1:N1
for i2=1:N2
for i3=1:N3  
    counter=counter+1;
    disp(['Series pairing ',num2str(counter),' of ',num2str(length(group1)*length(group2)*length(group3))]);

    tcode_i=[tcode1(i1),tcode2(i2),tcode3(i3)];

    x1=[fredmd1_t1(:,i1,:),fredmd2_t1(:,i2,:),fredmd3_t1(:,i3,:)];
    x2=[fredmd1_t2(:,i1,:),fredmd2_t2(:,i2,:),fredmd3_t2(:,i3,:)];

    for l=1:length(lags)
        
      % Get lags
      p=lags{l};
      
      % Get rows corresponding to first date (T0), first date being
      % forecast (T1), and last date being forecast (T2)
      assert(length(dates)==size(x2,1))
      T0=find(~isnan(sum(x2,2)),1,'first');
      T2=sum(dates<=datenum(sample_end));

      % Loop through number of horizons considered
      for h_idx=1:H

        % Get horizon
        h=horizons(h_idx);

        % Transform series based on horizon
        x2h=accumulate_h(x2,h,tcode_i);

        % Loop through number of different orderings
        for k=1:K

          % Determine order of the series; the first series is the one being
          % forecasted
          switch k
              case 1
                order=[1,2,3];
              case 2
                order=[2,1,3];
              case 3
                order=[3,1,2];
              case 4
                order=[1,3,2];
              case 5
                order=[2,3,1];
              case 6
                order=[3,2,1];
          end

          % Put data in order
          y2=x2(:,order,:);
          y2h=x2h(:,order,:);

          y_is=y2(T0:T2,1)';
          yh_is=y2h(T0:T2,1)';
          X_is=y2(T0:T2,2:end)';

          % Get conditional values of variables not being forecasted
          X_cond=NaN(size(X_is,1),h);

          % Checks
          assert(sum(isnan(y_is))==0);
          assert(sum(isnan(yh_is((h+1):end)))==0);
          assert(sum(sum(isnan(X_is)))==0);

          % BIC for DMS model
          [~,~,~,~,~,BICdms]=cond_ARDL_DMS(yh_is,y_is,X_is,h,X_cond,p,12);
          BIC_DMS(1,h_idx,l,1,k,i1,i2,i3,run)=BICdms;
          
          % BIC for IMS model
          Y=[y_is;X_is];
          Yt=Y(:,pmax+1:end);
          Ylags=NaN(size(Y,1)*p,size(Yt,2));
          for pi=1:p
              Ylags((1+(pi-1)*size(Y,1)):(pi*size(Y,1)),:)=Y(:,(pmax+1-pi):(end-pi)); 
          end
          X=[ones(1,size(Ylags,2));Ylags];
          B=Yt*X'/(X*X');

          [N,T]=size(Yt);
          ehat1=Yt-B*X;
          kk=size(B,1)*size(B,2);
           
          BIC_IMS(1,h_idx,l,1,k,i1,i2,i3,run)=T*log(det((1/T)*(ehat1*ehat1')))+kk*log(T);
          
          


        end
      end
    end
end
end
end

end

% =========================================================================
% CREATE FIGURE

% -------------------------------------------------------------------------
% ORGANIZE OUTPUT

% Percent improvement in BIC from pre-GM period to post-GM period
IMSratio=100*(BIC_IMS(:,:,:,:,:,:,:,:,1)-BIC_IMS(:,:,:,:,:,:,:,:,2))./abs(BIC_IMS(:,:,:,:,:,:,:,:,1));
DMSratio=100*(BIC_DMS(:,:,:,:,:,:,:,:,1)-BIC_DMS(:,:,:,:,:,:,:,:,2))./abs(BIC_DMS(:,:,:,:,:,:,:,:,1));

% -------------------------------------------------------------------------
% FIGURE SETUP

% x-axis and y-axis labels
xlab='IMS Percent Improvement';
ylab='DMS Percent Improvement';

% Second lag-selection method (i.e. p=4)
l=2;

% Bounds on DMS percent improvement
DMSmax=40;
DMSmin=-DMSmax;

% x-axis and y-axis limits
yl=[DMSmin,DMSmax];
xl=[-20,40];

% Create figure
width=9; 
height=6.5;
x0=0;
y0=0;
figure('Units','inches',...
    'Position',[x0 y0 width height],...
    'PaperPositionMode','auto');

% Number of plots across and down
numRecsAcross=2;
numRecsDown=2;

% gaps between plots
gap1=0.1;
gap2=0.06;

% Format of text box in each subplot
format='%3s %3s\n%3s %3s\n%3s %3s\n%3s %3s\n%3s %3s\n%3s %3s';

% Preallocate memory for matrix containing fraction of observations in each
% region for each plot
frac=NaN(4,6);

% -------------------------------------------------------------------------
% SUBPLOT 1
% Real variables
% p=4
% h=12

% First row and first column
j=1;
i=1;

% Create subplot
subplot('Position',[(j-1)*1/numRecsAcross+gap2, (numRecsDown-i)*1/numRecsDown+gap1/2, 1/numRecsAcross-gap2, 1/numRecsDown-gap1])

% First series group, second horizon
k_idx=1;
h_idx=2;

% Vectorize percent improvements 
IMSratio_i=IMSratio(:,h_idx,l,1,k_idx,:,:,:);
DMSratio_i=DMSratio(:,h_idx,l,1,k_idx,:,:,:);
IMScol=IMSratio_i(:);
DMScol=DMSratio_i(:);

% Replace DMS percent improvement values beyond limits with those limits
DMScol(DMScol>DMSmax)=DMSmax;
DMScol(DMScol<DMSmin)=DMSmin;

% Identify series group
switch k_idx
    case 1
        grp='Real';
    case 2
        grp='Nominal';
    case 3
        grp='Financial';
end

% Plot observations
scatter(IMScol,DMScol);
hold on
plot([-100000,100000],[-100000,100000],'k--');
plot([-100000,100000],[0,0],'k');
plot([0,0],[-100000,100000],'k');
xlim(xl);
ylim(yl);
ylabel(ylab);
title([grp,', h=',num2str(horizons(h_idx))]);  

% Report percentage of points in each region
frac1A=sum(IMScol>=0 & DMScol>=0 & IMScol>=DMScol)/length(IMScol);
frac1B=sum(IMScol>=0 & DMScol>=0 & IMScol<DMScol)/length(IMScol);
frac2=sum(IMScol<0 & DMScol>=0)/length(IMScol);
frac3A=sum(IMScol<0 & DMScol<0 & IMScol<DMScol)/length(IMScol);
frac3B=sum(IMScol<0 & DMScol<0 & IMScol>=DMScol)/length(IMScol);
frac4=sum(IMScol>=0 & DMScol<0)/length(IMScol);
frac(1,:)=[frac1A,frac1B,frac2,frac3A,frac3B,frac4];
message=sprintf(format,'1A:','14%','1B:','53%','2:','33%','3A:','0%','3B:','0%','4:','0%');
annotation('textbox', [0.405, 0.55, 0.08, 0.2], ...
    'string', message,...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

% Change font
set(gca,'FontName','FixedWidth');

% Label regions (1A, 1B, 2, 3A, 3B, 4)
annotation('textbox', [0.405, 0.72, 0.08, 0.2], ...
    'string', '1A',...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

annotation('textbox', [0.26, 0.8, 0.08, 0.2], ...
    'string', '1B',...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

annotation('textbox', [0.07, 0.75, 0.08, 0.2], ...
    'string', ' 2 ',...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

annotation('textbox', [0.053, 0.62, 0.08, 0.2], ...
    'string', '3A',...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

annotation('textbox', [0.11, 0.55, 0.08, 0.2], ...
    'string', '3B',...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

annotation('textbox', [0.26, 0.55, 0.08, 0.2], ...
    'string', ' 4 ',...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

% -------------------------------------------------------------------------
% SUBPLOT 2
% Financial variables
% p=4
% h=12

% First row, second column
j=2;
i=1;

% Create subplot
subplot('Position',[(j-1)*1/numRecsAcross+gap2, (numRecsDown-i)*1/numRecsDown+gap1/2, 1/numRecsAcross-gap2, 1/numRecsDown-gap1])

% Third series group, second horizon
k_idx=3;
h_idx=2;

% Vectorize percent improvements 
IMSratio_i=IMSratio(:,h_idx,l,1,k_idx,:,:,:);
DMSratio_i=DMSratio(:,h_idx,l,1,k_idx,:,:,:);
IMScol=IMSratio_i(:);
DMScol=DMSratio_i(:);

% Replace DMS percent improvement values beyond limits with those limits
DMScol(DMScol>DMSmax)=DMSmax;
DMScol(DMScol<DMSmin)=DMSmin;

% Identify series group
switch k_idx
    case 1
        grp='Real';
    case 2
        grp='Nominal';
    case 3
        grp='Financial';
end

% Plot observations
scatter(IMScol,DMScol);
hold on
plot([-100000,100000],[-100000,100000],'k--');
plot([-100000,100000],[0,0],'k');
plot([0,0],[-100000,100000],'k');
xlim(xl);
ylim(yl);
title([grp,', h=',num2str(horizons(h_idx))]);

% Report percentage of points in each region
frac1A=sum(IMScol>=0 & DMScol>=0 & IMScol>=DMScol)/length(IMScol);
frac1B=sum(IMScol>=0 & DMScol>=0 & IMScol<DMScol)/length(IMScol);
frac2=sum(IMScol<0 & DMScol>=0)/length(IMScol);
frac3A=sum(IMScol<0 & DMScol<0 & IMScol<DMScol)/length(IMScol);
frac3B=sum(IMScol<0 & DMScol<0 & IMScol>=DMScol)/length(IMScol);
frac4=sum(IMScol>=0 & DMScol<0)/length(IMScol);
frac(2,:)=[frac1A,frac1B,frac2,frac3A,frac3B,frac4];
message=sprintf(format,'1A:','0%','1B:','47%','2:','3%','3A:','1%','3B:','29%','4:','20%');
annotation('textbox', [0.905, 0.55, 0.08, 0.2], ...
    'string', message,...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

% Change font
set(gca,'FontName','FixedWidth');

% -------------------------------------------------------------------------
% SUBPLOT 3
% Nominal variables
% p=4
% h=3

% Second row, first column
j=1;
i=2;

% Create subplot
subplot('Position',[(j-1)*1/numRecsAcross+gap2, (numRecsDown-i)*1/numRecsDown+0.75*gap1, 1/numRecsAcross-gap2, 1/numRecsDown-gap1])

% Second series group, first horizon
k_idx=2;
h_idx=1;

% Vectorize percent improvements
IMSratio_i=IMSratio(:,h_idx,l,1,k_idx,:,:,:);
DMSratio_i=DMSratio(:,h_idx,l,1,k_idx,:,:,:);
IMScol=IMSratio_i(:);
DMScol=DMSratio_i(:);

% Replace DMS percent improvement values beyond limits with those limits
DMScol(DMScol>DMSmax)=DMSmax;
DMScol(DMScol<DMSmin)=DMSmin;

% Identify series groiup
switch k_idx
    case 1
        grp='Real';
    case 2
        grp='Nominal';
    case 3
        grp='Financial';
end

% Plot observations
scatter(IMScol,DMScol);
hold on
plot([-100000,100000],[-100000,100000],'k--');
plot([-100000,100000],[0,0],'k');
plot([0,0],[-100000,100000],'k');
xlim(xl);
ylim(yl);
xlabel(xlab);
ylabel(ylab);
title([grp,', h=',num2str(horizons(h_idx))]);

% Report percentage of points in each region
frac1A=sum(IMScol>=0 & DMScol>=0 & IMScol>=DMScol)/length(IMScol);
frac1B=sum(IMScol>=0 & DMScol>=0 & IMScol<DMScol)/length(IMScol);
frac2=sum(IMScol<0 & DMScol>=0)/length(IMScol);
frac3A=sum(IMScol<0 & DMScol<0 & IMScol<DMScol)/length(IMScol);
frac3B=sum(IMScol<0 & DMScol<0 & IMScol>=DMScol)/length(IMScol);
frac4=sum(IMScol>=0 & DMScol<0)/length(IMScol);
frac(3,:)=[frac1A,frac1B,frac2,frac3A,frac3B,frac4];
message=sprintf(format,'1A:','6%','1B:','13%','2:','1%','3A:','4%','3B:','28%','4:','48%');
annotation('textbox', [0.076, 0.28, 0.08, 0.2], ...
    'string', message,...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

% Change font
set(gca,'FontName','FixedWidth');

% -------------------------------------------------------------------------
% SUBPLOT 4
% Nominal variables
% p=4
% h=24

% Second row and second column
j=2;
i=2;

% Create subplot
subplot('Position',[(j-1)*1/numRecsAcross+gap2, (numRecsDown-i)*1/numRecsDown+0.75*gap1, 1/numRecsAcross-gap2, 1/numRecsDown-gap1])

% Second series group, third horizon
k_idx=2;
h_idx=3;

% Vectorize percent improvements
IMSratio_i=IMSratio(:,h_idx,l,1,k_idx,:,:,:);
DMSratio_i=DMSratio(:,h_idx,l,1,k_idx,:,:,:);
IMScol=IMSratio_i(:);
DMScol=DMSratio_i(:);

% Replace DMS percent improvement values beyond limits with those limits
DMScol(DMScol>DMSmax)=DMSmax;
DMScol(DMScol<DMSmin)=DMSmin;

% Identify series group
switch k_idx
    case 1
        grp='Real';
    case 2
        grp='Nominal';
    case 3
        grp='Financial';
end

% Plot observations
scatter(IMScol,DMScol);
hold on
plot([-100000,100000],[-100000,100000],'k--');
plot([-100000,100000],[0,0],'k');
plot([0,0],[-100000,100000],'k');
xlim(xl);
ylim(yl);
xlabel(xlab);
title([grp,', h=',num2str(horizons(h_idx))]);

% Report percentage of points in each region
frac1A=sum(IMScol>=0 & DMScol>=0 & IMScol>=DMScol)/length(IMScol);
frac1B=sum(IMScol>=0 & DMScol>=0 & IMScol<DMScol)/length(IMScol);
frac2=sum(IMScol<0 & DMScol>=0)/length(IMScol);
frac3A=sum(IMScol<0 & DMScol<0 & IMScol<DMScol)/length(IMScol);
frac3B=sum(IMScol<0 & DMScol<0 & IMScol>=DMScol)/length(IMScol);
frac4=sum(IMScol>=0 & DMScol<0)/length(IMScol);
frac(4,:)=[frac1A,frac1B,frac2,frac3A,frac3B,frac4];
message=sprintf(format,'1A:','11%','1B:','47%','2:','23%','3A:','0%','3B:','11%','4:','9%');
annotation('textbox', [0.905, 0.075, 0.08, 0.2], ...
    'string', message,...
    'Fontsize',10,...
    'FontName','FixedWidth',...
    'EdgeColor',[0,0,0],...
    'VerticalAlignment','middle',...
    'HorizontalAlignment','center',...
    'FitBoxToText','on',...
    'BackgroundColor',[1,1,1]);

% Change font
set(gca,'FontName','FixedWidth');

% -------------------------------------------------------------------------
% PRINT FIGURE

print(outfile,'-depsc'); 
