clear
close all
clc

% =========================================================================
% DESCRIPTION
% This script runs the pseudo-out-of-sample forecasting exercises for the
% bivariate systems.

% -------------------------------------------------------------------------
% USER-DEFINED VARIABLES

% Switch for using great moderation data (0 => in-sample period begins in
% Jan. 1959, 1 => in-sample period begins in Jan. 1984)
great_mod=0;

% Vintage of FRED-MD used (YYYY-MM)
vintage='2017-06';

% Excel file and sheet containing groups
infile='groups.xlsx';
insheet='groups';

% Name of text file for writing out names of bad series that are removed
% from consideration
bad_outfile='FinalOutput\bad_series.txt';

% Horizons (months ahead)
horizons=[3,6,12,24];

% Methods used to determine lag order
%   number - lag order is fixed at that number
%   'BIC' - lag order chosen by the BIC
%   'AIC' - lag order chosen by the AIC
lags={4,12,'BIC','AIC'};

% Options for what values the forecasts should be conditioned on
cond_options={'none','full'};

% Maximum allowable start date
maxstart='1/31/1967';

% Minimum number of observations that each model must be estimated with for
% the forecasts for a certain forecast date to be considered
nobs_min=120;

% Number of trials per group pairing
ntrials=200;

% Seed for random number generator
sd=42;

% Folder containng data
datafolder='Data';

% Folder containing auxiliary functions
auxfcn='FcnsFcast';

% Folder to where .mat files with forecats are saved
outfolder='RawOutput';

% =========================================================================
% PART 1: SETUP

addpath(auxfcn);

% -------------------------------------------------------------------------
% DEFINE IN-SAMPLE AND OUT-OF-SAMPLE PERIODS

% Check switch is defined properly
assert(great_mod==0 || great_mod==1);

% Define in-sample start and out-of-sample periods
%   sample_start = first in-sample period 
%   OOS_start = last in-sample period for the first forecast produced (so
%       the first period being forecast is OOS_start+h where h is the 
%       forecasting horizon)
%   OOSF_end = last period to be forecast
switch great_mod
    case 0
        sample_start='1/31/1959';
        OOS_start='1/31/1979';
    case 1
        sample_start='1/31/1984';
        OOS_start='12/31/2002';
        
    otherwise
        error('The switch great_mod is misspecified.');
end
OOSF_end='12/31/2016';

% -------------------------------------------------------------------------
% SEED THE RANDOM NUMBER GENERATOR

rng(sd);

% -------------------------------------------------------------------------
% LOAD DATA

% Import from csv
raw=importdata([datafolder,'\',vintage,'.csv']);

% Data
fredmd_raw=raw.data(2:end,:);

% Series names
series=raw.textdata(1,2:end);

% Transformation codes
tcode=raw.data(1,:);

% Dates
dates_raw=datenum(raw.textdata(3:end,1));

% Load series groups
[~,groups]=xlsread([datafolder,'\',infile],insheet);

% -------------------------------------------------------------------------
% ADJUST/LABEL DATA

% Adjust all dates to be the final day of the month
sample_start=eom(sample_start);
OOS_start=eom(OOS_start);
OOSF_end=eom(OOSF_end);
dates_raw=eom(dates_raw);
maxstart=eom(maxstart);

% Remove excess dates
fredmd=fredmd_raw(dates_raw>=sample_start & dates_raw<=OOSF_end,:);
dates=dates_raw(dates_raw>=sample_start & dates_raw<=OOSF_end);

% Get all possble forecast dates (all months after OOS_start)
assert(sample_start<OOS_start);
fdates=dates(dates>OOS_start);

% -------------------------------------------------------------------------
% REMOVE PROBLEMATIC SERIES

% Call function that identifies problematic series
bad=find_bad(fredmd_raw(dates_raw<=OOSF_end,:),...
    dates_raw(dates_raw<=OOSF_end),series,maxstart,OOSF_end,bad_outfile);

% Remove bad series from the dataset
series=series(~bad);
tcode=tcode(~bad);
fredmd=fredmd(:,~bad);

% -------------------------------------------------------------------------
% SERIES INDEXES AND GROUP VARIABLES

% Get needed dimensions
%   T = number of observations per series
%   N = number of series
%   ng = number of groups
[T,N]=size(fredmd); 
ng=size(groups,2);

% Index number of each series within the main dataset
series_idx=1:N;

% Group number for each series (gcode) and number of series in each
% group (grp_size)
gcode=zeros(size(tcode));
grp_size=NaN(1,ng);
for g=1:ng 
    gcode(ismember(series,groups(:,g)))=g;
    grp_size(g)=sum(gcode==g);
end

% Check that each series is assigned to a group
assert(sum(gcode==0)==0);

% All possible pairs of series groups
grp_pairs=nchoosek(1:ng,2);

% -------------------------------------------------------------------------
% PREALLOCATE MEMORY

% Get dimensions
nF=sum(dates<=OOSF_end)-sum(dates<=OOS_start);
H=length(horizons);
L=length(lags);
C=length(cond_options);
K=2;
GP=size(grp_pairs,1);

% Check number of forecast dates equals number of forecasts
assert(nF==length(fdates));

% IMS forecasts
%   Dimensions:
%       1 = period being forecast
%       2 = horizon
%       3 = lag-selection method
%       4 = conditioning exercise
%       5 = variable in system is being forecast
%       6 = system 
IMSF=NaN(nF,H,L,C,K,GP,ntrials);

% DMS forecasts
DMSF=IMSF;

% Actauls
act=IMSF;

% Number of lags selected 
P_IMS=IMSF;
P_DMS=IMSF;

% IMS forecast before undoing differencing
IMSF_diff=IMSF;

% Actuals before undoing differencing
act_diff=IMSF;

% Expectation of difference between conditional vs. uncondtional squared 
% forecast error
Ecvu=IMSF;

% -------------------------------------------------------------------------
% RANDOMLY SELECT SERIES TO USE

% Preallocate memory for series pairs to be used
srs_pairs_use_all=NaN(ntrials,2,GP);

% Loop through number of group pairs
for gp=1:GP   
    
    % Group numbers of the two groups (a and b) in group pair gp
    grp_a=grp_pairs(gp,1); 
    grp_b=grp_pairs(gp,2); 
    
    % All possible series pairings from group a and group b (using index
    % number of each series within its group, e.g. the fifth series in
    % a group is represented by a 5 here)
    srs_pairs=NaN(grp_size(grp_a)*grp_size(grp_b),2);
    srs_pairs(:,1)=ceil((1:size(srs_pairs,1))/grp_size(grp_b));
    srs_pairs(:,2)=repmat((1:grp_size(grp_b))',grp_size(grp_a),1);
    
    % Randomly select subsample of these series pairs; this is done by
    % putting all the pairs in a random order and selecting the first
    % 'ntrail' pairs
    [~,Irand]=sort(rand(size(srs_pairs,1),1));
    srs_pairs=srs_pairs(Irand,:);
    srs_pairs_use=srs_pairs(1:ntrials,:);
    
    % Save series pairs to final matrix
    srs_pairs_use_all(:,:,gp)=srs_pairs_use;
    
end
    

% =========================================================================
% PART 2: RUN FORECASTING EXERCISES

% -------------------------------------------------------------------------
% SAVE INPUT VALUES
% Save values to structure 'input' to be passed along as an input into the
% function that performs the forecasting exercises

input.nF=nF;
input.horizons=horizons;
input.dates=dates;
input.cond_options=cond_options;
input.lags=lags;
input.OOS_start=OOS_start;
input.OOSF_end=OOSF_end;
input.nobs_min=nobs_min;

input.grp_pairs=grp_pairs;
input.series_idx=series_idx;
input.gcode=gcode;
input.srs_pairs_use_all=srs_pairs_use_all;
input.fredmd=fredmd;
input.tcode=tcode;
input.ntrials=ntrials;

% -------------------------------------------------------------------------
% PERFORM FORECASTING EXERCISES

fprintf('\n');
disp('RUNNING FORECASTING EXERCISES');

% Loop through group pairs
parfor gp=1:GP
    
    % Display group pairing number
    display(['Group pairing ',num2str(gp),' of ',num2str(GP)]);

     % Call function that performs forecasting exercises 
    [IMSF_gp,DMSF_gp,act_gp,P_IMS_gp,P_DMS_gp,F_IMS_diff_gp,act_diff_gp,Ecvu_gp]=bivar_fcast(input,gp);
    
    % Save output
    IMSF(:,:,:,:,:,gp,:)=IMSF_gp;
    DMSF(:,:,:,:,:,gp,:)=DMSF_gp;
    act(:,:,:,:,:,gp,:)=act_gp;
    
    P_IMS(:,:,:,:,:,gp,:)=P_IMS_gp;
    P_DMS(:,:,:,:,:,gp,:)=P_DMS_gp;
    
    IMSF_diff(:,:,:,:,:,gp,:)=F_IMS_diff_gp;
    act_diff(:,:,:,:,:,gp,:)=act_diff_gp;
    
    Ecvu(:,:,:,:,:,gp,:)=Ecvu_gp;
    
    
end   

% =========================================================================
% PART 3: SAVE OUTPUT

disp('SAVING RESULTS');

% Put into structure for saving
S.IMSF=IMSF;
S.DMSF=DMSF;
S.act=act;
S.P_IMS=P_IMS;
S.P_DMS=P_DMS;
S.IMSF_diff=IMSF_diff;
S.IMSF_diff_u=IMSF_diff(:,:,:,strcmp(cond_options,'none'),:,:,:);
S.act_diff=act_diff;
S.Ecvu=Ecvu;

S.fdates=fdates;
S.horizons=horizons;
S.cond_options=cond_options;
S.sample_start=sample_start;
S.grp_pairs=grp_pairs;
S.lags=lags;

% .mat file name
switch sample_start
    case eom('1/31/1959')
        yr=59;
    case eom('1/31/1984')
        yr=84;
end
matfilename=[outfolder,'\bivar_start',num2str(yr)];
save(matfilename,'S','-v7.3');

