function [MSEr,MSEr1,MSEr2,PCtstat,better1,better2]=get_stats(F1,F2,act,horizons)
% =========================================================================
% DESCRIPTION
% This function computes various statistics from the MSEs of two models.
% These statistics are as follows.
%   1) Ratios of MSEs from model 2 over MSEs from model 1 (MSEr)
%   2) Ratios of MSEs from model 1 over MSEs from model 1 with first lag
%   structure (MSEr1)
%   3) Ratios of MSEs from model 2 over MSEs from model 1 with first lag
%   structure (MSEr2)
%   4) t-statistics from testing null of MSE1-MSE2 equals 0 where MSE1 is
%   the MSE from model 1 and MSE2 is the MSE from model 2 using a
%   Newey-West HAC with a lag length set to ceil(1.5*h) (PCtstat)
%
% It also creates dummies identifying which t-statistics are significant in
% favor of each model (better1 and better2)
%
% INPUTS
%   F1 = forecasts from model 1
%   F2 = forecasts from model 2
%   act = actuals corresponding to the forecasts in F1 and F2
%   horizons = set of horizons considered
%
% =========================================================================
% FUNCTION

% Check that inputs F1, F2, and act have the same dimensions
assert(sum(size(F1)~=size(F2))==0);
assert(sum(size(F1)~=size(act))==0);

% Check that inputs F1 and F2 have no more than 8 dimensions each
assert(length(size(F1))<=9);

% Get number of lag structures
sizeF=size(F1);
H=sizeF(2);
L=sizeF(3);
C=sizeF(4);
K=sizeF(5);

% Compute squared errors from each model
SE1=(F1-act).^2;
SE2=(F2-act).^2;

% Check that missing values in SE1 match up with missing values in SE2 (and
% vice versa)
assert(sum(sum(sum(sum(sum(sum(sum(sum(isnan(SE1(~isnan(SE2)))))))))))==0);
assert(sum(sum(sum(sum(sum(sum(sum(sum(isnan(SE2(~isnan(SE1)))))))))))==0);

% Compute mean squared errors
MSE1=nanmean(SE1,1);
MSE2=nanmean(SE2,1);

% Get base MSE, the MSE of model 1 with first type of lag structure
base_MSE=repmat(MSE1(:,:,1,:,:,:,:,:,:),1,1,L,1,1,1,1,1);

% Ratio of mean squared errors, model 2 over model 1
MSEr=MSE2./MSE1;

% Ratio of mean squared errors, model 1 over base and model 2 over base
MSEr1=MSE1./base_MSE;
MSEr2=MSE2./base_MSE;

% Get t-stat for equal MSE
diff=SE1-SE2;

% Preallocate memory for pairwise-comparison t-stat
PCtstat=NaN(1,H,L,C,K,size(diff,6),size(diff,7),size(diff,8),size(diff,9));

% Get t-stats calcualted with Newey-West standard errors
for h_idx=1:H

    h=horizons(h_idx);
    
    % Lag length for the Newey-West HAC
    nlag=ceil(1.5*h); 

    for l=1:L
        for c=1:C
            for k=1:K
                for j1=1:size(diff,6)
                    for j2=1:size(diff,7)
                        for j3=1:size(diff,8)
                            for bs=1:size(diff,9)
                                y=diff(:,h_idx,l,c,k,j1,j2,j3,bs);
                                allnan=sum(~isnan(y))==0;
                                if ~allnan
                                    
                                    % Run Newey-West adjusted heteroscedastic-serial
                                    % consistent Least-squares regression
                                    % on a constant
                                    y=y(~isnan(y));
                                    x=ones(size(y,1),1);

                                    results=nwest(y,x,nlag); 

                                    % Save t-stats
                                    PCtstat(1,h_idx,l,c,k,j1,j2,j3,bs)=results.tstat;
                                    
                                    % Done for cases when forecasts are
                                    % identical throughout (only when
                                    % comparing VAR-based IMS to ARDL-based
                                    % IMS and lag order is always 0???)
                                    if sum(abs(y))<(1e-10)
                                        PCtstat(:,h_idx,l,c,k,j1,j2,j3,bs)=0;
                                    end
                                    
                                else
                                    assert(k>3);
                                end
                            end
                        end
                    end
                end
            end
        end
    end
end
                                

% Idenify which differences are statistically significant
crit=1.96; % critical value
better1=PCtstat<-crit; % model 1 bette r
better2=PCtstat>crit; % model 2 better

