function [T5,T5g]=getT5_tstat_trivar(stat,rej_down,rej_up,cond_options)
% =========================================================================
% DESCRIPTION
% This function organizes t-stats from the trivariate exercises into
% tables of the form of Table 5 from Marcellino, Stock, and Watson (2006),
% but with the addition of including the including the number of models 
% where the null was rejeceted in favor of each method and the total number
% of models considered.
%
% INPUTS
%   stat = set of t-stats to be organized
%   rej_down = array of logicals indicating rejections of the null in the
%           downward direction (i.e. the t-stat is negative)
%   rej_up = array of logicals indicating rejections of the null in the
%           upward direction (i.e. the t-stat is positive)
%   cond_options = set of conditioning options
%
% OUTPUTS
%   T5 = main results of the t-stats in format of Table 5 from MSW 2006
%   T5g = results from subset of forecasts separated by specific groups
%
% =========================================================================

% Get table type
%   type 1: no rejections (rej_down and rej_up are both empty)
%   type 2: rejections identified but directions are not (rej_down is
%             non-empty but rej_up is empty)
%   type 3: rejections identified and so is the direction of each rejection
%             (rej_down and rej_up are both non-empty)
if isempty(rej_down)
    if isempty(rej_up)
        type=1;
    else
        error('First set of rejections must be non-empty if second set of rejections is empty.');
    end
else
    if isempty(rej_up)
        type=2;
    else
        type=3;
    end
end

% Check that MSEr has right number of dimensions
assert(length(size(stat))==8);
if type>=2
    assert(sum(size(stat)~=size(rej_down))==0)
end
if type==3
    assert(sum(size(stat)~=size(rej_up))==0)
end

% Get desired dimensions
[~,H,L,C,K,~,~,~]=size(stat);

% Number of models total
if type>=2
    mtotal=~isnan(rej_down);
end

% Number of rows in table
switch type
    case 1
        nrow=6;
    case 2
        nrow=8;
    case 3
        nrow=9;
end

% Preallocate memory for table 5
T5=NaN(nrow*L,H,C);
T5g=NaN(nrow*L,H,C,K/2);

% Create table 5
for l=1:L
    for h_idx=1:H
        for c=1:C
            
            % Main results
            if strcmp(cond_options{c},'last1') || strcmp(cond_options{c},'full1')
                MSEr_i=stat(1,h_idx,l,c,1:K,:,:,:);

                if type>=2
                    rej_down1_i=rej_down(1,h_idx,l,c,1:K,:,:,:);
                    mtotal_i=mtotal(1,h_idx,l,c,1:K,:,:,:);
                end
                if type==3
                    rej_up1_i=rej_up(1,h_idx,l,c,1:K,:,:,:);
                end
                
                
            else
                MSEr_i=stat(1,h_idx,l,c,1:(K/2),:,:,:);
                
                if type>=2
                    rej_down1_i=rej_down(1,h_idx,l,c,1:(K/2),:,:,:);
                    mtotal_i=mtotal(1,h_idx,l,c,1:(K/2),:,:,:);
                end
                if type==3
                    rej_up1_i=rej_up(1,h_idx,l,c,1:(K/2),:,:,:);
                end
                
                
            end

            MSEr_i_col=MSEr_i(:);
            assert(sum(isnan(MSEr_i_col))==0);
            
            if type>=2
                rej_down1_i_col=rej_down1_i(:);
                assert(sum(isnan(rej_down1_i_col))==0);
                
                mtotal_i_col=mtotal_i(:);
                assert(sum(isnan(mtotal_i_col))==0);
            end
            if type==3
                rej_up1_i_col=rej_up1_i(:);
                assert(sum(isnan(rej_up1_i_col))==0);
            end
            
        
            T5((1+(l-1)*nrow):(1+(l-1)*nrow+5),h_idx,c)=[mean(MSEr_i_col);prctile(MSEr_i_col,[10,25,50,75,90])'];
            if type==2
                T5((1+(l-1)*nrow+6),h_idx,c)=sum(rej_down1_i_col);
                T5(l*nrow,h_idx,c)=sum(mtotal_i_col);
            elseif type==3
                T5((1+(l-1)*nrow+6):(1+(l-1)*nrow+7),h_idx,c)=[sum(rej_down1_i_col),sum(rej_up1_i_col)];
                T5(l*nrow,h_idx,c)=sum(mtotal_i_col);
            end
            
            
            % Group results
            for k=1:(K/2)
                if strcmp(cond_options{c},'last1') || strcmp(cond_options{c},'full1')
                    MSEr_ik=stat(1,h_idx,l,c,[k,k+(K/2)],:,:,:);
                    
                    if type>=2
                        rej_down1_ik=rej_down(1,h_idx,l,c,[k,k+(K/2)],:,:,:);
                        mtotal_ik=mtotal(1,h_idx,l,c,[k,k+(K/2)],:,:,:);
                    end
                    if type==3
                        rej_up1_ik=rej_up(1,h_idx,l,c,[k,k+(K/2)],:,:,:);
                    end
                    
                else
                    MSEr_ik=stat(1,h_idx,l,c,k,:,:,:);
                    
                    if type>=2
                        rej_down1_ik=rej_down(1,h_idx,l,c,k,:,:,:);
                        mtotal_ik=mtotal(1,h_idx,l,c,k,:,:,:);
                    end
                    if type==3
                        rej_up1_ik=rej_up(1,h_idx,l,c,k,:,:,:);
                    end
                    
                end  
                
                MSEr_ik_col=MSEr_ik(:);
                assert(sum(isnan(MSEr_ik_col))==0);
                
                if type>=2
                    rej_down1_ik_col=rej_down1_ik(:);
                    assert(sum(isnan(rej_down1_ik_col))==0);
                    
                    mtotal_ik_col=mtotal_ik(:);
                    assert(sum(isnan(mtotal_ik_col))==0);
                end
                if type==3
                    rej_up1_ik_col=rej_up1_ik(:);
                    assert(sum(isnan(rej_up1_ik_col))==0);
                end
              
                
                T5g((1+(l-1)*nrow):(1+(l-1)*nrow+5),h_idx,c,k)=[mean(MSEr_ik_col);prctile(MSEr_ik_col,[10,25,50,75,90])'];
                
                if type==2
                    T5g((1+(l-1)*nrow+6),h_idx,c,k)=sum(rej_down1_ik_col);
                    T5g(l*nrow,h_idx,c,k)=sum(mtotal_ik_col);
                elseif type==3
                    T5g((1+(l-1)*nrow+6):(1+(l-1)*nrow+7),h_idx,c,k)=[sum(rej_down1_ik_col),sum(rej_up1_ik_col)];
                    T5g(l*nrow,h_idx,c,k)=sum(mtotal_ik_col);
                end
                
                
            end
            
          
        end
    end
end


