function [T5,T5g]=getT5_tstat_bivar(stat,rej_down,rej_up,grp_pairs,ycell,xcell)
% =========================================================================
% DESCRIPTION
% This function organizes t-stats from the bivariate exercises into
% tables of the form of Table 5 from Marcellino, Stock, and Watson (2006),
% but with the addition of including the number of models where the null is 
% rejeceted in favor of each method and the total number of models 
% considered.
%
% INPUTS
%   stat = set of t-stats to be organized
%   rej_down = array of logicals indicating rejections of the null in the
%           downward direction (i.e. the t-stat is negative)
%   rej_up = array of logicals indicating rejections of the null in the
%           upward direction (i.e. the t-stat is positive)
%   grp_pairs = set of all group pairs considered
%   ycell = groups of series being forecast for the subset of resutls
%   xcell = groups of series not being forecast for the subset of results
% E.g. if ycell{i}=[1,2,3,4] and xcell{i}=5, then the ith subset of results
% will be tabulated for forecasts of series in groups 1, 2, 3, 4 where the
% other variable in the system is from group 5.
%
% OUTPUTS
%   T5 = main results of the t-stats in format of Table 5 from MSW 2006
%   T5g = results from subset of forecasts separated by specific groups in
%       the format of Table 5 from MSW
% =========================================================================
% Check that inputs ycell and xcell are specified correctly
assert(iscell(ycell));
assert(iscell(xcell));
assert(size(ycell,1)==size(xcell,1));
assert(size(ycell,2)==1);
assert(size(xcell,2)==1);

% Check that MSEr has right number of dimensions
assert(length(size(stat))==7);

% Check that input stat has no missing values
assert(sum(sum(sum(sum(sum(sum(isnan(stat)))))))==0);

% Get desired dimensions
[~,H,L,C,K,GP,~]=size(stat);

% Number of models total
mtotal=~isnan(rej_down(1,:,:,:,:,:,:));

nrow=9;

% Preallocate memory for table 5
T5=NaN(nrow*L,H,C);
T5g=NaN(nrow*L,H,C,length(ycell));

% Create table 5
for l=1:L
  for h_idx=1:H
    for c=1:C
            
      % Get stat for given lag structure, horizon, and conditioning
      % exercise
      stat_i=stat(1,h_idx,l,c,:,:,:);
      rej_down_i=rej_down(1,h_idx,l,c,:,:,:);
      rej_up_i=rej_up(1,h_idx,l,c,:,:,:);
      mtotal_i=mtotal(1,h_idx,l,c,:,:,:);
      
      % Get stat_i as a column vector
      stat_i_col=stat_i(:);
      rej_down_i_col=rej_down_i(:);
      rej_up_i_col=rej_up_i(:);
      mtotal_i_col=mtotal_i(:);
      
      assert(sum(isnan(stat_i_col))==0);    
      assert(sum(isnan(rej_down_i_col))==0);
      assert(sum(isnan(rej_up_i_col))==0);
      assert(sum(isnan(mtotal_i_col))==0);
      
      T5((1+(l-1)*nrow):(1+(l-1)*nrow+5),h_idx,c)=[mean(stat_i_col);prctile(stat_i_col,[10,25,50,75,90])'];
      T5((1+(l-1)*nrow+6):(1+(l-1)*nrow+7),h_idx,c)=[sum(rej_down_i_col),sum(rej_up_i_col)];
      T5(l*nrow,h_idx,c)=sum(mtotal_i_col);
      
      % Loop through desired group breakdowns of the results
      for g=1:length(ycell)

        % Get group numbers for the dependent variable and independent
        % variable for the desired group breakdown
        y=ycell{g};
        x=xcell{g};

        % Establish array for stats corresponding to the desired group
        % breakdown
        stat_ig=NaN(size(stat_i));
        rej_down_ig=NaN(size(rej_down_i));
        rej_up_ig=NaN(size(rej_up_i));
        mtotal_ig=NaN(size(mtotal_i));

        % Establish array for which stats from the main results to keep
        keep=zeros(size(stat_i));
        
        % Loop through orderings and group pairings
        for k=1:K
          for gp=1:GP
              
            % Determine whether or not the y variables from pairing gp and
            % ordering k are in one of the desired groups
            y_in=ismember(grp_pairs(gp,k),y);
            
            % Determine whether or not the x variables from pairing gp and
            % ordering k are in one of the desired groups
            x_in=ismember(grp_pairs(gp,3-k),x);
            
            % If both of the above conditions are met, the results are to
            % be kept
            if y_in==1 && x_in==1
              keep(:,:,:,:,k,gp,:)=1;
            end
            
          end
        end
        
        % Turn keep into a logical array
        keep=keep==1;

        % Keep desired stats from the main results
        stat_ig(keep)=stat_i(keep);
        rej_down_ig(keep)=rej_down_i(keep);
        rej_up_ig(keep)=rej_up_i(keep);
        mtotal_ig(keep)=mtotal_i(keep);

        % Turn stats into a column vector
        stat_ig_col=stat_ig(:);
        rej_down_ig_col=rej_down_ig(:);
        rej_up_ig_col=rej_up_ig(:);
        mtotal_ig_col=mtotal_ig(:);

        % Fill in table 5 for given lag structure, horizon, conditioning
        % exercise, and group breakdown
        T5g((1+(l-1)*nrow):(1+(l-1)*nrow+5),h_idx,c,g)=[nanmean(stat_ig_col);...
            prctile(stat_ig_col,[10,25,50,75,90])'];
        T5g((1+(l-1)*nrow+6):(1+(l-1)*nrow+7),h_idx,c,g)=[nansum(rej_down_ig_col),nansum(rej_up_ig_col)];
        T5g(l*nrow,h_idx,c,g)=nansum(mtotal_ig_col);
      end
      
    end
  end
end
      
      
      
      
      
      