function [L,U]=getCritVal(tstat,sig_lev)
% =========================================================================
% DESCRIPTION
% This function obtains the critical values for a two-sided test at a given
% significance level based on the distribution of a given set of bootstrap 
% test statistics.
%
% -------------------------------------------------------------------------
% INPUTS
%   tstat = distribution of test statistics
%   sig_level = significance level (e.g. 0.05 indicates a 5 percent
%       significance level
%   
% OUTPUTS
%   L = critical value for the lower tail
%   U = critical value for the upper tail
%
% =========================================================================
% FUNCTION

% Checks
assert(length(size(tstat))==2);
assert(size(tstat,2)==1);
assert(sum(isnan(tstat))==0);
assert(sig_lev>0 && sig_lev<1);

% Produce warning if number of bootstraps is too small
if 1/length(tstat)>sig_lev
    warning('Number of bootstraps is too small; min and max values are returned as the critical values');
    L=min(tstat);
    U=max(tstat);
else

% Sort test statistics
tstat=sort(tstat,1,'ascend');

% -------------------------------------------------------------------------
% LOWER TAIL CRITICAL VALUE
% Fraction of observations less than or equal to each observation
frac_lte=NaN(size(tstat));
for i=1:length(tstat)
    frac_lte(i)=sum(tstat<=tstat(i))/length(tstat);
end

% If exactly sig_lev/2 of the observations are less than or equal to
% one of the observations, that is the lower tail critical value; otherwise
% take the midpoint of the observation with the largest frac_lte smaller
% than sig_lev/2 and the observation with the smallest frac_tle greater
% than sig_lev/2.
if (sum(frac_lte==(sig_lev/2))==1)
    L=tstat(frac_lte==(sig_lev/2));
elseif (sum(frac_lte==(sig_lev/2))==0)
    rowL1=find(frac_lte<(sig_lev/2),1,'last'); 
    rowL2=rowL1+1;
    L=mean(tstat([rowL1,rowL2],1));
else
    error('Something went wrong');
end

% -------------------------------------------------------------------------
% UPPER TAIL CRITICAL VALUE

% Fraction of observations less than or equal to each observation
frac_gte=NaN(size(tstat));
for i=1:length(tstat)
    frac_gte(i)=sum(tstat>=tstat(i))/length(tstat);
end

if (sum(frac_gte==(sig_lev/2))==1)
    U=tstat(frac_gte==(sig_lev/2));
elseif (sum(frac_gte==(sig_lev/2))==0)
    rowU1=find(frac_gte<(sig_lev/2),1,'first'); 
    rowU2=rowU1-1;
    U=mean(tstat([rowU1,rowU2],1));
else
    error('Something went wrong');
end




end



