function [tstat_a,tstat_b]=efficiency_test(UF,act,bivar)
% =========================================================================
% DESCRIPTION
% This function calculates and returns the t-stats associated with bias and
% efficiency tests.
%
% -------------------------------------------------------------------------
% INPUTS
%   UF = unconditional VAR-based IMS forecasts (post stationary-inducing
%       transformation); dimensions depend on whether they are the
%       bivariate or trivariate results (see notes below for details)
%   act = actual values corresponding to the forecasts in UF; same
%       dimensions as UF
%   bivar = switch variable: 1 if bivariate results, 0 if trivariate 
%       results
%
% OUTPUTS
%   tstat_a = t-statistics corresponding to bias test
%   tstat_b = t-statistics corresponding to efficiency test 
%
% -------------------------------------------------------------------------
% SUBFUNCTION
%   run_eff_test() - estimates regression from the bias/efficiency tests
%       and computes t-statistics for the intercept and slope estimates
%
% -------------------------------------------------------------------------
% NOTES
% The dimensions of UF (and also act) depend on whether it contains the
% forecasts for the bivariate exercises or the trivariate exercises.
%   Bivariate Case:
%       UF ~ T x H x L x C x K x GP x SP
%       where
%           T = number of time periods
%           H = number of horizons
%           L = number of lag-selection methods
%           C = number of conditioning exercises
%           K = number of different orderings of the series (2x1=2 in this
%               case)
%           GP = number of group pairs (10 in the case where we have 5
%               groups)
%           SP = number of series pairs (2000 per group)
%   Trivariate Case:
%       UF ~ T x H x L x C x K x N1 x N2 x N3 x nbs
%       where
%           T = number of time periods
%           H = number of horizons
%           L = number of lag-selection methods
%           C = number of conditioning exercises
%           K = number of different orderings of the series (3x2x1=6 in
%               this case, although half are redundant because with the
%               unconditional forecasts the order of the second and third
%               variables is irrelevant)
%           N1 = number of real series (6)
%           N2 = number of nominal series (5)
%           N3 = number of financial series (5)
%           nbs = number of bootstrap samples plus 1 (the extra 1 is for
%               the actual sample)
%
% =========================================================================
% FUNCTION
    
% -------------------------------------------------------------------------
% BIVARIATE MODELS 

if bivar==1
    
    % Check that UF hase 7 dimensions
    assert(length(size(UF))==7);
    
    % Check that UF and act are the same size
    assert(sum(size(UF)~=size(act))==0);
    
    % Get forecast errors
    UE=act-UF;
    
    % Get dimensions of UE
    [~,H,L,C,K,GP,SP]=size(UE);
    
    % Make sure the number of conditioning exercises is 1 (because should
    % only be dealing with unconditional forecasts)
    assert(C==1);
    
    % Preallocate memory for t-statistics
    tstat_a=NaN(1,H,L,C,K,GP,SP);
    tstat_b=NaN(1,H,L,C,K,GP,SP);
    
    % Loop through all dimensions except time
    for gp=1:GP
        for sp=1:SP
            for h_idx=1:H
                for k=1:K
                    for l=1:L
                        
                        % Get unconditional forecasts for the given model
                        % and turn into a row vector
                        yhat=UF(:,h_idx,l,1,k,gp,sp);
                        yhat_row=yhat(:)';
                        
                        % Get realized values corresponding to the above
                        % forecasts and turn into a row vector
                        y=act(:,h_idx,l,1,k,gp,sp);
                        y_row=y(:)';
                        
                        % Call function to estimate model Y2-X2=a+B*X2+e and
                        % get t-stats for a and b
                        [t_a,t_b]=run_eff_test(y_row,yhat_row);
                        
                        % Instert t-stats into output arrays
                        tstat_b(1,h_idx,l,1,k,gp,sp)=t_b;
                        tstat_a(1,h_idx,l,1,k,gp,sp)=t_a;
                        
                    end
                end
            end
        end
    end
    
% -------------------------------------------------------------------------
% TRIVARIATE MODELS

elseif bivar==0
    
    % Check that UF hase 8 dimensions
    assert(length(size(UF))==8 ||length(size(UF))==9);
    
    % Check that UF and act are the same size
    assert(sum(size(UF)~=size(act))==0);
    
    % Get dimensions of UF
    [~,H,L,C,K,N1,N2,N3,nbs]=size(UF);

    % Make sure the number of conditioning exercises is 1 (because should
    % only be dealing with unconditional forecasts)
    assert(C==1);

    % Preallocate memory for t-statistics
    tstat_a=NaN(1,H,L,C,K,N1,N2,N3,nbs);
    tstat_b=NaN(1,H,L,C,K,N1,N2,N3,nbs);
    
    % Loop through all dimensions except time
    for bs=1:nbs
        for i1=1:N1
            for i2=1:N2
                for i3=1:N3
                    for h_idx=1:H
                        for k=1:(K/2)
                            for l=1:L

                                % Get unconditional forecasts for the given
                                % model and turn into a row vector
                                yhat=UF(:,h_idx,l,1,k,i1,i2,i3,bs);
                                yhat_row=yhat(:)';

                                % Get realized values corresponding to the
                                % above forecasts and turn into a row
                                % vector
                                y=act(:,h_idx,l,1,k,i1,i2,i3,bs);
                                y_row=y(:)';

                                % Call function to estimate model Y2=a+B*X2+e
                                % and get t-stats for a and b
                                [t_a,t_b]=run_eff_test(y_row,yhat_row);

                                % Instert t-stats into output arrays
                                tstat_b(1,h_idx,l,1,k,i1,i2,i3,bs)=t_b;
                                tstat_a(1,h_idx,l,1,k,i1,i2,i3,bs)=t_a;
                                
                            end
                        end
                    end
                end
            end
        end
    end

% -------------------------------------------------------------------------
% INPUT MISSPECIFIED
else
    error('Input bivar must equal either 0 or 1');
end

end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SUBFUNCTION
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [t_a,t_b]=run_eff_test(y,yhat)
% =========================================================================
% DESCRIPTION
% This function regresses y minus yhat on yhat and a constant (where y is
% the realized values and yhat is the unconditional forecasts) and returns
% the t-statistics associated with the estimated coefficients. Inputs y and
% yhat are allowed to have missing values as long as they match up with
% each other. These missing values are removed prior to estimating the
% model.
%
% -------------------------------------------------------------------------
% INPUTS
%   y = realized values (must be a row vector)
%   yhat = unconditional forecasts (must be a row vector)
%
% OUTPUTS
%   t_a = t-statistic associated with the intercept estimate 
%   t_b = t-statistic associated with the slope estimate 
% 
% =========================================================================
% FUNCTION

% Check that y and yhat are both row vectors
assert(size(y,1)==1);
assert(length(size(y))==2);
assert(size(yhat,1)==1);
assert(length(size(yhat))==2);

% Check that y and yhat have same number of columns
assert(size(y,2)==size(yhat,2));

% Check that missing values match up between y and yhat
assert(sum(~isnan(yhat(isnan(y))))==0);
assert(sum(~isnan(y(isnan(yhat))))==0);

% Remove missing values from y and yhat
keep=~isnan(yhat);
yhat=yhat(keep);
y=y(keep);

% Subtract yhat from y
uhat=y-yhat;

% Create matrix of predictors (constant and yhat)
X=[ones(1,length(yhat));yhat];

% Estimate coefficients
B=uhat*X'/(X*X');

% Get predicted values and residuals
zhat=B*X;
ehat=uhat-zhat;

% Total variation in x
SSTb=sum((X(2,:)-mean(X(2,:))).^2);

% Standard error from the model
se=(sum(ehat.^2)/(length(uhat)-length(B)))^0.5;

% Standard error of the slope estimate
se_b=se/(SSTb^0.5);

% Standard error of the intercept estimate
se_a=se_b*((mean(X(2,:).^2))^0.5);

% t-statistics for the slope (b) and intercept (a) estimates
t_a=B(1)/se_a;
t_b=B(2)/se_b;


end
                        
                        


                        