function Y=adjust_sample_split(X,fdates,horizons,OOS_start,OOSF_end)
% =========================================================================
% DESCRIPTION
% This function adjusts the sample split by removing undesired
% out-of-sample periods. Specifically, the user selects a desired first
% forecast date and desired last date being forecasted.
%
% -------------------------------------------------------------------------
% INPUTS
%   X         = data over full set of out-of-sample periods
%   fdates    = dates of periods being forecasted (corresponds to first
%               dimension of X)
%   horizons  = array containing values of horizons considered (corresponds
%               to second dimension of X)
%   OOS_start = desired date that first forecast is made; must be in date 
%               number format
%   OOSF_end  = desired date corresponding to the last period being 
%               forecasted; must be in date number format
%
% OUTPUTS
%   Y =         data over desired out-of-sample periods as determined by
%               OOS_start and OOSF_end
%
% -------------------------------------------------------------------------
% NOTES
%   
% Frequency must be monthly.
%
% It is assumed that fdates, OOS_start, and OOSF_end are all either the
% first day of the month or the last day of the month. 
%
% =========================================================================
% FUNCTION

% -------------------------------------------------------------------------
% GET DIMENSIONS OF X

% Dimensions of X
sizeX=size(X);

% Number of out-of-sample periods
T=sizeX(1);

% Number of horizons
H=sizeX(2);

% -------------------------------------------------------------------------
% CHECKS

% Check that X has no more than 8 dimensions
assert(length(sizeX)<=9);

% Check that dimensions of fdates and horizons match up with corresponding
% dimensions in X
assert(length(fdates)==T);
assert(length(horizons)==H);

% Check that fdates is a column vector
assert(size(fdates,2)==1);

% Check that OOS_start and OOSF_end are numbers with length==1
assert(isnumeric(OOS_start));
assert(isnumeric(OOSF_end));
assert(length(OOS_start)==1);
assert(length(OOSF_end)==1);

% Make sure OOSF_end does not occur after the last date of the full set of
% periods being forecasted
assert(OOSF_end<=max(fdates));

% Make sure that OOS_start is no more than one month before the first
% date of the full set of periods being forecasted
OOS_start_vec=datevec(OOS_start);
OOSF_start_vec=OOS_start_vec;
OOSF_start_vec(2)=OOSF_start_vec(2)+1;
if OOSF_start_vec(2)==13
    OOSF_start_vec(1)=OOSF_start_vec(1)+1;
    OOSF_start_vec(2)=1;
end
OOSF_start=datenum(OOSF_start_vec);
assert(OOSF_start>=min(fdates));

% -------------------------------------------------------------------------
% REMOVE UNECESSARY PERIODS

% Keep only periods after OOS_start and up to OOSF_end
keep=fdates>OOS_start & fdates<=OOSF_end;
Y=X(keep,:,:,:,:,:,:,:,:);
fdates=fdates(keep);

% Remove first h-1 forecasts because first forecast period is OOS_start+h
for h_idx=1:H
    h=horizons(h_idx);
    if h>1
        Y(1:(h-1),h_idx,:,:,:,:,:,:,:)=NaN;
    end
end


