function tstat=MSEtest(F,Fu,E,act,cond_options,bivar)
% =========================================================================
% DESCRIPTION
% This function returns the t-statistics associated with the MSE-test.
%
% -------------------------------------------------------------------------
% INPUTS
%   F = forecasts
%   E = expected values of the differences between the conditional
%       forecasts and the unconditional forecasts (full sample)
%   act = actual values corresponding to the forecasts in F
%   cond_options = list of conditioning exercises
%
% OUTPUTS
%   tstat = t-statistics associated with the MSE-test
%
% =========================================================================
% FUNCTION
  
% Make sure number of dimensions in F is at least 7
assert(length(size(F))>=7);

% Make sure first dimension of E has length 1 (estimated using full sample)
assert(size(E,1)==1);

% Repeat E across all time periods in F
E=repmat(E,size(F,1),1,1,1,1,1,1,1,1);

% Make sure that F, E, and act are the same size
assert(sum(size(F)~=size(E))==0);
assert(sum(size(F)~=size(act))==0);

% Get desired dimensions of F
sizeF=size(F);
K=sizeF(5);

% Make sure that one of the conditioning exercises run was the
% unconditional exercise
%assert(sum(strcmp(cond_options,'none'))==1);

% Calculate squared forecast errors
SE=(act-F).^2;
SEu=(act(:,:,:,1,:,:,:,:,:)-Fu).^2;

% Squared forecast errors from the unconditional forecasts
if bivar==1
    SEu=repmat(SEu,1,1,1,length(cond_options),1,1,1);
elseif bivar==0
    SEu=SEu(:,:,:,:,1:(K/2),:,:,:,:);
    SEu=repmat(SEu,1,1,1,length(cond_options),2,1,1,1);
else
    error('Input bivar must equal 0 or 1');
end

% Difference in the squared forecast errors of all the exercises and the
% squared forecast errors of the unconditional forecasts, less the expected
% value of this difference
diff=SE-SEu-E;

% Flip difference around
diff=-diff;

% Compute the average of the above difference (the MSE-test statistic)
stat=nanmean(diff,1);

% Number of forecast errors
n=sum(~isnan(diff),1);

% Compute the standard error of the above difference
stat_se=nanstd(diff,1,1)./(n.^0.5);

% Compute the t-statistic for testing the null that the MSE-test statistic
% equals zero
tstat=stat./stat_se;

% Replace t-statistic with 0 if standard error is 0 (should only happen if
% order is always 0, so unconditional forecast always equals the
% conditional one)
tstat(stat_se<1e-12)=0;

% Set t-statistic to 0 for unconditional exercises
if bivar==1
    tstat(:,:,:,strcmp(cond_options,'none'),:,:,:)=0;
elseif bivar==0
    tstat(:,:,:,strcmp(cond_options,'none'),:,:,:,:)=0;
end


