function x=undoDiff(y,x0,tcode,h)
% =========================================================================
% DESCRIPTION
% This function undoes differencing of a set of series based on order of
% integration of each series. 
%
% -------------------------------------------------------------------------
% INPUTS
%   y     = set of series after stationary-inducing transformation and
%           aggregation by horizon (T x N matrix where N is the number of
%           series and T is the number of time periods); can only be one
%           row if aggregated over multiple periods
%   x0    = the prior two values of the series in y relative to the first
%           row in y before undergoing differencing (a 2 x N matrix); so if
%           the first row of y corresponds to period t, then row 1 of x0
%           corresponds to period t-h-1 and the second row corresponds to
%           period t-h
%   tcode = transformation code number
%               1 => level
%               2 => first difference
%               3 => second difference
%               4 => log
%               5 => first diffence of log
%               6 => second difference of log
%               7 => first difference of growth rate
%   h     = forecast horizon
%
% OUTPUT
%   x     = set of series after undoing any differencing from the
%           stationary-inducing transformation (but does not undo the
%           taking of logs)
%
% =========================================================================
% FUNCTION

% If accumulating the series across multiple periods, can only do one
% period at a time
if h>1
    assert(size(y,1)==1);
end

% Preallocate memory
x=NaN(size(y));

% Loop through series
for i=1:size(y,2)
    
    % Undo differencing based on order of integration
    switch tcode(i)
        
        % I(0) => x(t)=y(t)
        case {1,4}
            x(:,i)=y(:,i);
            
        % I(1) => x(t)=x(t-h)+y(t)
        case {2,5}
            x(1,i)=x0(2,i)+y(1,i);
            if size(y,1)>1
                for t=2:size(y,1)
                    x(t,i)=x(t-1,i)+y(t,i);
                end
            end
          
        % I(2) => x(t)=x(t-h)+h*(x(t-h)-x(t-h-1))+y(t)
        case {3,6}
            x(1,i)=x0(2,i)+h*(x0(2,i)-x0(1,i))+y(1,i);
            if size(y,1)>1
                x(2,i)=2*x(1,i)-x0(2,i)+y(2,i);
                for t=3:size(y,1)
                    x(t,i)=2*x(t-1,i)-x(t-2,i)+y(t,i);
                end
            end
            
    end
end