function [IMSF,DMSF,act,P_IMS,P_DMS,IMSF2,IMSF2u,act2,E]=trivar_fcast(input,w,run_bs)
% =========================================================================
% DESCRIPTION
% This script performs pseudo-out-of-sample forecasting exercises for all
% desired trivariate systems from a given group pairing.
%
% -------------------------------------------------------------------------
% INPUTS
%   input     = structure containing various input variables
%   w         = which worker is running the function (i.e. which block of
%               trivariate systems is being forecast)
%   run_bs    = dummy indicating whether the sample should be
%               bootstrapped
%
% OUTPUTS
%   IMSF      = VAR-based IMS forecasts after undoing transformation
%   DMSF      = ARDL-based DMS forecasts after undoing transformation
%   act       = actual values after undoing first/second differences
%   P_IMS     = lag structure VAR-based IMS models
%   P_DMS     = lag structure of ARDL-based DMS models
%   IMSF2     = VAR-based IMS forecasts of the stationary series
%   IMSF2u    = VAR-based unconditional IMS forecasts of the 
%               stationary series
%   act2      = actual values of the stationary series
%   E         = expected values of the squared conditional forecast errors 
%               less the squared unconditional forecast errors from the 
%               VAR-based IMS models
%
% =========================================================================
% SETUP

% -------------------------------------------------------------------------
% GET INPUT VALUES
% Obtain inputs from input structure

% Number of forecasts
nF=input.nF;

% Data for series in each group
fredmd1_t1=input.fredmd1_t1;
fredmd2_t1=input.fredmd2_t1;
fredmd3_t1=input.fredmd3_t1;
fredmd1_t2=input.fredmd1_t2;
fredmd2_t2=input.fredmd2_t2;
fredmd3_t2=input.fredmd3_t2;

% Group 3 (needed to find index of series TWEXMMTH)
group3=input.group3;

% Transformation numbers for series in each group
tcode1=input.tcode1;
tcode2=input.tcode2;
tcode3=input.tcode3;

% Forecast horizons
horizons=input.horizons;

% Dates corresponding to data observations
dates=input.dates;

% Conditional exercises being performed
cond_options=input.cond_options;

% Lag structures
lags=input.lags;

% First forecast date and last date being forecasted
OOS_start=input.OOS_start;
OOSF_end=input.OOSF_end;

% Minimum number of observations that each model must be estimated with for
% the forecasts for a certain forecast date to be considered
nobs_min=input.nobs_min;

% Block size for bootstrap
blocksize=input.blocksize;

triples=input.triples;
triples_w=triples(:,:,w);
triples_w=triples_w(~isnan(triples_w(:,1)),:);

% -------------------------------------------------------------------------
% GET DIMENSIONS
H=length(horizons);
L=length(lags);
C=length(cond_options);
K=6; 
N1=length(tcode1);
N2=length(tcode2);
N3=length(tcode3);

% -------------------------------------------------------------------------
% PREALLOCATE MEMORY

IMSF=NaN(nF,H,L,C,K,N1,N2,N3);
DMSF=IMSF;
act=IMSF;
P_IMS=IMSF;
P_DMS=IMSF;

IMSF2=IMSF;
IMSF2u=NaN(nF,H,L,1,K,N1,N2,N3);
act2=IMSF;
E=IMSF;


% =========================================================================
% PERFORM FORECASTS

% Identify which series in group 3 is TWEXMMTH (trade-weighted exchange
% rate)
i3_TWEXMMTH=find(strcmp(group3,'TWEXMMTH'),1,'first');

% Loop through each system being considered      
for grp=1:size(triples_w,1)
    
    % Identify the series in the system
    i1=triples_w(grp,1); % number of the series from group 1
    i2=triples_w(grp,2); % number of the series from group 2
    i3=triples_w(grp,3); % number of the series from group 3

    % Display series pairing number
    counter=i3+(i2-1)*N3+(i1-1)*N3*N2;
    display(['Series grouping ',num2str(counter),' of ',num2str(N1*N2*N3)]);

    % Get transformation numbers for series i1, i2, and i3
    tcode_i=[tcode1(i1),tcode2(i2),tcode3(i3)];
    
    % Get series before and after differencing
    x1=[fredmd1_t1(:,i1,:),fredmd2_t1(:,i2,:),fredmd3_t1(:,i3,:)];
    x2_act=[fredmd1_t2(:,i1,:),fredmd2_t2(:,i2,:),fredmd3_t2(:,i3,:)];

    % Loop through lag-selection methods
    for l=1:length(lags)
        
      % Get lag-selection method
      p=lags{l};
      
      % Display lag-selection method
      if isnumeric(p)
        display(['    Lag = ',num2str(p),'; ',num2str(l),' of ' num2str(length(lags))]);
      else
        display(['    Lag = ',p,'; ',num2str(l),' of ' num2str(length(lags))]);  
      end
      
      % If desired, bootstrap sample using residual-based moving block
      % bootsrap
      if run_bs==1
        t0=find(~isnan(sum(x2_act,2)),1,'first');
        x2bs=RMBbootstrap(x2_act(t0:end,:)',p,blocksize)';
        assert(sum(sum(isnan(x2bs)))==0);
        x2=NaN(size(x2_act));
        x2(t0:end,:)=x2bs;
      else
        x2=x2_act;
      end
              
      % Loop through number of horizons considered
      for h_idx=1:H
    
        % Get horizon
        h=horizons(h_idx);

        % Transform series based on horizon
        x2h=accumulate_h(x2,h,tcode_i);
            
        % Get rows corresponding to first date (T0), first date being
        % forecasted (T1), and last date being forecasted (T2)
        [T0,T1,T2]=get_T(x2,h,dates,OOS_start,OOSF_end);
      
        % If running bootstrap, make sure first nonzero observations match
        % up and that last observations are the end of the dataset
        if run_bs==1
            assert(T0==t0)
            assert(T2==size(x2,1));
        end

        % Loop through number of different orderings
        for k=1:6
          
          % Determine order of the series; the first series is the one being
          % forecasted
          order=get_order(k);

          % Put data in order
          y1=x1(:,order,:);
          y2=x2(:,order,:);
          y2h=x2h(:,order,:);
      
          % Put transformation numbers in order
          tcode_ik=tcode_i(order);         

          % Loop through conditional exercises
          for c=1:length(cond_options)

            % Orderings 4, 5, and 6 may be unnecessary depending on the
            % conditioning exercise; proceed only if necessary
            if (k<=3 || strcmp(cond_options{c},'last1') || strcmp(cond_options{c},'full1'))

              % Loop through periods being forecast
              for t=T1:T2
                
                % In-sample values: from T0 to h periods prior to period
                % being forecasted
                y_is=y2(T0:(t-h),1)';
                yh_is=y2h(T0:(t-h),1)';
                X_is=y2(T0:(t-h),2:end)';
                
                % Value of dependent variable in the last two in-sample
                % periods prior to differencing
                y1_last2=y1((t-h-1):(t-h),1)';
                
                % Get future values that forecasts will be conditioned on
                X_cond=get_cond_val(y2,t,h,cond_options{c});
                
                % Actual values of dependent variable after differencing
                y2_act=y2(t,1)'; 
                
                % Values of the dependent variable after differencing in
                % the h periods being forecast
                y2_act_path=y2((t-h+1):t,1); 
                
                % Checks
                assert(sum(isnan(y_is))==0);
                assert(sum(isnan(yh_is((h+1):end)))==0);
                assert(sum(sum(isnan(X_is)))==0);                
                
                % Number of observations required to run all
                % regressions (number of in-sample observations, minus 12
                % for 12 possible lags, minus (h-1) for the horizon (a
                % horizon of 1 is taken into account when subtracting for
                % the lags)
                nobs_low=size(y_is,2)-12-h+1;

                % Only obtain forecasts when number of observations
                % required to run all regressions meets the desired
                % threshold
                if nobs_low>=nobs_min

                  % IMS from VAR
                  [fc_var,fu_var,Pvar,R,Rtilda]=cond_VAR_IMS([y_is;X_is],h,[NaN(1,h);X_cond],p,12);
                  fc_var2=undoDiff(fc_var',[y1_last2;X_is(:,(end-1):end)]',...
                        tcode_ik,1);

                  % DMS
                  [fc_dir,Pdms]=cond_ARDL_DMS(yh_is,y_is,X_is,h,X_cond,p,12);
                  fc_dir2=undoDiff(fc_dir,y1_last2',tcode_ik(1),h);
                  
                  % Undo difference of actuals
                  y1_act=undoDiff(y2_act_path,y1_last2',tcode_ik(1),1);
                  
                  % Save output
                  IMSF(t-T1+h,h_idx,l,c,k,i1,i2,i3)=fc_var2(end,1);
                  assert(size(fc_dir2,1)==1 & size(fc_dir2,2)==1);
                  DMSF(t-T1+h,h_idx,l,c,k,i1,i2,i3)=fc_dir2; %fc_dir2(end,1);
                  
                  act(t-T1+h,h_idx,l,c,k,i1,i2,i3)=y1_act(end,1);
                  
                  P_IMS(t-T1+h,h_idx,l,c,k,i1,i2,i3)=Pvar;
                  P_DMS(t-T1+h,h_idx,l,c,k,i1,i2,i3)=Pdms;

                  IMSF2(t-T1+h,h_idx,l,c,k,i1,i2,i3)=fc_var(1,end);
                  IMSF2u(t-T1+h,h_idx,l,1,k,i1,i2,i3)=fu_var(1,end);
                  act2(t-T1+h,h_idx,l,c,k,i1,i2,i3)=y2_act;
                 
                  iota=[zeros(size(y2,2)*(h-1),1);1;0;0];
                  if ~isempty(Rtilda)
                    E(t-T1+h,h_idx,l,c,k,i1,i2,i3)=-iota'*R*Rtilda'/(Rtilda*Rtilda')*Rtilda*R'*iota;
                  else 
                    assert(strcmp(cond_options{c},'none'));
                    E(t-T1+h,h_idx,l,c,k,i1,i2,i3)=0;
                  end
                  
                else
                    
                    % Check that if the minimum number of observations was
                    % not met, that it was due to TWEXMMTH (trade-weighted
                    % exchange rate) being in the system
                    assert(i3==i3_TWEXMMTH);

                end
              end
            end
          end
        end
      end            
    end 
end

end


% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SUBFUNCTIONS
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function order=get_order(k)
% Get order of the series; first series is the one being forecast; second
% series is the one being conditioned on (if only one series is being
% conditioned on)

switch k
  case 1
    order=[1,2,3];
  case 2
    order=[2,1,3];
  case 3
    order=[3,1,2];
  case 4
    order=[1,3,2];
  case 5
    order=[2,3,1];
  case 6
    order=[3,2,1];
end

end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [T0,T1,T2]=get_T(x2,h,dates,OOS_start,OOSF_end)
% Get rows corresponding to first date (T0), first date being forecast
% (T1), and last date being forecast (T2)

assert(length(dates)==size(x2,1))

T0=max([find(~isnan(x2(:,1)),1,'first'),...
         find(~isnan(x2(:,2)),1,'first'),...
         find(~isnan(x2(:,3)),1,'first')]);
T1=sum(dates<=datenum(OOS_start))+h;
T2=sum(dates<=datenum(OOSF_end));
    

end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function X_cond=get_cond_val(y2,t,h,exercise)

% Get conditional values of variables not being forecasted
X_cond=y2((t-h+1):t,2:end)';

% Remove conditional values based on exercise
%   none = remove all values
%   full = keep all values
%   last = keep values of each variable only in the last period
%   full1 = keep all values of just the first variable
%   last1 = keep the value of the first variable only in the last period
switch exercise
  
  % Keep values of all series just in the last period
  case 'last'
    X_cond(:,1:(end-1))=NaN;
    assert(sum(isnan(X_cond(:,end)))==0);

  % Keep values of all variables; if vintage vh is missing any of the desired
  % values, get them from the next vintage, vh+1
  case 'full'
    assert(sum(sum(isnan(X_cond)))==0);

  % Keep none of the conditional values
  case 'none'
    X_cond(:,:)=NaN;

  % Keep just the value of the first variable in the last period
  case 'last1'
    X_cond(1,1:(end-1))=NaN;
    X_cond(2,:)=NaN;
    assert(~isnan(X_cond(1,end)));
    
  % Keep all values of the first variable ; if vintage vh is missing any of
  % the desired values, get them from the next vintage, vh+1
  case 'full1'
    X_cond(2,:)=NaN;
    assert(sum(isnan(X_cond(1,:)))==0);
end


end