function [y1,y2,y2h]=transform(x,tcode,h)
% =========================================================================
% DESCRIPTION
% This function transforms a set of series based on a transformation number
% and horizon. 
%
% INPUTS
%   x     = set of series (T x N x V matrix where N is the number of
%           series, T is the number of time periods, and V is the number of
%           vintages)
%   tcode = transformation code number
%               1 => level
%               2 => first difference
%               3 => second difference
%               4 => log
%               5 => first diffence of log
%               6 => second difference of log
%               7 => first difference of growth rate
%   h     = forecast horizon
%
% OUTPUTS
%   y1    = series after taking natural log if necessary
%   y2    = series after applying natural log and then first/second
%           differencing if necessary
%   y2h   = series after aggregating first/second differences across the
%           prior h periods
%
% =========================================================================
% FUNCTION

% Preallocate memory
y1=NaN(size(x));
y2=NaN(size(x));
y2h=NaN(size(x));

% Perform transformations based on case for each series in x
for i=1:size(x,2)
    switch tcode(i)
        case 1
            y1(:,i,:)=x(:,i,:);
            y2(:,i,:)=y1(:,i,:);
            y2h(:,i,:)=y2(:,i,:);
        case 2
            y1(:,i,:)=x(:,i,:);
            y2(2:end,i,:)=y1(2:end,i,:)-y1(1:(end-1),i,:);
            y2h((h+1):end,i,:)=y1((h+1):end,i,:)-y1(1:(end-h),i,:);
        case 3
            y1(:,i,:)=x(:,i,:);
            y2(3:end,i,:)=y1(3:end,i,:)-2*y1(2:(end-1),i,:)+y1(1:(end-2),i,:);
            y2h((h+2):end,i,:)=y1((h+2):end,i,:)-y1(2:(end-h),i,:)-h*(y1(2:(end-h),i,:)-y1(1:(end-h-1),i,:));
        case 4
            y1(:,i,:)=log(x(:,i,:));
            y2(:,i,:)=y1(:,i,:);
            y2h(:,i,:)=y2(:,i,:);
        case 5
            y1(:,i,:)=log(x(:,i,:));
            y2(2:end,i,:)=y1(2:end,i,:)-y1(1:(end-1),i,:);
            y2h((h+1):end,i,:)=y1((h+1):end,i,:)-y1(1:(end-h),i,:);
        case 6
            y1(:,i,:)=log(x(:,i,:));
            y2(3:end,i,:)=y1(3:end,i,:)-2*y1(2:(end-1),i,:)+y1(1:(end-2),i,:);
            y2h((h+2):end,i,:)=y1((h+2):end,i,:)-y1(2:(end-h),i,:)-h*(y1(2:(end-h),i,:)-y1(1:(end-h-1),i,:));
        case 7
            y1(2:end,i,:)=x(2:end,i,:)./x(1:(end-1),i,:)-1;
            y2(2:end,i,:)=y1(2:end,i,:)-y1(1:(end-1),i,:);
            y2h((h+1):end,i,:)=y1((h+1):end,i,:)-y1(1:(end-h),i,:);
    end   
end
    