function [IMSF,DMSF,act,P_IMS,P_DMS]=largevar_fcast(input,system)
% =========================================================================
% DESCRIPTION
% This script performs pseudo-out-of-sample forecasting exercises for a
% given system.
%
% -------------------------------------------------------------------------
% INPUTS
%   input  = structure containing a variety of inputs from the main script
%   system = set of variables that make up the system
%
% OUTPUTS
%   IMSF   = set of IMS forecasts for the given system
%   DMSF   = set of DMS forecasts for the given system
%   act    = set of realized values corresponding to the forecasts
%   P_IMS  = number of lags chosen for each IMS model
%   P_DMS  = number of lags chosen for each DMS model
%
% Note that each output variable is a (nF x H x L x C x K) array where
%   nF = number of forecast periods
%   H  = number of horizons
%   L  = number of lag-selection methods
%   C  = number of conditioning exercises
%   K  = number of variables in the system
%
% =========================================================================
% SETUP

% -------------------------------------------------------------------------
% INPUT VARIABLES

% Number of forecasts
nF=input.nF;

% Data for series in each group
fredmd_t1=input.fredmd_t1;
fredmd_t2=input.fredmd_t2;

% Transformation numbers for series in each group
tcode=input.tcode;

% Forecast horizons
horizons=input.horizons;

% Dates corresponding to data observations
dates=input.dates;

% Conditional exercises being performed
cond_options=input.cond_options;

% Lag structures
lags=input.lags;

% First forecast date and last date being forecasted
OOS_start=input.OOS_start;
OOSF_end=input.OOSF_end;

% Minimum number of observations that each model must be estimated with for
% the forecasts for a certain forecast date to be considered
nobs_min=input.nobs_min;

% -------------------------------------------------------------------------
% GET DIMENSIONS

H=length(horizons);
L=length(lags);
C=length(cond_options);
K=length(system); 

% -------------------------------------------------------------------------
% PREALLOCATE MEMORY

IMSF=NaN(nF,H,L,C,K);
DMSF=IMSF;
act=IMSF;
P_IMS=IMSF;
P_DMS=IMSF;

% =========================================================================
% PERFORM FORECASTING EXERCISE

tcode_i=tcode(system);
x1=fredmd_t1(:,system);
x2=fredmd_t2(:,system);

for l=1:length(lags)
  p=lags{l};

  if isnumeric(p)
    display(['    Lag = ',num2str(p),'; ',num2str(l),' of ' num2str(length(lags))]);
  else
    display(['    Lag = ',p,'; ',num2str(l),' of ' num2str(length(lags))]);  
  end

  % Loop through number of horizons considered
  for h_idx=1:H

    % Get horizon
    h=horizons(h_idx);

    % Display horizon     
    %display(['    Horizon h = ',num2str(h),'; ',num2str(h_idx),' of ' num2str(H)]);

    % Transform series based on horizon
    x2h=accumulate_h(x2,h,tcode_i);

    % Get rows corresponding to first date (T0), first date being
    % forecast (T1), and last date being forecast (T2)
    [T0,T1,T2]=get_T(x2,h,dates,OOS_start,OOSF_end);
    
    % Loop through series in system
    for k=1:K
        
      % Get order of series where series k of K is the first series (i.e.
      % the one being forecast)
      order=get_order(k,K);
        
      % Put data in order
      y1=x1(:,order);
      y2=x2(:,order);
      y2h=x2h(:,order);
      
      % Put transformation numbers in order
      tcode_ik=tcode_i(order);      

      % Loop through conditional exercises
      for c=1:C
        
        assert(strcmp(cond_options{c},'none'));

        % Loop through periods being forecast
        for t=T1:T2

          % In-sample values: from T0 to h periods prior to period
          % being forecast
          y_is=y2(T0:(t-h),1)';
          yh_is=y2h(T0:(t-h),1)';
          X_is=y2(T0:(t-h),2:end)';

          % Value of dependent variable in the last two in-sample
          % periods prior to differencing
          y1_last2=y1((t-h-1):(t-h),1)';

          % Get future values that forecasts will be conditioned on
          X_cond=NaN(size(y2,2)-1,h);

          % Actual values of dependent variable before 
          % differencing
          y1_act=y1(t,1)';

          % Checks
          assert(sum(isnan(y_is))==0);
          assert(sum(isnan(yh_is((h+1):end)))==0);
          assert(sum(sum(isnan(X_is)))==0);


          % Number of observations required to run all
          % regressions (number of in-sample observations, minus 12
          % for 12 possible lags, minus (h-1) for the horizon (a
          % horizon of 1 is taken into account when subtracting for
          % the lags)
          nobs_low=size(y_is,2)-12-h+1;

          % Only obtain forecasts when number of observations
          % required to run all regressions meets the desired
          % threshold
          assert(nobs_low>=nobs_min);

          % IMS forecast from VAR
          [fc_var,~,Pvar,~,~]=cond_VAR_IMS([y_is;X_is],h,[NaN(1,h);X_cond],p,12);
          fc_var2=undoDiff(fc_var',[y1_last2;X_is(:,(end-1):end)]',...
                tcode_ik,1);

          % DMS forecast
          [fc_dir,Pdms]=cond_ARDL_DMS(yh_is,y_is,X_is,h,X_cond,p,12);
          fc_dir2=undoDiff(fc_dir,y1_last2',tcode_ik(1),h);
          assert(size(fc_dir2,1)==1 & size(fc_dir2,2)==1);
          
          % Save output
          IMSF(t-T1+h,h_idx,l,c,k)=fc_var2(end,1);
          DMSF(t-T1+h,h_idx,l,c,k)=fc_dir2;
          act(t-T1+h,h_idx,l,c,k)=y1_act;
          P_IMS(t-T1+h,h_idx,l,c,k)=Pvar;
          P_DMS(t-T1+h,h_idx,l,c,k)=Pdms;


        end
      end
    end
  end            
end 

end


% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% SUBFUNCTIONS
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function order=get_order(k,K)
% Obtain order of the series where series k of K is the first series
% (because that is the series being forecast)

if k==1
    order=1:K;
elseif k==K
    order=[K,1:(K-1)];
else
    order=[k,1:(k-1),(k+1):K];
end

end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [T0,T1,T2]=get_T(x2,h,dates,OOS_start,OOSF_end)
% Get rows corresponding to first date (T0), first date being forecast
% (T1), and last date being forecast (T2)

assert(length(dates)==size(x2,1))

T0=find(~isnan(sum(x2,2)),1,'first');
T1=sum(dates<=datenum(OOS_start))+h;
T2=sum(dates<=datenum(OOSF_end));
        

end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
