function bad=find_bad(X,dates,series,maxstart,minend,outfile)
% =========================================================================
% DESCRIPTION
% This function identifies problematic series from a given dataset. A
% series is problematic if at least one of the following pertains to it:
%   a) it is missing any values between first and last observations
%   b) it starts after the maximum start date allowed
%   c) it ends before the minimum end date allowed
%
% -------------------------------------------------------------------------
% INPUTS
%   X = dataset (one series per column)
%   dates = column vector of dates corresponding to observations in X; must
%           be in date number format
%   series = cell array containing series names; must have 1 row only
%   maxstart = date corresponding to the maximum allowable start date for
%          any series; must be in date number format
%   minend = date corresponding to the minimum allowable end date for any
%          series; must be in date number format
%   outfile = name of text file where series names of the bad series are to
%          be written; extension must be '.txt'
%
% OUTPUTS
%   bad = logical array indicating which series in X are problematic

% =========================================================================
% FUNCTION

% Get dimensions of dataset
%   T = number of observations per series
%   N = number of series
[T,N]=size(X);

% Check that the series variable has the correct dimensions
assert(length(size(series))==2);
assert(size(series,1)==1);
assert(size(series,2)==N);

% Check that the date variables are numeric
assert(isnumeric(dates));
assert(isnumeric(maxstart));
assert(isnumeric(minend));

% Check that the date variables are correct dimensions
assert(size(dates,1)==T);
assert(size(dates,2)==1);
assert(length(size(dates))==2);
assert(length(maxstart)==1);
assert(length(minend)==1);

% Assert that outfile extension is .txt
assert(strcmp(outfile((end-3):end),'.txt'));


% Preallocate memory
bad=zeros(1,N);

% Open file
fileID=fopen(outfile,'wt');

% Write header to file
fprintf(fileID,'%6s \n','MAX START DATE:');
fprintf(fileID,'%6s \n',datestr(maxstart,'mmm yyyy'));
fprintf(fileID,'%6s \n','');

fprintf(fileID,'%6s \n','MIN END DATE:');
fprintf(fileID,'%6s \n',datestr(minend,'mmm yyyy'));
fprintf(fileID,'%6s \n','');

fprintf(fileID,'%6s \n','BAD SERIES:');

% Display variables
fprintf('\n');
disp(['Max Start Date: ',datestr(maxstart,'mmm yyyy')]);
fprintf('\n');
disp(['Min End Date: ',datestr(minend,'mmm yyyy')]);
fprintf('\n');
disp('Bad Series:');

% Loop through each series
for n=1:N
    
    % Find row number of first and last observations
    firstobs=find(~isnan(X(:,n)),1,'first');
    lastobs=find(~isnan(X(:,n)),1,'last');
    
    % A series is problematic if one of the following is true
    %   a) it is missing any values between first and last observations
    %   b) it starts after the maximum start date allowed
    %   c) it ends before the minimum end date allowed
    if sum(isnan(X(firstobs:lastobs,n)))>0
        bad(n)=1;
        output=[num2str(sum(bad)),') ',series{n},' -- missing at least one observation'];
        fprintf(fileID,'%6s \n',output);
        disp(['  ',output]);
    elseif dates(firstobs)>maxstart
        bad(n)=1;
        output=[num2str(sum(bad)),') ',series{n},' -- starts after maximum allowable start date'];
        fprintf(fileID,'%6s \n',output);
        disp(['  ',output]);
    elseif dates(lastobs)<minend
        bad(n)=1;
        output=[num2str(sum(bad)),') ',series{n},' -- ends before minimum allowable end date'];
        fprintf(fileID,'%6s \n',output);
        disp(['  ',output]);
    elseif strcmp(series{n},'NONBORRES')
        bad(n)=1;
        output=[num2str(sum(bad)),') ',series{n},' -- series switches to negative'];
        fprintf(fileID,'%6s \n',output);
        disp(['  ',output]);
        
    end
    
end

% Close file
fclose(fileID);

% Turn bad series identifier into a logical array
bad=bad==1;
    

