function [yf_c,yf_u,P,R,Rtilda,IC]=cond_VAR_IMS(y,h,ycond,lag,Pmax)
% =========================================================================
% DESCRIPTION
% This function estimates a VAR via OLS and uses an iterative multistep
% (IMS) approach to construct conditional point forecasts h periods ahead.
% Based on derivation in Jarocinski (2010).
%
% -------------------------------------------------------------------------
% INPUTS
%
%       y         = set of variables included in the model (an N x T matrix 
%                   where N is the number of variables and T is the number
%                   of time periods)
%
%       h         = forecast horizon (i.e. number of periods ahead to be 
%                   forecasted)
%
%       ycond     = conditional values of the series in y used to construct
%                   the forecasts (an N x h matrix); nonmissing values
%                   indicate which values of which series are to be
%                   conditioned on when constructing the forecasts (e.g.
%                   suppose N=3 and h=4 and that ycond takes the following
%                   form
%                               [  NaN    NaN    NaN    NaN]
%                       ycond = [  NaN    NaN    NaN    0.5]
%                               [ -0.1    0.2    0.4    0.1]
%                   then the forecasts will be constructed conditional on
%                   the second variable in y equaling 0.5 four periods from
%                   the forecast origin and the third variable in y
%                   equaling -0.1, 0.2, 0.4, and 0.1 in one, two, three,
%                   and four periods from the forecast origin,
%                   respectively)
%
%       lag       = value indicating how the order (P) of the VAR will be
%                   determined; options:
%                       1) lag is an integer => P=lag
%                       2) lag is the string 'AIC' => P determined by the 
%                          AIC
%                       3) lag is the string 'BIC' => P determined by the 
%                          BIC
%                   note that P is bounded by 0 and Pmax, the next input
%                   described below
%
%       Pmax      = maximum number of lags; regardless of the number of
%                   lags included in the model, this number of observations
%                   will be removed from the beginning of each series prior
%                   to estimation
%
% OUTPUTS
%
%       yf_c      = forecasts of y conditional on values in ycond
%       
%       yf_u      = unconditional forecasts of y
%       
%       P         = order of the VAR
%
%       R         = matrix from equation (6) from Jarocinski (2010) 
%       
%       Rtilda    = rows of R corresponding to conditional errors
%
%       IC        = value of information criterion for all lags considered
%                   when using AIC/BIC
%
% -------------------------------------------------------------------------
% REFERENCES
% 
% Jarocinski, Marek (2010). "Conditional Forecasts and Uncertainty about 
%   Forecast Revisions in Vector Autoregressions," Economics Letters 108, 
%   257-259.
%
% -------------------------------------------------------------------------
% NOTES
%
% Author: Joe McGillicuddy
% Date: 5/9/2017
% Version: 2014a
% Required Toolboxes: None
%
% =========================================================================
% SETUP

% -------------------------------------------------------------------------
% CHECKS

% Get dimensions: N = number of series, T = number of time periods
[N,T]=size(y);

% Check that N<T
assert(N<T);

% Check that ycond ~ N x h
assert(size(ycond,1)==N);
assert(size(ycond,2)==h);

% -------------------------------------------------------------------------
% DETERMINE NUMBER OF LAGS

% Assert that input lag takes on a valid value; use that value to determine
% the method for lag selection:
%   1) lag is an integer between 0 and Pmax => P=lag
%   2) lag is the string 'AIC' => P is determined by AIC
%   3) lag is the string 'BIC' => P is determined by BIC
if isnumeric(lag)
    if lag>=0 && lag<=Pmax && floor(lag)==lag
        P=lag;
        IC=NaN(Pmax+1,1);
    elseif lag<0 && floor(lag)==lag
        error('Number of lags selected is less than zero');
    elseif floor(lag)==lag
        error(['Number of lags selected exeeds allowed limit of ',...
            num2str(Pmax)]);
    else
        error('Input lag_selection must be an integer')
    end 
elseif strcmp(lag,'AIC') || strcmp(lag,'BIC')
    [P,IC]=getIC(y,Pmax,lag);
else
    error('Input lag_selection is misspecified');
end

% -------------------------------------------------------------------------
% OTHER

% Remove Pmax observations from y
yt=y(:,(Pmax+1):end);

% New number of time periods 
T2=size(yt,2);

% =========================================================================
% CASE: ORDER GREATER THAN ZERO

if P>0

    % ---------------------------------------------------------------------
    % SETUP
    
    % Get lags of y (note that Pmax observations are removed regardless of 
    % the value of P)
    ylags=NaN(N*P,T-Pmax);
    for l=1:P
        ylags((1+N*(l-1)):(N*l),:)=y(:,(Pmax+1-l):(end-l));
    end

    % Matrix of predictors (lags of y and a constant)
    X=[ones(1,T2);ylags];

    % ---------------------------------------------------------------------
    % ESTIMATE MODEL
    % y=B*X+C*epsilon

    % Estimate coefficients matrix B via OLS
    B=yt*X'/(X*X');
    
    % Separate components of B (see equation (1) from Jarocinski (2010)):
    %   B = [c,Phi_1,...,Phi_P] 
    c=B(:,1); % constants
    Phi=B(:,2:end); % Phi's

    % Fitted values
    yhat=B*X;

    % Residuals
    e=yt-yhat;
    
    % Cholesky decomposition
    C=chol(e*e'/(T2-(N*P)-1),'lower');
    
    % orthogonalized shocks: e = C*epsilon
    epsilon=C\e;

    % ---------------------------------------------------------------------
    % ALTERNATE FORM

    % Central level of yt; see equation (2) from Jarocinski (2010)
    Phi_sum=0;
    for l=1:P
        Phi_sum=Phi_sum+Phi(:,(1+(l-1)*N):(l*N));
    end
    mu=(eye(N)-Phi_sum)\c;

    % zt=deviations of yt from central level mu
    zt=yt-repmat(mu,1,T2);
    zlags=ylags-repmat(mu,P,T2);

    % ztilda(t) is the stacked vector of [z(t);z(t-1);...;z(t-P+1)]
    z_tilda=[zt;zlags(1:(N*(P-1)),:)];

    % F from equation (3) in Jarocinski (2010):
    %   ztilda(t)=F*ztilda(t-1)+G*e(t)
    F=[Phi;[eye(N*(P-1)),zeros(N*(P-1),N)]];

    % Get matrices H and R from equation (6) in Jarocinski (2010):
    %   [z(t+1);...;z(t+h)] = H*ztilda(t) + R*[epsilon(t+1);...;epsilon(t+h)]
    H=NaN(N*h,N*P);
    Psi=NaN(N,N,h+1);
    Psi(:,:,1)=eye(N);
    R=zeros(N*h,N*h);
    for t=1:h
        F_t=F^t;
        Psi(:,:,t+1)=F_t(1:N,1:N);
        H((1+N*(t-1)):(N*t),:)=F_t(1:N,:);
        for s=1:t
            R((1+N*(t-1)):(N*t),(1+N*(s-1)):(N*s))=Psi(:,:,t-s+1)*C;
        end
    end

    % Conditional values of y as a column array to match stacked form used
    % to construct forecasts
    ycond_col=reshape(ycond,N*h,1);

    % Rows corresponding to conditional values in the new derivation (i.e.
    % rows with nonmissing values in ycond_col)
    kappa=~isnan(ycond_col);
    
    % Number of known values
    q=sum(kappa);

    % Create stacked array of conditional values for z by subtracting mu
    % from conditional values of y
    zcond=ycond_col-repmat(mu,h,1);

    % r = conditional errors
    r=zcond(kappa==1)-H(kappa==1,:)*z_tilda(:,end);

    % Rows of R corresponding to conditional errors
    Rtilda=R(kappa==1,:);

    % Singular value decomposition of Rtilda
    [U,D0,V] = svd(Rtilda);

    % Get values necessary to construct conditional shocks
    D=D0(:,1:q);
    V1=V(:,1:q);
    V2=V(:,(q+1):end);

    % Conditional shocks
    epsilon_cond=V1/D*U'*r;

    % -------------------------------------------------------------------------
    % FORECASTS

    % Unconditional forecast of y
    yf_u=H*z_tilda(:,end)+repmat(mu,h,1);

    % Conditional forecast of y
    yf_c=H*z_tilda(:,end)+R*epsilon_cond+repmat(mu,h,1);

    % Reshape
    yf_u=reshape(yf_u,N,h);
    yf_c=reshape(yf_c,N,h);
    
% =========================================================================
% CASE: ORDER EQUALS ZERO

else
    
    
    % Unconditional mean of yt
    mu=mean(yt,2);
    
    % Fitted values (also just the unconditional mean of yt)
    yhat=repmat(mu,1,size(yt,2));
    
    % Residuals
    e=yt-yhat;
    
    % Cholesky decomposition
    C=chol(e*e'/(T2-1),'lower');
    
    % Matrix R
    R=zeros(N*h,N*h);
    for t=1:h
        R((1+N*(t-1)):(N*t),(1+N*(t-1)):(N*t))=C;
    end
    
    % Conditional values of y as a column array to match stacked form used
    % to construct forecasts
    ycond_col=reshape(ycond,N*h,1);

    % Rows corresponding to conditional values in the new derivation
    kappa=~isnan(ycond_col);
    
    % Number of known values
    q=sum(kappa);

    % Create stacked array of conditional values for z by subtracting mu
    % from conditional values of y
    zcond=ycond_col-repmat(mu,h,1);
    
    % r = conditional errors
    r=zcond(kappa==1);
    
    % Rows of R corresponding to conditional errors
    Rtilda=R(kappa==1,:);
    
    % Singular value decomposition of Rtilda
    [U,D0,V] = svd(Rtilda);

    % Get values necessary to construct conditional shocks
    D=D0(:,1:q);
    V1=V(:,1:q);
    V2=V(:,(q+1):end);

    % Conditional shocks
    epsilon_cond=V1/D*U'*r;
    
    % -------------------------------------------------------------------------
    % FORECASTS
    
    % Unconditional forecasts are just the unconditional mean of yt
    yf_u=repmat(mean(yt,2),1,h);
    
    % Conditional forecast
    yf_c=R*epsilon_cond+repmat(mu,h,1);
    
    % Reshape
    yf_c=reshape(yf_c,N,h);
    
end

end

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [Pbest,IC]=getIC(y,Pmax,method)
% =========================================================================
% DESCRIPTION
% This function returns the optimal number of lags (between 0 and Pmax) for
% the model described above based on either the AIC or BIC as specified by
% the user.
%
% -------------------------------------------------------------------------
% INPUTS
%
%       y         = set of variables included in the model (an N x S matrix 
%                   where N is the number of variables and S is the number
%                   of time periods)
%
%       Pmax      = maximum order of the VAR considered
%
%       method    = string determining which information criterion used to
%                   select the order; options:
%                       1) 'AIC' => Akaike information criterion 
%                       2) 'BIC' => Bayesian information criterion 
%
% OUTPUTS
%
%       P         = order of VAR as selected by the information criterion
%
%       IC        = array containing values of the information criterion
%                   for each possible order of the VAR considered (0 to
%                   Pmax)
%
% =========================================================================
% FUNCTION

% -------------------------------------------------------------------------
% SETUP

% Remove Pmax observations from y
yt=y(:,(Pmax+1):end);

% N = number of series in the VAR, T = number of observations per series
% used when estimating the model
[N,T]=size(yt);

% -------------------------------------------------------------------------
% CREATE LAGS
% Create Pmax lags of y
ylags_all=NaN(N*Pmax,T);
for l=1:Pmax
    ylags_all((1+N*(l-1)):(N*l),:)=y(:,(Pmax+1-l):(end-l));
end

% -------------------------------------------------------------------------
% CALCULATE VALUES OF INFORMATION CRITERION

% Preallocate memory for values of the information criterion
IC=NaN(Pmax+1,1);

% Loop through possible number of lags
for P=0:Pmax
    
    if P==0
        ylags=[];
    else
        % Include P lags of y as predictors
        ylags=ylags_all(1:(N*P),:);
    end
    
    % Create matrix of predictors (constant plus the lags of y)
    X=[ones(1,T);ylags];
    
    % Solve equation y=B*X
    B=yt*X'/(X*X');
    
    % Residuals
    ehat=yt-B*X;
    
    % Number of parameters estimated
    k=size(B,1)*size(B,2);
    
    % Calculate value of AIC or BIC depending on selected method
    switch method
        case 'AIC'
            IC(P+1)=T*log(det((1/T)*(ehat*ehat')))+2*k;
        case 'BIC'
            IC(P+1)=T*log(det((1/T)*(ehat*ehat')))+k*log(T);
    end
    
    
end

% -------------------------------------------------------------------------
% OPTIMAL NUMBER OF LAGS
% Find the number of lags that minimizes the information criterion
Pbest=find(IC==min(IC),1,'first')-1;


end








