function [IMSF,DMSF,act,P_IMS,P_DMS,IMSF_diff,act_diff,Ecvu]=bivar_fcast(input,gp)
% =========================================================================
% DESCRIPTION
% This script performs pseudo-out-of-sample forecasting exercises for all
% desired bivariate systems from a given group pairing.
%
% -------------------------------------------------------------------------
% INPUTS
%   input  = structure containing a variety of inputs from the main script
%   gp     = group pairing number
%
% OUTPUTS
%   IMSF   = set of IMS forecasts for the given systems
%   DMSF   = set of DMS forecasts for the given systems
%   act    = set of realized values corresponding to the forecasts
%   P_IMS  = number of lags chosen for each IMS model
%   P_DMS  = number of lags chosen for each DMS model
%   IMSF_diff = set of IMS forecasts for the given systems before undoing
%            any differencing
%   act_diff = set of realized values before undoing any differencing
%            corresponding to the forecasts
%   Ecvu   = expectation of the difference between the conditional and the
%            unconditional squared forecast errors
%
% Note that each output variable is a (nF x H x L x C x K x 1 x ntrials)
% array where
%   nF = number of forecast periods
%   H  = number of horizons
%   L  = number of lag-selection methods
%   C  = number of conditioning exercises
%   K  = number of variables in the system
%   ntrials = number of trials per group pairing
%
% =========================================================================
% SETUP

% -------------------------------------------------------------------------
% GET INPUT VALUES
% Obtain inputs from input structure

% Number of forecasts
nF=input.nF;

% Forecast horizons
horizons=input.horizons;

% Dates corresponding to data observations
dates=input.dates;

% Conditional exercises being performed
cond_options=input.cond_options;

% Lag structures
lags=input.lags;

% First forecast date and last date being forecasted
OOS_start=input.OOS_start;
OOSF_end=input.OOSF_end;

% Minimum number of observations that each model must be estimated with for
% the forecasts for a certain forecast date to be considered
nobs_min=input.nobs_min;

grp_pairs=input.grp_pairs;
series_idx=input.series_idx;
gcode=input.gcode;
srs_pairs_use_all=input.srs_pairs_use_all;
fredmd=input.fredmd;
tcode=input.tcode;
ntrials=input.ntrials;

% -------------------------------------------------------------------------
% GET DIMENSIONS

H=length(horizons);
L=length(lags);
C=length(cond_options);
K=2; 

% -------------------------------------------------------------------------
% PREALLOCATE MEMORY

IMSF=NaN(nF,H,L,C,K,1,ntrials);
DMSF=IMSF;
act=IMSF;

P_IMS=IMSF;
P_DMS=IMSF;

IMSF_diff=IMSF;
act_diff=IMSF;

Ecvu=IMSF;

% =========================================================================
% PERFORM FORECASTING EXERCISE

% Group numbers of the two groups (a and b) in group pair gp
grp_a=grp_pairs(gp,1); 
grp_b=grp_pairs(gp,2); 

% Series index numbers of all series in group a and group b
grp_a_idx=series_idx(gcode==grp_a);
grp_b_idx=series_idx(gcode==grp_b);
    
% All possible series pairings from group a and group b (using index
% number of each series within its group)
srs_pairs_use=srs_pairs_use_all(:,:,gp);
    
% Loop through series pairs
for sp=1:ntrials
        
  % Display series pair number
  display(['    Series pairing ',num2str(sp),' of ',num2str(ntrials)]);
        
  % Get the series index number for each series from the series pair sp
  idx_a=grp_a_idx(srs_pairs_use(sp,1));
  idx_b=grp_b_idx(srs_pairs_use(sp,2));
        
  % Get data and transformation numbers for each series from the
  % series pair sp
  raw_sp=fredmd(:,[idx_a,idx_b]);
  tcode_sp=tcode([idx_a,idx_b]);
        
  % Loop through horizons
  for h_idx=1:H
      
    % Get horizon
    h=horizons(h_idx);

    % Display horizon     
    display(['    Horizon h = ',num2str(h),'; ',num2str(h_idx),' of ' num2str(H)]);
            
    % Transform series based on horizon
    [x1,x2,x2h]=transform(raw_sp,tcode_sp,h);
            
    % Get rows corresponding to first date (T0), first date being
    % forecast (T1), and last date being forecast (T2)
    T0=max(find(~isnan(x2(:,1)),1,'first'),...
           find(~isnan(x2(:,2)),1,'first'));
    T1=sum(dates<=datenum(OOS_start))+h;
    T2=sum(dates<=datenum(OOSF_end));
            
    % Loop through orderings of the series
    for k=1:K
                
      % Determine eorder of the series; the first series is the
      % one being forecasted
      if k==1
        order=[1,2];
      elseif k==2
        order=[2,1];
      end
                
      % Put series in order
      y1=x1(:,order);
      y2=x2(:,order);
      y2h=x2h(:,order);
                
      % Put transformation numbers in order
      tcode_spk=tcode_sp(order);
                
      % Loop through conditional exercises being performed
      for c=1:C
                    
        % Loop through different lag-selection methods
        for l=1:L
                        
          % Get lag-selection method
          p=lags{l};
                        
          % Loop through periods being forecast
          for t=T1:T2
                            
            % In-sample values: from T0 to h periods prior to period
            % being forecast
            y_is=y2(T0:(t-h),1)';
            yh_is=y2h(T0:(t-h),1)';
            X_is=y2(T0:(t-h),2)';
                
            % Value of variables in the last two in-sample
            % periods prior to differencing
            y1_last2=y1((t-h-1):(t-h),:);
                            
            % Get future values that forecasts will be conditioned on
            X_cond=y2((t-h+1):t,2)';
            if strcmp(cond_options{c},'last')
              X_cond(:,1:(end-1))=NaN;
            elseif strcmp(cond_options{c},'none')
              X_cond(:,:)=NaN;
            end
                
            % Actual values of dependent variable before and after
            % differencing
            y1_act=y1(t,1)';
            y2_act=y2(t,1)';
                            
            % Checks
            assert(sum(isnan(y_is))==0);
            assert(sum(isnan(yh_is((h+1):end)))==0);
            assert(sum(sum(isnan(X_is)))==0);
                            
            % Number of observations required to run all
            % regressions (number of in-sample observations, minus 12
            % for 12 possible lags, minus (h-1) for the horizon; a
            % horizon of 1 is taken into account when subtracting for
            % the lags)
            nobs_low=size(y_is,2)-12-h+1;
                            
            if nobs_low>=nobs_min

              % VAR-based IMS forecast
              [fc_var_stn,~,p_var,R,Rtilda,~]=cond_VAR_IMS([y_is;X_is],h,[NaN(1,h);X_cond],p,12);
              fc_var=undoDiff(fc_var_stn',y1_last2,tcode_spk,1);

              % ARDL-based DMS forecast
              [fc_dms_stn,p_dms,~]=cond_ARDL_DMS(yh_is,y_is,X_is,h,X_cond,p,12);
              fc_dms=undoDiff(fc_dms_stn,y1_last2(:,1),tcode_spk(1),h);

              % Save output
              IMSF(t-T1+h,h_idx,l,c,k,1,sp)=fc_var(end,1);
              DMSF(t-T1+h,h_idx,l,c,k,1,sp)=fc_dms;
              
              act(t-T1+h,h_idx,l,c,k,1,sp)=y1_act;
              
              P_IMS(t-T1+h,h_idx,l,c,k,1,sp)=p_var;
              P_DMS(t-T1+h,h_idx,l,c,k,1,sp)=p_dms;
              
              IMSF_diff(t-T1+h,h_idx,l,c,k,1,sp)=fc_var_stn(1,end);
              act_diff(t-T1+h,h_idx,l,c,k,1,sp)=y2_act;
              
              iota=[zeros(size(y2,2)*(h-1),1);1;0];
              if ~isempty(Rtilda)
                Ecvu(t-T1+h,h_idx,l,c,k,1,sp)=-iota'*R*Rtilda'/(Rtilda*Rtilda')*Rtilda*R'*iota;
              else 
                assert(strcmp(cond_options{c},'none'));
                Ecvu(t-T1+h,h_idx,l,c,k,1,sp)=0;
              end
              
            end              
          end
        end  
      end
    end     
  end
end 

end