function yh=accumulate_h(y,h,tcode)
% =========================================================================
% DESCRIPTION
% This function accumulates the first/second differences of a series across
% the given forecast horizon based on the order of integration.
%
% -------------------------------------------------------------------------
% INPUTS
%   y     = set of series after stationary-inducing transformation (T x N x
%           V matrix where N is the number of series, T is the number of
%           time periods, and V is the number of vintages)
%   h     = forecast horizon
%   tcode = transformation code number
%               1 => level
%               2 => first difference
%               3 => second difference
%               4 => log
%               5 => first diffence of log
%               6 => second difference of log
%
% OUTPUT
%   yh    = set of series after accumulating the first/second differences:
%               I(0) => yh(t)=y(t)
%               I(1) => yh(t)=sum(y(t-h+i)) for i=1,...,h
%               I(2) => yh(t)=sum(sum(y(t-h+j))) for j=1,...,i, for
%                       i=1,...,h
%
% =========================================================================
% FUNCTION

% Number of series
N=size(y,2);

% Preallocate memory
yh=NaN(size(y));

% Loop through each series
for i=1:N
    
    % Perform calculation based on order of integration implied by
    % transformation number
    switch tcode(i)
        
        % I(0)
        case {1,4}
            yh(:,i,:)=y(:,i,:);
            
        % I(1)
        case {2,5,7}
            yh_temp=0;
            for t=1:h
                yh_temp=yh_temp+y((t+1):(end-h+t),i,:);
            end
            yh((h+1):end,i,:)=yh_temp;
        
        % I(2)
        case {3,6}
            yh_temp=0;
            for t=1:h
                for s=1:t
                    yh_temp=yh_temp+y((s+2):(end-h+s),i,:);
                end
            end
            yh((h+2):end,i,:)=yh_temp;
            
            
    end
end