function [ystar,B]=RMBbootstrap(y,p,l)
% =========================================================================
% DESCRIPTION
% This function performs a residual-based moving block bootstrap as
% outlined in section 5 of Bruggemann, Jentsch, and Trenkler (2014).
%
% -------------------------------------------------------------------------
% INPUTS
%       y       = set of variables included in the model (a K x T matrix 
%                 where K is the number of variables and T is the number
%                 of time periods)
%       p       = order of the VAR (an integer)
%       l       = block size (an integer)
%
% OUTPUT
%       ystar   = bootstrap sample (same dimensions as y)
%
% -------------------------------------------------------------------------
% OUTLINE OF FUNCTION
% Step 0: Setup
% Step 1: Fit VAR(p) to the data; save the residuals 'uhat' and the matrix
%         of coefficients 'B'
% Step 2: Bootstrap the residuals via a block bootstrap with block size 'l'
% Step 3: Center the boostrap residuals
% Step 4: Generate the bootstrap sample - recursively apply the VAR
%         estimated in Step 1 to generate fitted values and then add on the
%         bootstrap residuals; the first 'p' actual observations are used  
%         to initiate the recursion
%
% -------------------------------------------------------------------------
% REFERENCES
% Bruggemann, Ralf, Carsten Jentsch, and Carsten Trenkler (2014),
%   "Inference in VARs with Conditional Heteroskedasticity of Unknown 
%   Form," manuscript, University of Mannheim.
%
% -------------------------------------------------------------------------
% NOTES
% Author: Joe McGillicuddy
% Date: 9/19/2017
% MATLAB Version: R2017a
% Required Toolboxes: None
%
% =========================================================================
% FUNCTION

% -------------------------------------------------------------------------
% STEP 0: SETUP

% Get dimensions of y
assert(length(size(y))==2);
[K,T1]=size(y);

% Check that columns are the time dimension
assert(T1>K);

% Check that there are no missing observations
assert(sum(sum(isnan(y)))==0);

% Check inputs 'p' and 'l' are specified correctly
assert(isnumeric(p));
assert(isnumeric(l));
assert(p>0 && floor(p)==p);
assert(l>0 && floor(l)==l);

% -------------------------------------------------------------------------
% STEP 1: FIT VAR
% Fit VAR(p) model to the data and compute residuals

% New time dimension (remove p observations for lags)
T2=T1-p;
assert(T2>K);

% Get lags of y
ylags=NaN(K*p,T2);
for i=1:p
    ylags((1+K*(i-1)):(K*i),:)=y(:,(p+1-i):(end-i));
end

% Matrix of predictors (lags of y and a constant)
X=[ones(1,T2);ylags];

% Remove p observations from beginning of y to match lags
yt=y(:,(p+1):end);

% Estimate coefficients matrix B via OLS
B=yt*X'/(X*X');

% Fitted values
yhat=B*X;

% Residuals
uhat=yt-yhat;

% -------------------------------------------------------------------------
% STEP 2: BLOCK BOOTSTRAP RESIDUALS

% Check that block length is less than T2
assert(l<T2);

% Number of blocks
N=ceil(T2/l);

% Total of T2-l+1 possible blocks; sample N of them with replacement
% with equal probability assigned to each block (note that block indexes
% start at 0)
block_idx=floor((T2-l+1)*rand(1,N));

% Lay the N blocks end-to-end together
uhat_star=NaN(K,N*l);
for i=1:N
    uhat_star(:,(1+(i-1)*l):(i*l))=uhat(:,(1+block_idx(i)):(block_idx(i)+l));
end

% -------------------------------------------------------------------------
% STEP 3: CENTER BOOTSTRAP RESIDUALS

% Preallocate memory for expected value of the bootstrap residuals
Euhat_star=NaN(size(uhat_star));

% Calculate expected value of the bootstrap residuals
for j=0:(N-1)
    for s=1:l
        Euhat_star(:,j*l+s)=(1/(T2-l+1))*sum(uhat(:,s:(s+T2-l)),2);
    end
end

% Centered bootstrap residuals
ustar=uhat_star-Euhat_star;

% Discard last (N*l-T2) values so that we have exactly T2 bootsrap
% residuals
ustar=ustar(:,1:T2);

% -------------------------------------------------------------------------
% STEP 4: GENERATE BOOTSTRAP SAMPLE

% Preallocate memory for boostrap sample (time dimension is T1 instead of
% T2 because it will include the first p values from the original data to
% initiate the bootstrap)
ystar=NaN(K,T1);

% First p values from original data to initialize the bootsrap
ystar(:,1:p)=y(:,1:p);

% Lags of y for first iteration
ylags_t=NaN(K*p,1);
for i=1:p
    ylags_t((1+K*(i-1)):(K*i),:)=ystar(:,p+1-i);
end

% Loop through each time period to generate bootstrap sample
for t=(p+1):T1
    
    % Matrix of predictors
    X_t=[1;ylags_t];
    
    % Get fitted values
    yhat_t=B*X_t;
    
    % Apply bootstrap residuals to fitted values
    ystar(:,t)=yhat_t+ustar(:,t-p);
    
    % Adjust lags for the next time period
    ylags_t=[ystar(:,t);ylags_t(1:(K*(p-1)),:)];

end

