/*********************************************************************

ST_CH.PRC

This file contains the GAUSS procedure ST_CH

    written by

Bruce E. Hansen
Department of Economics
Social Science Building
University of Wisconsin
Madison, WI 53706-1393
bhansen@ssc.wisc.edu
http://www.ssc.wisc.edu/~bhansen/

Note: the procedure calls the procedures
pv_sup, pv_exp, and pv_ave
which are contained in separate ASCII files.

*************************************************************

ST_CH
Procedure to calculate test statistics and p-values for linear
regression models subject to structural change.

Format:

{kest,sup,exp,ave} = st_ch(y,x,t1,t2,h);

Inputs:
y = dependent variable (nx1 vector).
x = indenpendent variables (nxk matrix).
t1 = starting breakpoint percentage, number in (0,1)
     or starting breakpoint index, number in [k,T-k].
t2 = ending breakpoint percentage, number in (0,1)
     or ending breakpoint index, number in [k,T-k].
h  = Heteroskedasticity-Consistent Covariance Matrix dummy
     1 to calculate heteroskedasticity-consistent covariances
     0 to calculate conventional covariances.

Outputs:
kest = Quandt estimate of breakdate
sup  = Andrews/Quandt SupLM test
exp  = Andrews/Ploberger ExpLM test (c=infinity)
ave  = Andrews/Ploberger AveLM test (c=0)

kest is a (k+1)x2 vector.
  Estimates of Breakpoint.
  The first row reports the index of the element (i.e., the 22nd observation),
    while the second reports the index's relative place in the sample
    (i.e., if n=200, it equals 22/200=.11).
  The first element is the estimate of breakpoint obtained by allowing all
    parameter to shift.  The remaining k elements contain the estimates of
    the breakpoint when only a single parameter is allowed to shift.

Each of (sup, exp, ave) are (k+1)x2 matrices.
  The first column contains test statistics, the second contains
    asymptotic p-values calcuated from by the procedures pv_sup,
    pv_exp, and pv_ave.
  The first row reports the joint test for constancy of all regression
    parameters.
  The remaining rows report the tests for partial constancy of each
    regression parameter, separately.

Note:
For Sup test, Andrews (1993) recommends t1 = .15 and t2=.85.
For Exp and Ave tests, Andrews-Ploberger (1994) recommend t1 = .02 and t2=.98.

**************************************************************************/

#include pv_sup.prc;
#include pv_exp.prc;
#include pv_ave.prc;

/*********************************************************/


proc(5) = st_ch(y,x,t1,t2,h,arorder,trendflag,salesflag);
local xx,xxi,e,k,kt,n,sig,xe,xes,ss,lm,m,mi,sn,sn2,jw,n1,n2,lmt,i,
supt,expt,avet,k1,xi,pw,kest,g,mgm,tau1,tau2,psup,pave,pexp,pi0;

  xx = moment(x,0);
  xxi = invpd(xx);
  e = y - x*xxi*(x'y);
  k  = cols(x);
  kt = k-arorder-trendflag-salesflag-1;
  n  = rows(e);
  sig = (e'e)/(n-k);
  if h==1;
    xe = x.*e;
  else;
    xe = x*sqrt(sig);
  endif;
  xes = cumsumc(x.*e);
  mgm = xxi*moment(xe,0)*xxi;
  if t1<1;
    n1 = floor(n*t1);
    n2 = floor(n*t2);
    tau1 = t1;
    tau2 = t2;
  else;
    n1 = t1;
    n2 = t2;
    tau1 = t1/n;
    tau2 = t2/n;
  endif;

  if n1<k;
    "";"";
    "ERROR:  Starting Sample is smaller than Number of Parameters";
    "        You need to select a larger value for t1";
    "";"";
  endif;
  if n2>(n-k);
    "";"";
    "ERROR:  Ending Sample is smaller than Number of Paramters";
    "        You need to select a smaller value for t2";
    "";"";
  endif;

  lm = zeros(n,kt+1);
  m = moment(x[1:n1-1,.],0);
  g = moment(xe[1:n1-1,.],0);

  i = n1; do while i <= n2;
    m = m + moment(x[i,.],0);
    g = g + moment(xe[i,.],0);
    mi = g - m*xxi*g - g*xxi*m + m*mgm*m;
    sn = xes[i,.]';
    //pw = (sn.^2)./diag(mi); //joint test on all parameters
    pw = (sn[2+trendflag:k-arorder-salesflag].^2)./diag(mi[2+trendflag:k-arorder-salesflag,2+trendflag:k-arorder-salesflag]); //joint test on seasonal parameters
    //jw = sn'invpd(mi)*sn; //joint test on all parameters
    jw = sn[2+trendflag:k-arorder-salesflag]'invpd(mi[2+trendflag:k-arorder-salesflag,2+trendflag:k-arorder-salesflag])*sn[2+trendflag:k-arorder-salesflag]; //joint test on seasonal parameters
    lm[i,.] = jw~(pw');
  i = i+1; endo;

  supt = maxc(lm);
  kest = maxindc(lm);
  kest = kest~(kest/n);
  lmt = lm[n1:n2,.];
  expt  = ln(meanc(exp(lmt/2)));
  avet  = meanc(lmt);

  pi0 = 1/(1+sqrt(tau2*(1-tau1)/tau1/(1-tau2)));

  psup = pv_sup(supt[1],kt,pi0);
  pexp = pv_exp(expt[1],kt,pi0);
  pave = pv_ave(avet[1],kt,pi0);

  i = 1; do while i<=kt;
    psup = psup|pv_sup(supt[i+1],1,pi0);
    pexp = pexp|pv_exp(expt[i+1],1,pi0);
    pave = pave|pv_ave(avet[i+1],1,pi0);
  i = i + 1; endo;

retp(kest,supt~psup,expt~pexp,avet~pave,lm);
endp;

/******************************************************/


