
/*
**  US deficit sustainability: a new approach based on
**                                   multiple endogenous breaks
**
**  PROGRAM 3
**
**  Program produces all results in the `Model 2' column in Table 1,
**
**
**  Main aspects of program:
**
**     1) Triangular cointegration model with m endogenous breaks
**        assumed (Eqn (10) in paper), with m = 3
**     2) Hybrid Gibbs/Metropolis-Hastings simulation method used,
**        along the lines of the algorithm described in Appendix A
**        in the paper
**     3) Lag length of upto 5 allowed for in code for AR(p) polynomial for
**        the cointegrating error (ut). (p=1 specified, based on a
**        preliminary model selection procedure)
**     4) Program written for the change in G (expenditure) being i.i.d.
**        with an intercept term (as specified by a preliminary
**        model selection procedure)
**
*/


library pgraph,maxlik;
#include maxlik.ext;
graphset;
maxset;
gausset;

_pnotify = 0;

fonts("simplex complex microb simgrma");


/**
**  Loading in deficit data
**/

/*  debt.dat
**
**  The data are quarterly starting in 1947(2) and ending in 1992(3)
**
**  The data are:
**
**          1    real tax revenues
**          2    real government expenditures, inclusive of interest paid on
**               debt
*/

load debt[182,8] = c:\gauss\sbreak\debt.dat;
reven = debt[.,1];
expen = debt[.,2];

   output file = gaelm3.out reset;
   print " ";
   print "Program produces all results in the `Model 2' column in Table 1";
   print " ";
      yall = reven[1:182];
      xall = expen[1:182];
      n = 182;
      beg = 1947+1/4;
      period = 1/4;
      start = 6;
      ntime = n - start;
      time1n = seqa(beg+start*period,period,ntime);
      lowr1 = 100;
      lowr2 = 131;
      lowr3 = 150;
      numrgr1 = 30;
      numrgr2 = 18;
      numrgr3 = 20;
      begst1 = beg+start*period+(lowr1-1)*period;
      begst2 = beg+start*period+(lowr2-1)*period;
      begst3 = beg+start*period+(lowr3-1)*period;

      p11 = 1;  /* p in the paper */

   x   = (xall[start+1:n]);
   xu  = x;
   xl  = (xall[start:n-1]);
   xl2 = (xall[start-1:n-2]);
   xl3 = (xall[start-2:n-3]);
   xl4 = (xall[start-3:n-4]);
   xl5 = (xall[start-4:n-5]);

   y   = (yall[start+1:n]);
   yl  = (yall[start:n-1]);
   yl2 = (yall[start-1:n-2]);
   yl3 = (yall[start-2:n-3]);
   yl4 = (yall[start-3:n-4]);
   yl5 = (yall[start-4:n-5]);


  xlmatr = xl~(xl-xl2)~(xl2-xl3)~(xl3-xl4)~(xl4-xl5);
  xlmatr1 = xlmatr[.,1:p11];

  ylmatr = yl~(yl-yl2)~(yl2-yl3)~(yl3-yl4)~(yl4-yl5);
  ylmatr1 = ylmatr[.,1:p11];

timer1 = seqa(begst1,period,numrgr1);
timer2 = seqa(begst2,period,numrgr2);
timer3 = seqa(begst3,period,numrgr3);
rval1 = zeros(numrgr1,1);
rval2 = zeros(numrgr2,1);
rval3 = zeros(numrgr3,1);

xr1matl = zeros(ntime,p11);
x1r1matl = zeros(ntime,p11);
xr1mtlr = zeros(ntime,p11);
x1r1mtlr = zeros(ntime,p11);

xr2matl = zeros(ntime,p11);
x1r2matl = zeros(ntime,p11);
xr2mtlr = zeros(ntime,p11);
x1r2mtlr = zeros(ntime,p11);

xr3matl = zeros(ntime,p11);
x1r3matl = zeros(ntime,p11);
xr3mtlr = zeros(ntime,p11);
x1r3mtlr = zeros(ntime,p11);

ulmat = zeros(ntime,p11);
ulmatr = zeros(ntime,p11);

xrm = zeros(ntime,ntime);
x1rm = zeros(ntime,ntime);

   col = 1;
   do while col <= ntime;
      xrm[.,col] = x;
      x1rm[.,col] = ones(ntime,1);
      col = col + 1;
   endo;

   xrmat = lowmat(xrm);
   x1rmat = lowmat(x1rm);


      lalpha = -3.0;
      ualpha = 7.5;
      laldif1 = -6.5;
      ualdif1 = 7.0;
      laldif2 = -14.0;
      ualdif2 = 10.0;
      laldif3 = -12.0;
      ualdif3 = 14.0;
      lbeta = -0.4;
      ubeta = 1.7;
      la = -0.1;
      ua = 0.2;
      lbetdif1 = -1.1;
      ubetdif1 = 1.1;
      lbetdif2 = -1.1;
      ubetdif2 = 1.5;
      lbetdif3 = -1.1;
      ubetdif3 = 1.4;
      lphi1 = 0.6501;
      uphi1 = 1.3;
      widthb = 0.01;
      wida = 0.005;
      widthp = 0.02;

      nplotb = ((ubeta - lbeta)/widthb) + 1;
      widtha = (ualpha - lalpha)/nplotb;
      widthad1 = (ualdif1 - laldif1)/nplotb;
      widthad2 = (ualdif2 - laldif2)/nplotb;
      widthad3 = (ualdif3 - laldif3)/nplotb;
      widthbd1 = (ubetdif1 - lbetdif1)/nplotb;
      widthbd2 = (ubetdif2 - lbetdif2)/nplotb;
      widthbd3 = (ubetdif3 - lbetdif3)/nplotb;
      numphi = (uphi1 - lphi1)/widthp + 1;
      nplota = ((ua - la)/wida) + 1;


ts = hsec;


/*
** Generation of values of phi (rho in paper), r and a
** from their respective marginal
** posteriors via the hybrid Gibbs/Metropolis algorithm.
*/

   bur = 100;
   repl = 5000 + bur;
   it = 10;
   nmetrop = 1;
   seed = 223460;

print "Initial No. of Gibbs values          = " repl;
print "No. of iterations per repl.          = " it;
print "No. of Metropolis iterations per it. = " nmetrop;

fphv = zeros(repl,p11);
fav  = zeros(repl,1);
fr1v = zeros(repl,1);
fr2v = zeros(repl,1);
fr3v = zeros(repl,1);

ad1 = zeros(repl,1);
ad2 = zeros(repl,1);
ad3 = zeros(repl,1);
b1  = zeros(repl,1);
bd1 = zeros(repl,1);
bd2 = zeros(repl,1);
bd3 = zeros(repl,1);

phiv = zeros(it,p11);
phi11v = zeros(it,p11);
av   = zeros(it,1);
rr1v = zeros(it,1);
rr2v = zeros(it,1);
rr3v = zeros(it,1);

varhp = zeros(it,1);
fvarhpv = zeros(repl,1);

condr1 = zeros(numrgr1,1);
condr2 = zeros(numrgr2,1);
condr3 = zeros(numrgr3,1);

conda = zeros(nplota,1);
condp = zeros(numphi,1);

conr1 = zeros(numrgr1,1);
conr2 = zeros(numrgr2,1);
conr3 = zeros(numrgr3,1);
conp  = zeros(numphi,1);
cona  = zeros(nplota,1);

aval = zeros(nplota,1);
phi  = zeros(numphi,1);
densps = zeros(numphi,1);

condb = zeros(nplotb,1);

condbd1 = zeros(nplotb,1);
condbd2 = zeros(nplotb,1);
condbd3 = zeros(nplotb,1);

condal = zeros(nplotb,1);

condad1 = zeros(nplotb,1);
condad2 = zeros(nplotb,1);
condad3 = zeros(nplotb,1);

conal   = zeros(nplotb,1);
conad1  = zeros(nplotb,1);
conad2  = zeros(nplotb,1);
conad3  = zeros(nplotb,1);
conb    = zeros(nplotb,1);
conbd1  = zeros(nplotb,1);
conbd2  = zeros(nplotb,1);
conbd3  = zeros(nplotb,1);

alpha = seqa(lalpha,widtha,nplotb);
avd1 = seqa(laldif1,widthad1,nplotb);
avd2 = seqa(laldif2,widthad2,nplotb);
avd3 = seqa(laldif3,widthad3,nplotb);
bv   = seqa(lbeta,widthb,nplotb);
bvd1 = seqa(lbetdif1,widthbd1,nplotb);
bvd2 = seqa(lbetdif2,widthbd2,nplotb);
bvd3 = seqa(lbetdif3,widthbd3,nplotb);

ph = zeros(nmetrop+1,p11);
normp1 = zeros(numphi,1);
counter = 0;
mycount = 0;
integ = 1;


/* Starting values */

k = 1;
do while k <= repl;

   if k == 1;

phiv[1,.] = 0.9;
av[1,.] = 0.06;

rr1v[1] = 107;
rr2v[1] = 146;
rr3v[1] = 154;

   else;

phiv[1,.] = fphv[k-1,.];
av[1,.] = fav[k-1,.];
rr1v[1] = fr1v[k-1];
rr2v[1] = fr2v[k-1];
rr3v[1] = fr3v[k-1];
varhp[1] = fvarhpv[k-1];

   endif;

ranr1 = rndus(it,1,seed);
ranr2 = rndus(it,1,seed);
ranr3 = rndus(it,1,seed);
rana = rndus(it,1,seed);

i = 2;
do while i <= it;


/*
** Generation of phi (rho in paper) via Metropolis-Hastings with
** Normal candidate density
*/

   in = 1;
   do while in <= p11;
      xr1matl[.,in] = zeros(in+rr1v[i-1],1)|xu[rr1v[i-1]+1:ntime-in];
      x1r1matl[.,in] = zeros(in+rr1v[i-1],1)|ones(ntime-(rr1v[i-1]+in),1);
      xr2matl[.,in] = zeros(in+rr2v[i-1],1)|xu[rr2v[i-1]+1:ntime-in];
      x1r2matl[.,in] = zeros(in+rr2v[i-1],1)|ones(ntime-(rr2v[i-1]+in),1);
      xr3matl[.,in] = zeros(in+rr3v[i-1],1)|xu[rr3v[i-1]+1:ntime-in];
      x1r3matl[.,in] = zeros(in+rr3v[i-1],1)|ones(ntime-(rr3v[i-1]+in),1);
      in = in + 1;
   endo;

   lg = 1;
   do while lg <= p11;
         xr1mtlr[.,1] = xr1matl[.,1];
         x1r1mtlr[.,1] = x1r1matl[.,1];
         xr2mtlr[.,1] = xr2matl[.,1];
         x1r2mtlr[.,1] = x1r2matl[.,1];
         xr3mtlr[.,1] = xr3matl[.,1];
         x1r3mtlr[.,1] = x1r3matl[.,1];
         if p11 > 1;
            if lg > 1;
               xr1mtlr[.,lg] = xr1matl[.,lg-1] - xr1matl[.,lg];
               x1r1mtlr[.,lg] = x1r1matl[.,lg-1] - x1r1matl[.,lg];
               xr2mtlr[.,lg] = xr2matl[.,lg-1] - xr2matl[.,lg];
               x1r2mtlr[.,lg] = x1r2matl[.,lg-1] - x1r2matl[.,lg];
               xr3mtlr[.,lg] = xr3matl[.,lg-1] - xr3matl[.,lg];
               x1r3mtlr[.,lg] = x1r3matl[.,lg-1] - x1r3matl[.,lg];
            endif;
         endif;
   lg = lg + 1;
   endo;

f = (x - xl) - av[i-1];

xr1 = xrmat[.,rr1v[i-1]+1];
x1r1 = x1rmat[.,rr1v[i-1]+1];
xr2 = xrmat[.,rr2v[i-1]+1];
x1r2 = x1rmat[.,rr2v[i-1]+1];
xr3 = xrmat[.,rr3v[i-1]+1];
x1r3 = x1rmat[.,rr3v[i-1]+1];

c2 = zeros(8,8);
c3 = zeros(8,1);

proc pcondp(fi);
local yst,xst,onest,xr1st,x1r1st,xr2st,x1r2st,xr3st,x1r3st,c1,c4,ic2,lnconp;

   yst = y - ylmatr1*fi;
   xst = x - xlmatr1*fi;

   onest = ones(ntime,1) - ones(ntime,1)*fi;

   xr1st = xr1 - xr1mtlr*fi;
   x1r1st = x1r1 - x1r1mtlr*fi;
   xr2st = xr2 - xr2mtlr*fi;
   x1r2st = x1r2 - x1r2mtlr*fi;
   xr3st = xr3 - xr3mtlr*fi;
   x1r3st = x1r3 - x1r3mtlr*fi;

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];


c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

lnconp = ln(c4^(-(ntime-8)/2));

retp(lnconp);
endp;

proc pcondpm(fi,z);
local yst,xst,onest,xr1st,x1r1st,xr2st,x1r2st,xr3st,x1r3st,c1,c4,ic2,lnconp;

   yst = y - ylmatr1*fi;
   xst = x - xlmatr1*fi;

   onest = ones(ntime,1) - ones(ntime,1)*fi;
   xr1st = xr1 - xr1mtlr*fi;
   x1r1st = x1r1 - x1r1mtlr*fi;
   xr2st = xr2 - xr2mtlr*fi;
   x1r2st = x1r2 - x1r2mtlr*fi;
   xr3st = xr3 - xr3mtlr*fi;
   x1r3st = x1r3 - x1r3mtlr*fi;

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];


c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

lnconp = ln(c4^(-(ntime-8)/2));

retp(lnconp);
endp;


phstart = phiv[i-1,.]';

__output = 0;
_mlalgr = 6;
_mlcovp = 3;

{ pmle,fmax,g,cov,retcode } = maxlik(y~x~xr1~x1r1~xr2~x1r2~f~
  xlmatr1~ylmatr1~xr1mtlr~x1r1mtlr~xr2mtlr~x1r2mtlr~xr3mtlr~x1r3mtlr,
                       0,&pcondpm,phstart);

h = hessp(&pcondp,pmle);

if p11 == 1;
   if h < 0;
      varh = inv(-h);
   else;
      varh = varh;
   endif;
else;
   if det(h) < 0;
      varh = inv(-h);
   else;
      varh = varh;
   endif;
endif;

count = 0;
im = 2;
ime = 2;
ime2 = 2;
conphi = zeros(nmetrop+1,1);
normphi = zeros(nmetrop+1,1);
ph[im-1,.] = phiv[i-1,.];

stnphi = rndns(nmetrop+1,p11,seed);
stnphie = rndns(1000,p11,seed);
cphi1 = rndus(nmetrop+1,1,seed);

do until im == nmetrop + 2;

/*
** Generation of candidate value and evaluation of the actual and candidate
** densities at it
*/
   pst = chol(varh)*stnphi[im,.]' + pmle;

   phidraw:

   yst = y - ylmatr1*pst;
   xst = x - xlmatr1*pst;
   onest = ones(ntime,1) - ones(ntime,1)*pst;

   xr1st = xr1 - xr1mtlr*pst;
   x1r1st = x1r1 - x1r1mtlr*pst;
   xr2st = xr2 - xr2mtlr*pst;
   x1r2st = x1r2 - x1r2mtlr*pst;

   xr3st = xr3 - xr3mtlr*pst;
   x1r3st = x1r3 - x1r3mtlr*pst;

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];


c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

conphi1 = (c4^(-(ntime-8)/2))*(10^(100));

exph = (-1/2)*(pst-pmle)'*inv(varh)*(pst-pmle);
nc = (2*pi)^(-p11/2)*(det(varh))^(-1/2);
normphi1 = nc*exp(exph);

if exph <= -745;
   ime = ime + 1;
   print " exph <= -745 pst ph[im-1,.] varh pmle";
   print exph pst ph[im-1,.] varh pmle;
   pst = chol(varh)*stnphie[ime,.]' + pmle;
   goto phidraw;
endif;


/*
** Evaluation of the actual and candidate densities at the previous value
*/

   yst = y - ylmatr1*ph[im-1,.]';
   xst = x - xlmatr1*ph[im-1,.]';
/* onest = ones(ntime,1) - (ones(ntime,1))~(zeros(ntime,p11-1))*ph[im-1.,]';*/
   onest = ones(ntime,1) - ones(ntime,1)*ph[im-1,.]';

   xr1st = xr1 - xr1mtlr*ph[im-1,.]';
   x1r1st = x1r1 - x1r1mtlr*ph[im-1,.]';
   xr2st = xr2 - xr2mtlr*ph[im-1,.]';
   x1r2st = x1r2 - x1r2mtlr*ph[im-1,.]';
   xr3st = xr3 - xr3mtlr*ph[im-1,.]';
   x1r3st = x1r3 - x1r3mtlr*ph[im-1,.]';


c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];


c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

conphi[im-1] = (c4^(-(ntime-8)/2))*(10^(100));


expph = (-1/2)*(ph[im-1,.]'-pmle)'*inv(varh)*(ph[im-1,.]'-pmle);
nc = (2*pi)^(-p11/2)*(det(varh))^(-1/2);
normphi[im-1] = nc*exp(expph);

if conphi[im-1] == 0;
   ratio = 1;
else;
   ratio = (conphi1/normphi1)*(normphi[im-1]/conphi[im-1]);
endif;

alph = ones(2,1);
alph[1] = ratio;
alphan = minc(alph);
if cphi1[im] <= alphan;
   ph[im,.] = pst';
   count = count + 1;
else;
   ph[im,.] = ph[im-1,.];
endif;

im = im + 1;
endo;

if k >= bur;
counter = counter + count/nmetrop;
mycount = mycount + 1;
endif;

phiv[i,.] = ph[im-1,.];


/*
** Transformation of phi into phi11 (rho into phi in terms of notation
**                                     in paper)
*/

   if p11 == 1;
      phi11v[i,1] = phiv[i,1];
   else;
      phi11v[i,1] = phiv[i,1] + phiv[i,2];
      phi11v[i,p11] = -phiv[i,p11];
   endif;
   if p11 > 2;
      l = 2;
      do while l < p11;
         phi11v[i,l] = -(phiv[i,l] - phiv[i,l+1]);
         l = l + 1;
      endo;
    endif;


   yst = y - ylmatr1*phiv[i,.]';
   xst = x - xlmatr1*phiv[i,.]';
   onest = ones(ntime,1) - ones(ntime,1)*phiv[i,.]';


   xr1st = xr1 - xr1mtlr*phiv[i,.]';
   x1r1st = x1r1 - x1r1mtlr*phiv[i,.]';
   xr2st = xr2 - xr2mtlr*phiv[i,.]';
   x1r2st = x1r2 - x1r2mtlr*phiv[i,.]';
   xr3st = xr3 - xr3mtlr*phiv[i,.]';
   x1r3st = x1r3 - x1r3mtlr*phiv[i,.]';



/*
**  Generation of r1 via Griddy Gibbs
*/

f = (x - xl) - av[i-1];

m = 1;
do while m <= numrgr1;
   r1 = lowr1+(m-1);
   rval1[m] = lowr1+(m-1);
   xr1 = xrmat[.,r1+1];
   x1r1 = x1rmat[.,r1+1];

   in = 1;
   do while in <= p11;
      xr1matl[.,in] = zeros(in+r1,1)|xu[r1+1:ntime-in];
      x1r1matl[.,in] = zeros(in+r1,1)|ones(ntime-(r1+in),1);
      in = in + 1;
   endo;

   xr1st = xr1 - xr1matl*phi11v[i,.]';
   x1r1st = x1r1 - x1r1matl*phi11v[i,.]';

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];

c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

   condr1[m] = c4^(-(ntime-8)/2);
   m = m + 1;
endo;

cumr1den = cumsumc(condr1)./sumc(condr1);


/**
**  The following selection criterion is employed in order
**  to avoid the possibility of encountering
**  numerical problems when applying the inverse cumulative
**  distribution technique to a highly concentrated
**  (conditional) mass function
**  The program would probably be fine without it
**/


   if maxc(condr1./sumc(condr1)) >= 0.90;
      rr1v[i] = lowr1 + (maxindc(condr1./sumc(condr1)) - 1);
      goto contin1;
   else;
      goto ggibb1;
   endif;


ggibb1:

diffr1 = cumr1den - ranr1[i];
minr1 = minindc(abs(cumr1den - ranr1[i]));

rr1v[i] = lowr1 + (minr1-1);

contin1:

   xr1 = xrmat[.,rr1v[i]+1];
   x1r1 = x1rmat[.,rr1v[i]+1];

   in = 1;
   do while in <= p11;
      xr1matl[.,in] = zeros(in+rr1v[i],1)|xu[rr1v[i]+1:ntime-in];
      x1r1matl[.,in] = zeros(in+rr1v[i],1)|ones(ntime-(rr1v[i]+in),1);
      in = in + 1;
   endo;

   xr1st = xr1 - xr1matl*phi11v[i,.]';
   x1r1st = x1r1 - x1r1matl*phi11v[i,.]';


/*
**  Generation of r2 via Griddy Gibbs
*/


m = 1;
do while m <= numrgr2;
   r2 = lowr2+(m-1);
   rval2[m] = lowr2+(m-1);
   xr2 = xrmat[.,r2+1];
   x1r2 = x1rmat[.,r2+1];

   in = 1;
   do while in <= p11;
      xr2matl[.,in] = zeros(in+r2,1)|xu[r2+1:ntime-in];
      x1r2matl[.,in] = zeros(in+r2,1)|ones(ntime-(r2+in),1);
      in = in + 1;
   endo;

   xr2st = xr2 - xr2matl*phi11v[i,.]';
   x1r2st = x1r2 - x1r2matl*phi11v[i,.]';

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];

c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

   condr2[m] = c4^(-(ntime-8)/2);
   m = m + 1;
endo;

cumr2den = cumsumc(condr2)./sumc(condr2);


   if maxc(condr2./sumc(condr2)) >= 0.95;
      rr2v[i] = lowr2 + (maxindc(condr2./sumc(condr2)) - 1);
      goto contin2;
   else;
      goto ggibb2;
   endif;


ggibb2:

diffr2 = cumr2den - ranr2[i];
minr2 = minindc(abs(cumr2den - ranr2[i]));

rr2v[i] = lowr2 + (minr2-1);

contin2:

   xr2 = xrmat[.,rr2v[i]+1];
   x1r2 = x1rmat[.,rr2v[i]+1];

   in = 1;
   do while in <= p11;
      xr2matl[.,in] = zeros(in+rr2v[i],1)|xu[rr2v[i]+1:ntime-in];
      x1r2matl[.,in] = zeros(in+rr2v[i],1)|ones(ntime-(rr2v[i]+in),1);
      in = in + 1;
   endo;

   xr2st = xr2 - xr2matl*phi11v[i,.]';
   x1r2st = x1r2 - x1r2matl*phi11v[i,.]';


/*
**  Generation of r3 via Griddy Gibbs
*/


m = 1;
do while m <= numrgr3;
   r3 = lowr3+(m-1);
   rval3[m] = lowr3+(m-1);
   xr3 = xrmat[.,r3+1];
   x1r3 = x1rmat[.,r3+1];

   in = 1;
   do while in <= p11;
      xr3matl[.,in] = zeros(in+r3,1)|xu[r3+1:ntime-in];
      x1r3matl[.,in] = zeros(in+r3,1)|ones(ntime-(r3+in),1);
      in = in + 1;
   endo;

   xr3st = xr3 - xr3matl*phi11v[i,.]';
   x1r3st = x1r3 - x1r3matl*phi11v[i,.]';

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];

c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

   condr3[m] = c4^(-(ntime-8)/2);
   m = m + 1;
endo;

cumr3den = cumsumc(condr3)./sumc(condr3);


   if maxc(condr3./sumc(condr3)) >= 0.95;
      rr3v[i] = lowr3 + (maxindc(condr3./sumc(condr3)) - 1);
      goto contin3;
   else;
      goto ggibb3;
   endif;


ggibb3:

diffr3 = cumr3den - ranr3[i];
minr3 = minindc(abs(cumr3den - ranr3[i]));

rr3v[i] = lowr3 + (minr3-1);

contin3:


/*
**  Generation of a (drift term for x) via Griddy Gibbs
*/

   xr3= xrmat[.,rr3v[i]+1];
   x1r3 = x1rmat[.,rr3v[i]+1];

   in = 1;
   do while in <= p11;
      xr3matl[.,in] = zeros(in+rr3v[i],1)|xu[rr3v[i]+1:ntime-in];
      x1r3matl[.,in] = zeros(in+rr3v[i],1)|ones(ntime-(rr3v[i]+in),1);
      in = in + 1;
   endo;

   lg = 1;
   do while lg <= p11;
         xr1mtlr[.,1] = xr1matl[.,1];
         x1r1mtlr[.,1] = x1r1matl[.,1];
         xr2mtlr[.,1] = xr2matl[.,1];
         x1r2mtlr[.,1] = x1r2matl[.,1];
         xr3mtlr[.,1] = xr3matl[.,1];
         x1r3mtlr[.,1] = x1r3matl[.,1];
         if p11 > 1;
            if lg > 1;
               xr1mtlr[.,lg] = xr1matl[.,lg-1] - xr1matl[.,lg];
               x1r1mtlr[.,lg] = x1r1matl[.,lg-1] - x1r1matl[.,lg];
               xr2mtlr[.,lg] = xr2matl[.,lg-1] - xr2matl[.,lg];
               x1r2mtlr[.,lg] = x1r2matl[.,lg-1] - x1r2matl[.,lg];
               xr3mtlr[.,lg] = xr3matl[.,lg-1] - xr3matl[.,lg];
               x1r3mtlr[.,lg] = x1r3matl[.,lg-1] - x1r3matl[.,lg];
            endif;
         endif;
   lg = lg + 1;
   endo;

   xst = x - xlmatr1*phiv[i,.]';

   xr3st = xr3 - xr3matl*phi11v[i,.]';
   x1r3st = x1r3 - x1r3matl*phi11v[i,.]';


m = 1;
do while m <= nplota;
   aval[m] = la+ ((m-1)*wida);
   ef = (x - xl) - aval[m];

c1 = (yst'yst)*(ef'ef) - (yst'ef)^2;

c2[1,1] = (onest'onest)*ef'ef   - (onest'ef)^2;
c2[2,2] = (x1r1st'x1r1st)*ef'ef - (x1r1st'ef)^2;
c2[3,3] = (x1r2st'x1r2st)*ef'ef - (x1r2st'ef)^2;
c2[4,4] = (x1r3st'x1r3st)*ef'ef - (x1r3st'ef)^2;
c2[5,5] = (xst'xst)*ef'ef       - (xst'ef)^2;
c2[6,6] = (xr1st'xr1st)*ef'ef   - (xr1st'ef)^2;
c2[7,7] = (xr2st'xr2st)*ef'ef   - (xr2st'ef)^2;
c2[8,8] = (xr3st'xr3st)*ef'ef   - (xr3st'ef)^2;


c2[1,2] = (onest'x1r1st)*ef'ef - (onest'ef)*(x1r1st'ef);
c2[1,3] = (onest'x1r2st)*ef'ef - (onest'ef)*(x1r2st'ef);
c2[1,4] = (onest'x1r3st)*ef'ef - (onest'ef)*(x1r3st'ef);
c2[1,5] = (onest'xst)*ef'ef    - (onest'ef)*(xst'ef);
c2[1,6] = (onest'xr1st)*ef'ef  - (onest'ef)*(xr1st'ef);
c2[1,7] = (onest'xr2st)*ef'ef  - (onest'ef)*(xr2st'ef);
c2[1,8] = (onest'xr3st)*ef'ef  - (onest'ef)*(xr3st'ef);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*ef'ef - (x1r1st'ef)*(x1r2st'ef);
c2[2,4] = (x1r1st'x1r3st)*ef'ef - (x1r1st'ef)*(x1r3st'ef);
c2[2,5] = (x1r1st'xst)*ef'ef    - (x1r1st'ef)*(xst'ef);
c2[2,6] = (x1r1st'xr1st)*ef'ef  - (x1r1st'ef)*(xr1st'ef);
c2[2,7] = (x1r1st'xr2st)*ef'ef  - (x1r1st'ef)*(xr2st'ef);
c2[2,8] = (x1r1st'xr3st)*ef'ef  - (x1r1st'ef)*(xr3st'ef);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*ef'ef - (x1r2st'ef)*(x1r3st'ef);
c2[3,5] = (x1r2st'xst)*ef'ef   - (x1r2st'ef)*(xst'ef);
c2[3,6] = (x1r2st'xr1st)*ef'ef - (x1r2st'ef)*(xr1st'ef);
c2[3,7] = (x1r2st'xr2st)*ef'ef - (x1r2st'ef)*(xr2st'ef);
c2[3,8] = (x1r2st'xr3st)*ef'ef - (x1r2st'ef)*(xr3st'ef);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*ef'ef   - (x1r3st'ef)*(xst'ef);
c2[4,6] = (x1r3st'xr1st)*ef'ef - (x1r3st'ef)*(xr1st'ef);
c2[4,7] = (x1r3st'xr2st)*ef'ef - (x1r3st'ef)*(xr2st'ef);
c2[4,8] = (x1r3st'xr3st)*ef'ef - (x1r3st'ef)*(xr3st'ef);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];

c2[5,6] = (xst'xr1st)*ef'ef - (xst'ef)*(xr1st'ef);
c2[5,7] = (xst'xr2st)*ef'ef - (xst'ef)*(xr2st'ef);
c2[5,8] = (xst'xr3st)*ef'ef - (xst'ef)*(xr3st'ef);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*ef'ef - (xr1st'ef)*(xr2st'ef);
c2[6,8] = (xr1st'xr3st)*ef'ef - (xr1st'ef)*(xr3st'ef);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*ef'ef - (xr2st'ef)*(xr3st'ef);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*ef'ef  - (yst'ef)*(onest'ef);
c3[2] = (yst'x1r1st)*ef'ef - (yst'ef)*(x1r1st'ef);
c3[3] = (yst'x1r2st)*ef'ef - (yst'ef)*(x1r2st'ef);
c3[4] = (yst'x1r3st)*ef'ef - (yst'ef)*(x1r3st'ef);
c3[5] = (yst'xst)*ef'ef    - (yst'ef)*(xst'ef);
c3[6] = (yst'xr1st)*ef'ef  - (yst'ef)*(xr1st'ef);
c3[7] = (yst'xr2st)*ef'ef  - (yst'ef)*(xr2st'ef);
c3[8] = (yst'xr3st)*ef'ef  - (yst'ef)*(xr3st'ef);

c4 = c1 - c3'ic2*c3;

   conda[m] = (c4^(-(ntime-8)/2))*(ef'ef)^(-(2*(3+1)-1)/2);
   m = m + 1;
endo;

cumaden = cumsumc(conda)./sumc(conda);


diffa = cumaden - rana[i];
mina = minindc(abs(cumaden - rana[i]));

av[i] = aval[mina];

/**
**  Specification of all conditional densities at it(th) iteration
**/

if k >= bur;


   if i == it;

   xr3= xrmat[.,rr3v[i]+1];
   x1r3 = x1rmat[.,rr3v[i]+1];

   in = 1;
   do while in <= p11;
      xr3matl[.,in] = zeros(in+rr3v[i],1)|xu[rr3v[i]+1:ntime-in];
      x1r3matl[.,in] = zeros(in+rr3v[i],1)|ones(ntime-(rr3v[i]+in),1);
      in = in + 1;
   endo;

   lg = 1;
   do while lg <= p11;
         xr1mtlr[.,1] = xr1matl[.,1];
         x1r1mtlr[.,1] = x1r1matl[.,1];
         xr2mtlr[.,1] = xr2matl[.,1];
         x1r2mtlr[.,1] = x1r2matl[.,1];
         xr3mtlr[.,1] = xr3matl[.,1];
         x1r3mtlr[.,1] = x1r3matl[.,1];
         if p11 > 1;
            if lg > 1;
               xr1mtlr[.,lg] = xr1matl[.,lg-1] - xr1matl[.,lg];
               x1r1mtlr[.,lg] = x1r1matl[.,lg-1] - x1r1matl[.,lg];
               xr2mtlr[.,lg] = xr2matl[.,lg-1] - xr2matl[.,lg];
               x1r2mtlr[.,lg] = x1r2matl[.,lg-1] - x1r2matl[.,lg];
               xr3mtlr[.,lg] = xr3matl[.,lg-1] - xr3matl[.,lg];
               x1r3mtlr[.,lg] = x1r3matl[.,lg-1] - x1r3matl[.,lg];
            endif;
         endif;
   lg = lg + 1;
   endo;

   xst = x - xlmatr1*phiv[i,.]';

   xr3st = xr3 - xr3matl*phi11v[i,.]';
   x1r3st = x1r3 - x1r3matl*phi11v[i,.]';

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];

c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];

ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;


bmean = ic2*c3;


/* a */

amean = bmean[1];

c11 = c2[1,1];
c12 = c2[1,2:8];
c21 = c12';

c22 = c2[2:8,2:8];

consa = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invara = c4^(-1)*(c11 - c12*invpd(c22)*c21);


/*  ad1  */

ad1mean = bmean[2];

c11 = c2[2,2];

let v1 = 2;
let v2 = 1 3 4 5 6 7 8;
c12 = submat(c2,v1,v2);
c21 = c12';

let v1 = 1 3 4 5 6 7 8;
let v2 = 1 3 4 5 6 7 8;
c22 = submat(c2,v1,v2);

consad1 = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invarad1 = c4^(-1)*(c11 - c12*invpd(c22)*c21);

/* ad2 */

ad2mean = bmean[3];

c11 = c2[3,3];

let v1 = 3;
let v2 = 1 2 4 5 6 7 8;
c12 = submat(c2,v1,v2);
c21 = c12';

let v1 = 1 2 4 5 6 7 8;
let v2 = 1 2 4 5 6 7 8;
c22 = submat(c2,v1,v2);

consad2 = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invarad2 = c4^(-1)*(c11 - c12*invpd(c22)*c21);

/*  ad3  */

ad3mean = bmean[4];

c11 = c2[4,4];

let v1 = 4;
let v2 = 1 2 3 5 6 7 8;
c12 = submat(c2,v1,v2);
c21 = c12';

let v1 = 1 2 3 5 6 7 8;
let v2 = 1 2 3 5 6 7 8;
c22 = submat(c2,v1,v2);


consad3 = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invarad3 = c4^(-1)*(c11 - c12*invpd(c22)*c21);

/* b1 */

b1mean = bmean[5];

c11 = c2[5,5];

let v1 = 5;
let v2 = 1 2 3 4 6 7 8;
c12 = submat(c2,v1,v2);
c21 = c12';

let v1 = 1 2 3 4 6 7 8;
let v2 = 1 2 3 4 6 7 8;
c22 = submat(c2,v1,v2);


consb1 = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invarb1 = c4^(-1)*(c11 - c12*invpd(c22)*c21);

/*  bd1  */

bd1mean = bmean[6];

c11 = c2[6,6];

let v1 = 6;
let v2 = 1 2 3 4 5 7 8;
c12 = submat(c2,v1,v2);
c21 = c12';

let v1 = 1 2 3 4 5 7 8;
let v2 = 1 2 3 4 5 7 8;
c22 = submat(c2,v1,v2);

consbd1 = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invarbd1 = c4^(-1)*(c11 - c12*invpd(c22)*c21);

/*  bd2 */

bd2mean = bmean[7];

c11 = c2[7,7];

let v1 = 7;
let v2 = 1 2 3 4 5 6 8;
c12 = submat(c2,v1,v2);
c21 = c12';

let v1 = 1 2 3 4 5 6 8;
let v2 = 1 2 3 4 5 6 8;
c22 = submat(c2,v1,v2);


consbd2 = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invarbd2 = c4^(-1)*(c11 - c12*invpd(c22)*c21);

/*  bd3 */

bd3mean = bmean[8];

c11 = c2[8,8];

let v1 = 8;
let v2 = 1 2 3 4 5 6 7;
c12 = submat(c2,v1,v2);
c21 = c12';

let v1 = 1 2 3 4 5 6 7;
let v2 = 1 2 3 4 5 6 7;
c22 = submat(c2,v1,v2);


consbd3 = c4^(-1/2)*det((c11 - c12*invpd(c22)*c21))^(1/2);
invarbd3 = c4^(-1)*(c11 - c12*invpd(c22)*c21);

cons1 = gamma((ntime-7)/2)/(pi^(1/2)*gamma((ntime-8)/2));

l = 1;
do while l <= nplotb;

         condal[l] = cons1*consa*(1 + ((alpha[l] - amean)^2)
                           *invara)^(-(ntime-7)/2);
         condad1[l] = cons1*consad1*(1 + ((avd1[l] - ad1mean)^2)
                           *invarad1)^(-(ntime-7)/2);
         condad2[l] = cons1*consad2*(1 + ((avd2[l] - ad2mean)^2)
                           *invarad2)^(-(ntime-7)/2);
         condad3[l] = cons1*consad3*(1 + ((avd3[l] - ad3mean)^2)
                           *invarad3)^(-(ntime-7)/2);
         condb[l] = cons1*consb1*(1 + ((bv[l] - b1mean)^2)
                           *invarb1)^(-(ntime-7)/2);
         condbd1[l] = cons1*consbd1*(1 + ((bvd1[l] - bd1mean)^2)
                           *invarbd1)^(-(ntime-7)/2);
         condbd2[l] = cons1*consbd2*(1 + ((bvd2[l] - bd2mean)^2)
                           *invarbd2)^(-(ntime-7)/2);
         condbd3[l] = cons1*consbd3*(1 + ((bvd3[l] - bd3mean)^2)
                           *invarbd3)^(-(ntime-7)/2);
l = l + 1;
endo;

conal  = conal + condal;
conad1 = conad1 + condad1;
conad2 = conad2 + condad2;
conad3 = conad3 + condad3;
conb   = conb   + condb;
conbd1 = conbd1 + condbd1;
conbd2 = conbd2 + condbd2;
conbd3 = conbd3 + condbd3;


/* phi (rho in paper) */

t = 1;
do while t <= numphi;
   phi[t] = lphi1 + (t-1)*widthp;
   p = phi[t];

if p11 == 1;

   yst = y - p*yl;
   xst = x - p*xl;
   onest = ones(ntime,1) - p*ones(ntime,1);
   xr1st = xr1 - p*xr1matl;
   x1r1st = x1r1 - p*x1r1matl;
   xr2st = xr2 - p*xr2matl;
   x1r2st = x1r2 - p*x1r2matl;
   xr3st = xr3 - p*xr3matl;
   x1r3st = x1r3 - p*x1r3matl;


else;

   yst = y - ylmatr1*(p|phiv[i,2:p11]');
   xst = x - xlmatr1*(p|phiv[i,2:p11]');
   onest = ones(ntime,1)
            - (ones(ntime,p11))~(zeros(ntime,p11-1))*(p|phiv[i,2:p11]');

   xr1st = xr1 - xr1mtlr*(p|phiv[i,2:p11]');
   x1r1st = x1r1 - x1r1mtlr*(p|phiv[i,2:p11]');
   xr2st = xr2 - xr2mtlr*(p|phiv[i,2:p11]');
   x1r2st = x1r2 - x1r2mtlr*(p|phiv[i,2:p11]');
   xr3st = xr3 - xr3mtlr*(p|phiv[i,2:p11]');
   x1r3st = x1r3 - x1r3mtlr*(p|phiv[i,2:p11]');

endif;

c1 = (yst'yst)*(f'f) - (yst'f)^2;

c2[1,1] = (onest'onest)*f'f   - (onest'f)^2;
c2[2,2] = (x1r1st'x1r1st)*f'f - (x1r1st'f)^2;
c2[3,3] = (x1r2st'x1r2st)*f'f - (x1r2st'f)^2;
c2[4,4] = (x1r3st'x1r3st)*f'f - (x1r3st'f)^2;
c2[5,5] = (xst'xst)*f'f       - (xst'f)^2;
c2[6,6] = (xr1st'xr1st)*f'f   - (xr1st'f)^2;
c2[7,7] = (xr2st'xr2st)*f'f   - (xr2st'f)^2;
c2[8,8] = (xr3st'xr3st)*f'f   - (xr3st'f)^2;


c2[1,2] = (onest'x1r1st)*f'f - (onest'f)*(x1r1st'f);
c2[1,3] = (onest'x1r2st)*f'f - (onest'f)*(x1r2st'f);
c2[1,4] = (onest'x1r3st)*f'f - (onest'f)*(x1r3st'f);
c2[1,5] = (onest'xst)*f'f    - (onest'f)*(xst'f);
c2[1,6] = (onest'xr1st)*f'f  - (onest'f)*(xr1st'f);
c2[1,7] = (onest'xr2st)*f'f  - (onest'f)*(xr2st'f);
c2[1,8] = (onest'xr3st)*f'f  - (onest'f)*(xr3st'f);

c2[2,1] = c2[1,2];
c2[3,1] = c2[1,3];
c2[4,1] = c2[1,4];
c2[5,1] = c2[1,5];
c2[6,1] = c2[1,6];
c2[7,1] = c2[1,7];
c2[8,1] = c2[1,8];


c2[2,3] = (x1r1st'x1r2st)*f'f - (x1r1st'f)*(x1r2st'f);
c2[2,4] = (x1r1st'x1r3st)*f'f - (x1r1st'f)*(x1r3st'f);
c2[2,5] = (x1r1st'xst)*f'f    - (x1r1st'f)*(xst'f);
c2[2,6] = (x1r1st'xr1st)*f'f  - (x1r1st'f)*(xr1st'f);
c2[2,7] = (x1r1st'xr2st)*f'f  - (x1r1st'f)*(xr2st'f);
c2[2,8] = (x1r1st'xr3st)*f'f  - (x1r1st'f)*(xr3st'f);

c2[3,2] = c2[2,3];
c2[4,2] = c2[2,4];
c2[5,2] = c2[2,5];
c2[6,2] = c2[2,6];
c2[7,2] = c2[2,7];
c2[8,2] = c2[2,8];

c2[3,4] = (x1r2st'x1r3st)*f'f - (x1r2st'f)*(x1r3st'f);
c2[3,5] = (x1r2st'xst)*f'f   - (x1r2st'f)*(xst'f);
c2[3,6] = (x1r2st'xr1st)*f'f - (x1r2st'f)*(xr1st'f);
c2[3,7] = (x1r2st'xr2st)*f'f - (x1r2st'f)*(xr2st'f);
c2[3,8] = (x1r2st'xr3st)*f'f - (x1r2st'f)*(xr3st'f);

c2[4,3] = c2[3,4];
c2[5,3] = c2[3,5];
c2[6,3] = c2[3,6];
c2[7,3] = c2[3,7];
c2[8,3] = c2[3,8];

c2[4,5] = (x1r3st'xst)*f'f   - (x1r3st'f)*(xst'f);
c2[4,6] = (x1r3st'xr1st)*f'f - (x1r3st'f)*(xr1st'f);
c2[4,7] = (x1r3st'xr2st)*f'f - (x1r3st'f)*(xr2st'f);
c2[4,8] = (x1r3st'xr3st)*f'f - (x1r3st'f)*(xr3st'f);

c2[5,4] = c2[4,5];
c2[6,4] = c2[4,6];
c2[7,4] = c2[4,7];
c2[8,4] = c2[4,8];


c2[5,6] = (xst'xr1st)*f'f - (xst'f)*(xr1st'f);
c2[5,7] = (xst'xr2st)*f'f - (xst'f)*(xr2st'f);
c2[5,8] = (xst'xr3st)*f'f - (xst'f)*(xr3st'f);

c2[6,5] = c2[5,6];
c2[7,5] = c2[5,7];
c2[8,5] = c2[5,8];

c2[6,7] = (xr1st'xr2st)*f'f - (xr1st'f)*(xr2st'f);
c2[6,8] = (xr1st'xr3st)*f'f - (xr1st'f)*(xr3st'f);

c2[7,6] = c2[6,7];
c2[8,6] = c2[6,8];

c2[7,8] = (xr2st'xr3st)*f'f - (xr2st'f)*(xr3st'f);
c2[8,7] = c2[7,8];


ic2 = invpd(c2);

c3[1] = (yst'onest)*f'f  - (yst'f)*(onest'f);
c3[2] = (yst'x1r1st)*f'f - (yst'f)*(x1r1st'f);
c3[3] = (yst'x1r2st)*f'f - (yst'f)*(x1r2st'f);
c3[4] = (yst'x1r3st)*f'f - (yst'f)*(x1r3st'f);
c3[5] = (yst'xst)*f'f    - (yst'f)*(xst'f);
c3[6] = (yst'xr1st)*f'f  - (yst'f)*(xr1st'f);
c3[7] = (yst'xr2st)*f'f  - (yst'f)*(xr2st'f);
c3[8] = (yst'xr3st)*f'f  - (yst'f)*(xr3st'f);

c4 = c1 - c3'ic2*c3;

   condp[t] = c4^(-(ntime-8)/2);

t = t + 1;
endo;

   etar1 = sumc(condr1);
   etar2 = sumc(condr2);
   etar3 = sumc(condr3);
   etaa = sumc(conda)*wida;

   condrr1 = condr1./etar1;
   condrr2 = condr2./etar2;
   condrr3 = condr3./etar3;
   condaa = conda./etaa;

   conr1 = conr1 + condrr1;
   conr2 = conr2 + condrr2;
   conr3 = conr3 + condrr3;
   cona = cona + condaa;


   etap = sumc(condp)*widthp;
   condpp = condp./etap;
   conp = conp + condpp;


endif;

endif;

i = i + 1;

endo;

/*
** Gibbs values
*/


fphv[k,.] = phiv[it,.];
fav[k,.] = av[it,.];
fr1v[k] = rr1v[it];
fr2v[k] = rr2v[it];
fr3v[k] = rr3v[it];
fvarhpv[k] = varhp[it];

if k/10==integ;
    print " k= " k;
    integ = integ + 1;
endif;


k = k + 1;
endo;

/*
** Choice of "burnout" period
*/


repl = repl - bur + 1;


densr1 = (1/repl)*conr1;
densr2 = (1/repl)*conr2;
densr3 = (1/repl)*conr3;
densa = (1/repl)*cona;
densa1 = (1/repl)*conal;
densad1 = (1/repl)*conad1;
densad2 = (1/repl)*conad2;
densad3 = (1/repl)*conad3;
densb1 = (1/repl)*conb;
densbd1 = (1/repl)*conbd1;
densbd2 = (1/repl)*conbd2;
densbd3 = (1/repl)*conbd3;
densp1 = (1/repl)*conp;

et = hsec - ts;

/*xy(alpha,densa1);
xy(avd1,densad1);
xy(avd2,densad2);
xy(avd3,densad3);
xy(bv,densb1);
xy(bvd1,densbd1);
xy(bvd2,densbd2);
xy(bvd3,densbd3);
xy(aval,densa);
bar(seqa(lowr1,1,numrgr1),densr1);
bar(timer1,densr1);
bar(seqa(lowr2,1,numrgr2),densr2);
bar(timer2,densr2);
bar(seqa(lowr3,1,numrgr3),densr3);
bar(timer3,densr3);
xy(phi,densp1);*/


gmodea1 = alpha[maxindc(densa1)];
gmodead1 = avd1[maxindc(densad1)];
gmodead2 = avd2[maxindc(densad2)];
gmodead3 = avd3[maxindc(densad3)];
gmodeb1 = bv[maxindc(densb1)];
gmodebd1 = bvd1[maxindc(densbd1)];
gmodebd2 = bvd2[maxindc(densbd2)];
gmodebd3 = bvd3[maxindc(densbd3)];
gmodea = aval[maxindc(densa)];
gmoder1 = timer1[maxindc(densr1)];
gmoder2 = timer2[maxindc(densr2)];
gmoder3 = timer3[maxindc(densr3)];
r1mode = rval1[maxindc(densr1)];
r2mode = rval2[maxindc(densr2)];
r3mode = rval3[maxindc(densr3)];
gmodep1 = phi[maxindc(densp1)];

print " ";
print " Gibbs mode for alpha  " gmodea1;
print " Gibbs mode for adiff1 " gmodead1;
print " Gibbs mode for adiff2 " gmodead2;
print " Gibbs mode for adiff3 " gmodead3;
print " Gibbs mode for b1     " gmodeb1;
print " Gibbs mode for bdiff1 " gmodebd1;
print " Gibbs mode for bdiff2 " gmodebd2;
print " Gibbs mode for bdiff3 " gmodebd3;
print " Gibbs mode for x drift" gmodea;
print " Gibbs mode for r1     " r1mode gmoder1;
print " Gibbs mode for r2     " r2mode gmoder2;
print " Gibbs mode for r3     " r3mode gmoder3;
print " Gibbs mode for rho    " gmodep1;
print " ";

i = 1;
do while i <= numphi;
    phi[i] = (lphi1) + (i-1)*widthp;
    if phi[i] < 1;
       densps[i] = densp1[i];
    endif;
    i = i + 1;
endo;

gsprob = sumc(densps)*widthp;

pmean2 = counter/mycount;


print " Final no. of replications after burnout =               " repl;
print " ";
print " Average proportion of accepted Metropolis draws (phi) = " pmean2;



print " ";
print " Probability of stationarity  (Gibbs) =                  " gsprob;
print " ";

print " Time (in minutes) for MCMC computations =               " et/6000;
print " ";

output off;


