// This program implements ML estimation of a one-factor model,
// computes the updated and smoothed estimates of the common factor,
// computes the implicit weights for the updated estimates,
// and returns the Stock-Watson coincident indices (updated and smoothed).
// To calculate the mean of the common factor,
// the input data should NOT be demeaned.

#include <oxstd.h>
#include <oxfloat.h>
#import <maximize>

class MLFA
{
	decl m_mY, m_iP, m_iQ;

	MLFA(const mY, const iP, const iQ);
	guess(const cN, const iP, const iQ);
	DFM(const vTheta, const cN, const iP, const iQ,
		const avBeta, const avPhi_f, const amPhi_u, const amSig_22);
	SSR(const cN, const iP, const iQ, const vBeta, const vPhi_f, const mPhi_u, const mSig_22,
		const amF, const amG, const amH, const amSig_vv);
	KF(const mY, const mF, const mG, const mH, const mSig_vv,
		const avLnl, const avComfac_u);
	loglikelihood(const vTheta, const adLnL, const avScore, const amHess);
	estimate(const mY, const iP, const iQ);
	ssKG(const mF, const mG, const mH, const mSig_vv);
	weight(const iLag, const vTheta, const cN, const iP, const iQ);
	LRweight(const vTheta, const cN, const iP, const iQ);
	comfac_u(const mY, const iP, const iQ, const vTheta);
	smoothing(const mY, const mF, const mG, const mH, const mSig_vv);
	comfac_s(const mY, const iP, const iQ, const vTheta);
}

MLFA::MLFA(const mY, const iP, const iQ)
{
	m_mY = mY;
	m_iP = iP;
	m_iQ = iQ;
}

MLFA::guess(const cN, const iP, const iQ)
{
	decl vbeta = .5 * ones(cN, 1);
	decl vtheta = vbeta;
	if (iP > 0)
	{
		decl vphi_f = vec(zeros(1, iP));
		vtheta = vtheta | vphi_f;
	}
	if (iQ > 0)
	{
		decl vphi_u = vec(zeros(cN, iQ));
		vtheta = vtheta | vphi_u;
	}
	decl vsig_22 = .5 * ones(cN, 1);
	vtheta = vtheta | vsig_22;
	return vtheta;
}

MLFA::DFM(const vTheta, const cN, const iP, const iQ,
	const avBeta, const avPhi_f, const amPhi_u, const amSig_22)
{
	decl cbeta = cN;
	decl cphi_f = iP;
	decl cphi_u = cN * iQ;
	decl cphi = cphi_f + cphi_u;
	decl ctheta = rows(vTheta);

	avBeta[0] = vTheta[0: cbeta - 1];
	avPhi_f[0] = 0;
	if (iP > 0)
	{
		avPhi_f[0] = vTheta[cbeta: cbeta + cphi_f - 1]';
	}
	amPhi_u[0] = 0;
	if (iQ > 0)
	{
		amPhi_u[0] = diag(vTheta[cbeta + cphi_f: cbeta + cphi_f + cN - 1]);
		if (iQ > 1)
		{
			decl i;
			for (i = 2; i <= iQ; ++i)
			{
				amPhi_u[0] = amPhi_u[0]
					~ diag(vTheta[cbeta + cphi_f + (i - 1) * cN: cbeta + cphi_f + i * cN - 1]);
			}
		}
	}
	amSig_22[0] = fabs(diag(vTheta[cbeta + cphi: ctheta - 1]));
}

MLFA::SSR(const cN, const iP, const iQ, const vBeta, const vPhi_f, const mPhi_u, const mSig_22,
	const amF, const amG, const amH, const amSig_vv)
{
	decl csta = 5 + 5 * cN;

	decl mf = zeros(csta, csta);
	mf[1: 4][0: 3] = unit(4);
	mf[5 + cN: 5 + 5 * cN - 1][5: 5 + 4 * cN - 1] = unit(4 * cN);
	if (iP > 0)
	{
		mf[0][0: iP - 1] = vPhi_f;
	}
	if (iQ > 0)
	{
		mf[5: 5 + cN - 1][5: 5 + iQ * cN - 1] = mPhi_u;
	}
	amF[0] = mf;

	decl mg = zeros(csta, 1 + cN);
	mg[0][0] = 1;
	mg[5: 5 + cN - 1][1: 1 + cN - 1] = unit(cN);
	amG[0] = mg;

	decl mh = zeros(cN, csta);
	mh[][0] = vBeta;
	mh[][5: 5 + cN - 1] = unit(cN);
	amH[0] = mh;

	decl msig_vv = unit(1 + cN);
	msig_vv[1: 1 + cN - 1][1: 1 + cN - 1] = mSig_22;
	amSig_vv[0] = msig_vv;
}

MLFA::KF(const mY, const mF, const mG, const mH, const mSig_vv,
	const avLnl, const avComfac_u)
{
	decl cn = rows(mY);
	decl cobs = columns(mY);
	decl csta = 5 + 5 * cn;

	decl vlnl = zeros(cobs, 1);
	decl vcomfac_u = zeros(cobs, 1);
	decl vs_u = zeros(csta, 1);
	decl mp_u = zeros(csta, csta);
	decl t;
	for (t = 0; t < cobs; ++t)
	{
		// Prediction

		decl vs_p = mF * vs_u;
		decl mp_p = mF * mp_u * mF' + mG * mSig_vv * mG';

		// Log-likelihood
		
		decl vy = mY[][t];
		decl ve = vy - mH * vs_p;
		decl msig_ee = mH * mp_p * mH';
		vlnl[t] = - (cn / 2) * log(2 * M_PI)
			- (1 / 2) * log(determinant(msig_ee))
			- (1 / 2) * ve' * invertsym(msig_ee) * ve;

		// Updating

		decl mgain = mp_p * mH' * invertsym(mH * mp_p * mH');
		vs_u = vs_p + mgain * (vy - mH * vs_p);
		mp_u = mp_p - mgain * mH * mp_p;
		vcomfac_u[t] = vs_u[0];
	}
	avLnl[0] = vlnl;
	avComfac_u[0] = vcomfac_u;
}

MLFA::loglikelihood(const vTheta, const adLnL, const avScore, const amHess)
{
	decl my = m_mY;
	decl cn = rows(my);
	decl ip = m_iP;
	decl iq = m_iQ;

	decl vbeta, vphi_f, mphi_u, msig_22;
	DFM(vTheta, cn, ip, iq, &vbeta, &vphi_f, &mphi_u, &msig_22);

	decl mf, mg, mh, msig_vv;
	SSR(cn, ip, iq, vbeta, vphi_f, mphi_u, msig_22, &mf, &mg, &mh, &msig_vv);

	decl vlnl, vcomfac_u;
	KF(my, mf, mg, mh, msig_vv, &vlnl, &vcomfac_u);

	adLnL[0] = meanc(vlnl);
	return 1;
}

MLFA::estimate(const mY, const iP, const iQ)
{
	decl cn = rows(mY);
	decl cobs = columns(mY);

	decl vtheta = guess(cn, iP, iQ);
	decl dfunc;
	MaxControl(-1, 1);
	MaxBFGS(loglikelihood, &vtheta, &dfunc, 0, 1);
//	decl mHess;
//	Num2Derivative(loglikelihood, vtheta, &mHess);
//	print(sqrt(diagonal(invertsym(-cobs * mHess)))');

	decl dlnL = cobs * dfunc;
	decl cpar = cn + iP + cn * iQ + cn;
	decl dAIC = (dlnL - cpar) / cobs;
	decl dBIC = (dlnL - cpar * log(cobs) / 2) / cobs;
	print(iP, iQ, dlnL, dAIC, dBIC);

	return vtheta;
}

MLFA::ssKG(const mF, const mG, const mH, const mSig_vv)
{
	decl csta = rows(mF);
	decl mp_u = zeros(csta, csta);
	decl t, mgain;
	for (t = 0; t < 100; ++t)
	{
		decl mp_p = mF * mp_u * mF' + mG * mSig_vv * mG';
		mgain = mp_p * mH' * invertsym(mH * mp_p * mH');
		mp_u = mp_p - mgain * mH * mp_p;
	}
	return mgain;
}

MLFA::weight(const iLag, const vTheta, const cN, const iP, const iQ)
{
	decl vbeta, vphi_f, mphi_u, msig_22;
	DFM(vTheta, cN, iP, iQ, &vbeta, &vphi_f, &mphi_u, &msig_22);

	decl mf, mg, mh, msig_vv;
	SSR(cN, iP, iQ, vbeta, vphi_f, mphi_u, msig_22, &mf, &mg, &mh, &msig_vv);

	decl mgain = ssKG(mf, mg, mh, msig_vv);

	decl csta = rows(mf);
	decl mweight = zeros(iLag + 1, cN);
	decl j;
	for (j = 0; j <= iLag; ++j)
	{
		decl mw_j = ((unit(csta) - mgain * mh) * mf) ^ j * mgain;
		mweight[j][] = mw_j[0][];
	}
	return mweight / sumr(sumc(mweight));
}

MLFA::LRweight(const vTheta, const cN, const iP, const iQ)
{
	decl vbeta, vphi_f, mphi_u, msig_22;
	DFM(vTheta, cN, iP, iQ, &vbeta, &vphi_f, &mphi_u, &msig_22);

	decl mf, mg, mh, msig_vv;
	SSR(cN, iP, iQ, vbeta, vphi_f, mphi_u, msig_22, &mf, &mg, &mh, &msig_vv);

	decl mgain = ssKG(mf, mg, mh, msig_vv);

	decl csta = rows(mf);
	decl mw = invert(unit(csta) - (unit(csta) - mgain * mh) * mf) * mgain;
	decl vweight = mw[0][]';
	return vweight;
}

MLFA::comfac_u(const mY, const iP, const iQ, const vTheta)
{
	decl cn = rows(mY);

	decl vbeta, vphi_f, mphi_u, msig_22;
	DFM(vTheta, cn, iP, iQ, &vbeta, &vphi_f, &mphi_u, &msig_22);

	decl mf, mg, mh, msig_vv;
	SSR(cn, iP, iQ, vbeta, vphi_f, mphi_u, msig_22, &mf, &mg, &mh, &msig_vv);

	decl vlnl, vcomfac_u;
	KF(mY, mf, mg, mh, msig_vv, &vlnl, &vcomfac_u);

	return vcomfac_u;
}

MLFA::smoothing(const mY, const mF, const mG, const mH, const mSig_vv)
{
	decl cn = rows(mY);
	decl cobs = columns(mY);
	decl csta = 5 + 5 * cn;
	decl vs_u = zeros(csta, 1);
	decl mp_u = zeros(csta, csta);
	decl ms_p = zeros(csta, cobs);
	decl mvechP_p = zeros(csta * (csta + 1) / 2, cobs);
	decl t;
	for (t = 0; t < cobs; ++t)
	{
		// Prediction

		decl vs_p = mF * vs_u;
		decl mp_p = mF * mp_u * mF' + mG * mSig_vv * mG';
		ms_p[][t] = vs_p;
		mvechP_p[][t] = vech(mp_p);

		// Updating

		decl vy = mY[][t];
		decl mgain = mp_p * mH' * invertsym(mH * mp_p * mH');
		vs_u = vs_p + mgain * (vy - mH * vs_p);
		mp_u = mp_p - mgain * mH * mp_p;
	}

	// Smoothing

	decl ms_s = zeros(csta, cobs);
	decl vr = zeros(csta, 1);
	for (t = cobs - 1; t >= 0; --t)
	{
		decl vs_p = ms_p[][t];
		decl mp_p = unvech(mvechP_p[][t]);

		decl vy = mY[][t];
		decl mgain = mp_p * mH' * invertsym(mH * mp_p * mH');
		vr = mH' * invertsym(mH * mp_p * mH') * (vy - mH * vs_p)
			+ (unit(csta) - mH' * mgain') * mF' * vr;
		ms_s[][t] = vs_p + mp_p * vr;
	}
	return ms_s;
}

MLFA::comfac_s(const mY, const iP, const iQ, const vTheta)
{
	decl cn = rows(mY);

	decl vbeta, vphi_f, mphi_u, msig_22;
	DFM(vTheta, cn, iP, iQ, &vbeta, &vphi_f, &mphi_u, &msig_22);

	decl mf, mg, mh, msig_vv;
	SSR(cn, iP, iQ, vbeta, vphi_f, mphi_u, msig_22, &mf, &mg, &mh, &msig_vv);

	decl ms_s = smoothing(mY, mf, mg, mh, msig_vv);
	return ms_s[0][]';
}

main()
{
	decl mY = loadmat("myprg_ox/00b/BCIQ0M4.mat");
	decl vmu_Y = meanc(mY)';
	mY = (mY - vmu_Y') ./ sqrt(varc(mY));
	mY = mY';
	
	decl cn = rows(mY);
	decl iP = 1;
	decl iQ = 3;
	decl iLag = 40;

	decl mlfaobj = new MLFA(mY, iP, iQ);
	decl vtheta = mlfaobj.estimate(mY, iP, iQ);
	decl mweight = mlfaobj.weight(iLag, vtheta, cn, iP, iQ);
	decl vLRweight = mlfaobj.LRweight(vtheta, cn, iP, iQ);
	decl vcomfac_u = mlfaobj.comfac_u(mY, iP, iQ, vtheta);
	decl vcomfac_s = mlfaobj.comfac_s(mY, iP, iQ, vtheta);
	delete mlfaobj;

	decl vSWI_u = vLRweight' * vmu_Y + vcomfac_u;
	decl vSWI_s = vLRweight' * vmu_Y + vcomfac_s;

	print(mweight);
	print(vLRweight);
	savemat("c:/Program Files/Ox/myprg_ox/00b/temp.xls", vSWI_u ~ vSWI_s);
}
