/*------------------------------------------*/
/* JAE - Data Archive - MS 9266 forthcoming */
/* James G. MacKinnon & Matthew D. Webb     */
/* Wild Bootstrap Inference for Wildy       */
/*    Different Cluster Sizes               */  
/*------------------------------------------*/

/*------------------------------------------------------------*/
/* This file will do one replication of the placebo laws      */
/* It generates P values using Cluster Robust standard        */
/* errors, OLS, and the wild cluster bootstrap.               */
/* The default is for 10 'treated' states, but this can be    */
/* altered in the code below.                                 */ 
/*------------------------------------------------------------*/

/*------------------------------------------*/
/* The replications themselves are based on */
/* the "Placebo Laws" experiments from      */
/* Bertrand, Duflo, and Mullainathan (2004) */ 
/* Quarterly Journal of Economics           */
/*------------------------------------------*/


/*Set the directory here if necessary*/
	global SOURCE "C:\wildly\"

	cd ${SOURCE}
	
	capture log close
	log using "${SOURCE}placebo-laws.txt", text replace
	
/*Set the number of bootstraps here*/
	global BS = 999

/*Set the number of replications here - default of 1*/
	global REPS = 1

/*Set the number of treated states here */
	global TREAT = 10

/*preamble */
	set more off
	clear all

	set mem 300m


/*set up matrix to store results*/
/* add different matrices for each p-value, across all treated */

	mata: pval_ols = J($REPS,50,.)
	mata: pval_crve = J($REPS,50,.)
	mata: pval_wild = J($REPS,50,.)

/*start of replication loop*/
forvalues r = 1/$REPS {

	local sd = 42*`r'
	set seed `sd'

	/* start of treated loop */
		/*alter the first treat to loop over the number of treated states*/
	forvalues t = $TREAT/$TREAT { 

		/*code to generate unique set of treated states */
			clear 
			set obs 51
			gen n = _n
			sample `t', count
			mkmat n, matrix(sts)

		use placebo-laws.dta, clear

		/*set the indicator variable = 1 for all ''treated'' states*/
		qui gen treat = 0

		forvalues i = 1/`t' {
			/*pick a year at random between 1985 and 1995*/
				local yrt = floor((1995-1985+1)*runiform() + 1985)
			/*change value of treatment for states in sample*/ 
				qui replace treat = 1 if year>=`yrt' & state==sts[`i',1]
		} /*end of i loop */


		/*demean the treatment variable*/
			qui egen treat_m = mean(treat), by(state)
			qui gen treat_dm = treat - treat_m
			drop treat_m

	
		/*run the regressions*/
		
			/*generate a set of year dummy variables*/
				qui tab year, gen(yrs)

		/*ols*/
			qui reg log_wage_dm  treat_dm age_dm age2_dm educ1_dm educ2_dm educ4_dm yrs* [pweight=weight]
			
			/*store the ols p-value*/
				local pval_ols = 2*min(  normal(_b[treat_dm]/_se[treat_dm]) , 1 - normal(_b[treat_dm]/_se[treat_dm]) )
				mata: pval_ols[`r',`t'] = `pval_ols'

		/*cluster*/
			qui reg log_wage_dm  treat_dm age_dm age2_dm educ1_dm educ2_dm educ4_dm yrs* [pweight=weight], cluster(state)
		
			/*store the crve p-value*/
				local pval_crve = 2*min(  normal(_b[treat_dm]/_se[treat_dm]) , 1 - normal(_b[treat_dm]/_se[treat_dm]) )
				mata: pval_crve[`r',`t'] = `pval_crve'

			/*store the t-statistics for the CRVE regression, for WCR*/
			global maint = (_b[treat_dm]/_se[treat_dm])


		*************************
		/*bootstrap*/
		*************************
		
		/*impose the null hypothesis, beta_treatment=0 */
			qui reg log_wage_dm age_dm age2_dm educ1_dm educ2_dm educ4_dm yrs* [pweight=weight]
			qui predict er, resid
			qui predict xbr, xb 

		/*bootstrap setup*/
			qui gen uni = . 
			qui gen ernew = . 
			qui gen pos = . 
			qui gen ywild = . 
			
		/* store bootstrap statistics */
		mata: bs_ts = J(${BS},1,.)
		 

		/*bootstrap loop*/
		forvalues b = 1/$BS {
		
			display "bootstrap number = " `b'

			sort state
			qui by state: replace uni = uniform() 
			/*cluster level rademacher indicator*/
			qui by state: replace pos = uni[1]<.5  
			
			/*transform residuals */
			qui replace ernew = (2*pos-1)*er  
			
			/*Wild Wald (impose the null) -- Rademacher */
			qui replace ywild = xbr + ernew  
			
			/* run the regression with bootstrap sample*/
				qui reg ywild treat_dm age_dm age2_dm educ1_dm educ2_dm educ4_dm yrs* [pweight=weight], cluster(state)
			/*store the t-statistic*/
				local t_stat = (_b[treat_dm])/(_se[treat_dm])
				mata: bs_ts[`b',1]= `t_stat'


		} /*end of bootstrap loop*/

		/*calculate bootstrap p-value*/
			cap drop n
			qui gen n =. 
			
			getmata (Ts_r)=bs_ts, force 
			sort Ts_r 
			
			qui replace n = _n 
			qui cap drop test
			qui gen test = (${maint}>=Ts_r)	
			qui summ n  if test ==1
			local myp = r(max) / ${BS}
			global pval_max_r = 2 * min(`myp',(1-`myp')) 
			
			/*store the p-value*/
				mata: pval_wild[`r',`t'] = ${pval_max_r}
			
		} /*end of treated loop*/	
		
		/*display p-values - comment out if desired */
			display "three p-values "
			display "OLS pvalue -->  " `pval_ols'
			display "CRVE pvalue -->  " `pval_crve'
			display "B.S. pvalue -->  " ${pval_max_r}
			
} /*end of rep loop*/

/*output results*/



/*if doing more than one replication, the p-values are stored */
/*  as mata matrices  */
/* remove the * below to retrieve the p-values*/

	/*ols*/
		*clear
		*getmata (pvals*) = pval_ols

	/*crve*/
		*clear
		*getmata (pvals*) = pval_crve

	/*wild*/
		*clear
		*getmata (pvals*) = pval_wild

	

