% ML_VARbias - Computes the Distortion in the VAR as in Kilian (1998)
%
% A2=ML_VARbias(u,A,det,ndraws,e,R);
%
%      Y - Variables on which the VAR is Estimated
%      u - residual from the VAR
%      A - Matrix Containing VAR Coefficients
%      p - Lag of the VAR 
%    det - Deterministic component in the VAR
% ndraws - number of draws for the bootstrap algorithm
%      e - structural shocks
%      R - Rotation Matrix
%
% Reference: Kilian, L. (1998). Small-sample confidence intervals for impulse response functions. 
%                               The Review of Economics and Statistics 80, 218230.
%

% Written by Matteo Luciani (matteo.luciani@ulb.ac.be)

function A2=ML_VARbias(Y,u,A,p,det,ndraws,e,R,PHI,nblk,r);

disp('Computing Bias Induced by VAR Estimation')
if nargin>8;
    data=2; [fhat,lambda,chat,xsihat]=ML_efactors2(Y,r,2);  
else 
    data=1;
end;


% -------------------------------------------------- %
% Computing the Distortion in the VAR, Kilian (1998) %
% -------------------------------------------------- %
n=max(size(u)); 
uhat=u-repmat(mean(u),n,1);                                               % Centered residual
for bb=1:ndraws;        
    if data ==1;
        Ystar=GenerateVAR(Y,A,e,det,R,n);
    elseif data == 2;
        Ystar=GenerateFactors(PHI,fhat,xsihat,nblk,lambda,A,e,det,n,R,r); 
    end
    [Ahat,ustar]=ML_VAR(Ystar,p,det);                                                   % Estimating the Law of Motion fo the Static Factors   
    psi(:,:,bb)=Ahat-A;                                                              % Bias in VAR Estimates
    clear  omstar xsistar fstar xstar fhatstar Ahatstar ustar                               % clear worthless objects
end

% -------------------------------------------------- %
% Applaying the Correction suggested in Kilian(1998) %
% -------------------------------------------------- %
PSI=mean(psi,3);  clear psi;                                                                % Average bias in VAR estimates  
A2=A-PSI;                                                                             % Correcting for the Bias in VAR estimates
Atilda=ML_VAR_companion_matrix(ML_VAR_polynomial(A2,det),det); m=1;    
while abs(max(eig(Atilda)))>=1; disp(abs(max(eig(Atilda))));                                % That the loop ensure that the VAR is stationary after having removed the bias            
    A2=A2*(1-m*0.01); m=m+1;                                                          % If it is not stationary modify the Dtilda matrix up to reaching stationarity
    Atilda=ML_VAR_companion_matrix(ML_VAR_polynomial(A2,det),det);
end;


%%% Generates Data from a VAR %%%
function Ystar=GenerateVAR(Y,A,e,det,R,n);
bootsam=ceil(n*rand(n,1));                                                              % reshuffling residuals    
Ystar=ML_decomposition(Y,A,e(bootsam,:),det,R); 

%%% Generates Factors from a Factor structure %%%
function Ystar=GenerateFactors(PHI,fhat,xsihat,nblk,lambda,A,e,det,n,R,r);
xstar = ML_boostrap_BB(PHI(:,:,1:21), xsihat, nblk,lambda,fhat,A,e,det,n,R);    %Generates data
xstar = ML_center(xstar)/diag(std(xstar));                                      % Standardize Data
Ystar = ML_efactors2(xstar,r,2);                                          % Estimating Static Factors with the normalization lambda'*lamnda/N=I
