%% ML_ConditionalForecasts - Conditional Forecast of the Common Factors
%
% SF - Structure containing starting values, and parameters, for Kalman Filter
%  y - Standardize Variables
%
%

% Written by Matteo Luciani
% matteo.luciani@.ulb.ac.be

function F=ML_ConditionalForecasts(y,SF)

[xitt,xittm,Ptt,Pttm]=SmokingFilter(SF.initx,SF.initV,y,SF.A,SF.C,SF.R,SF.Q); 	% Kalman Filter Estimates of the Factors
[xsmooth, ~, ~]=SmokingSmoother(SF.A,xitt,xittm,Ptt,Pttm,SF.C,SF.R);            % Kalman Smoother Estimates of the Factors
F =  xsmooth'/sqrt(size(y,2));                                                  % Static factors


%% SmokingFilter - Kalman Filter Estimation of a Factor Model
%
% This procedure was downloaded from Domenico Giannone's web page as part
% of the replication files of the paper Doz, C., Giannone, D., and Reichlin, L.
% "A Quasi Maximum Likelihood Approach for Large Approximate Dynamic Factor Models",
% Review of Economics and Statistics, forthcoming
% This codes were successively customized and commented 
% by Matteo Luciani (matteo.luciani@ulb.ac.be)
%

function [xitt,xittm,Ptt,Pttm,loglik]=SmokingFilter(initx,initV,x,A,CC,RR,Q)

T=size(x,1);                                                                % Number of Observations
r=size(A,1);                                                                % Number of states
xittm=[initx zeros(r,T)];                                                   % S_{t|t-1}
xitt=zeros(r,T);                                                            % S_{t|t} 
Pttm=zeros(r,r,T); Pttm(:,:,1)=initV;                                       % P_{t|t-1}
Ptt=zeros(r,r,T);                                                           % P_{t|t}
loglik=0;                                                                   % Initialize the log-likelihood
Y=x';                                                                       % transpose for convenience

for j=1:T;
    [y,C,R] = CutMissData(Y(:,j),CC,RR);                                    % Elimnate Missing Values

    %%% _____________ %%%
    %%% Updating Step %%%
    %%%  %%%   
    X=C*xittm(:,j);                                                         % X_{t|t-1} - Observables    
    H=C*Pttm(:,:,j)*C'+R;                                                   % H_{t|t-1} - Conditional Variance of the Observable   
    Hinv = inv(H);     
    e = y - X;                                                              % error (innovation)
    xitt(:,j)=xittm(:,j)+Pttm(:,:,j)*C'*Hinv*e;                             % S_{t|t}
    Ptt(:,:,j)=Pttm(:,:,j)-Pttm(:,:,j)*C'*Hinv*C*Pttm(:,:,j);               % P_{t|t}   

    %%% _______________ %%%
    %%% Prediction Step %%%
    %%%  %%%
    xittm(:,j+1)=A*xitt(:,j);                                               % S_{t|t-1} - States
    Pttm(:,:,j+1)=A*Ptt(:,:,j)*A'+Q;                                        % P_{t|t-1} - Conditional Variance of the State 
    loglik = loglik + 0.5*(log(det(H))  - e'*Hinv*e);                       % Log Likelihood   
    
end

%% SmokingSmoother - Kalman Smoother for a Factor Model
%
% This procedure was downloaded from Domenico Giannone's web page as part
% of the replication files of the paper Doz, C., Giannone, D., and Reichlin, L.
% "A Quasi Maximum Likelihood Approach for Large Approximate Dynamic Factor Models",
% Review of Economics and Statistics, forthcoming
% This codes were successively customized and commented 
% by Matteo Luciani (matteo.luciani@ulb.ac.be)
%
function [xitT,PtT,PtTm]=SmokingSmoother(A,xitt,xittm,Ptt,Pttm,C,R)

[T]=size(xitt,2);                                                           % Number of Observations
[N r]=size(C);                                                              % Number of Variables and Number of States
Pttm=Pttm(:,:,1:end-1);                                                     % P_{t|t-1}, remove the last observation because it has dimension T+1
xittm=xittm(:,1:end-1);                                                     % S_{t|t-1}, remove the last observation because it has dimension T+1
J=zeros(r,r,T);
xitT=[zeros(r,T-1)  xitt(:,T)];                                             % S_{t|T} 
PtT=cat(3,zeros(r,r,T-1),Ptt(:,:,T));                                       % P_{t|T} 
PtTm=zeros(r,r,T);                                                          % P_{t+1|T} 

for ii=1:T-1; J(:,:,ii)=Ptt(:,:,ii)*A'*inv(Pttm(:,:,ii+1)); end;

for jj = 1:T-1;    
    xitT(:,T-jj)= xitt(:,T-jj)+J(:,:,T-jj)*(xitT(:,T+1-jj)-xittm(:,T+1-jj));                    % S_{t|T} 
    PtT(:,:,T-jj)=Ptt(:,:,T-jj)+J(:,:,T-jj)*(PtT(:,:,T+1-jj)-Pttm(:,:,T+1-jj))*J(:,:,T-jj)';    % P_{t|T}         
end

L=zeros(N,N,T);
K=zeros(r,N,T);

for i=1:T
    L(:,:,i)=inv(C*Pttm(:,:,i)*C'+R);
    K(:,:,i)=Pttm(:,:,i)*C'*L(:,:,i);
end

PtTm(:,:,T)=(eye(r)-K(:,:,T)*C)*A*Ptt(:,:,T-1);                             % P_{t+1|T} 
for j = 1:T-2
    PtTm(:,:,T-j)=Ptt(:,:,T-j)*J(:,:,T-j-1)'+J(:,:,T-j)*(PtTm(:,:,T-j+1)-A*Ptt(:,:,T-j))*J(:,:,T-j-1)';
end

%% CutMissData - eliminates rows corresponding to missing in Y C and R
function [y,C,R]  = CutMissData(y,C,R)

jj = ~isnan(y);
y  =  y(jj,:);
C  =  C(jj,:);
R  =  R(jj,jj);
