#delim;
capture program drop plreg_rob;
program define plreg_rob, eclass;
        syntax varlist [if] [in],
               nlf(varname numeric)   							// name of the variable that enters non-linearly into the equation
               [
               GENerate(namelist max=1)        			// name of the new variable -- nonlinear function
               slope(namelist max=1)            		// name of the new variable -- first derivative of nonlinear funciton             
               grid(numlist min=1 max=1 int > 0) 		// specify grid to calculate the non-parametric approximation                             
               COLLinear
               UTrim(numlist min=1 max=1)   				// drop observations with y > utrim
               LTrim(numlist min=1 max=1)   				// drop observations with y < ltrim               
               Level(cilevel) 
               bs_grid(numlist min=3 max=3)					// grid for bootstrap low bound, up bound, npoints
							 *];

        display in yellow "   Partial linear regression model: Robinson's method";
       
        quietly {; // quietly

        if (!missing("`generate'")) confirm new variable `generate';
        if (!missing("`slope'"))    confirm new variable `slope'   ;

        marksample touse;

        tokenize `varlist'; local y `1'; macro shift; local x `*';
        unab x: `x';

        tempvar id id_org;
        generate `id_org' = _n;              // id to restore the order at the end, no if conditions
        generate `id'     = _n if (`touse'); // id for future merging

        preserve;

        markout `touse' `varlist' `nlf';

        keep if (`touse');

        // trimming observations
        if (!missing("`utrim'")) {;
                count if (`y'>`utrim');
                noi display in yellow r(N) " observations are deleted because of trimming from the top";
                drop if (`y'>`utrim');
                };
        if (!missing("`ltrim'")) {;
                count if (`y'<`ltrim');
                noi display in yellow r(N) " observations are deleted because of trimming from the bottom";
                drop if (`y'<`ltrim');
                };

				// reading parameters of bootstrap
				if (!missing("`bs_grid'")) {;
					local bs_lb: word 1 of `bs_grid';
					local bs_ub: word 2 of `bs_grid';
					local ngrid: word 3 of `bs_grid';

					if (`bs_lb'>=`bs_ub') {;
		  			noi display in red "Bootstrap lower bound must be smaller than upper bound";
		   		exit;
		   		};
	        tempname bs_xgrid delta;
					scalar `delta' = (`bs_ub'-`bs_lb')/(`ngrid'-1);
					generate `bs_xgrid' = `bs_lb'+(_n-1)*`delta' in 1/`ngrid';
	   		};


        foreach var of varlist `varlist' {; drop if (missing(`var')); }; // drop observations with missings
                                            drop if (missing(`nlf'));    // drop observations with missings

        /************ remove collinear variables ***************************/
        _rmcoll `x' if `touse', `collinear';      
        
        local result "`r(varlist)'";
        local colls: list x - result;

        if !missing("`colls'") {;
                noisily display as text "note: `colls' dropped due to collinearity";
                local x `result';
        }; //if
        /*******************************************************************/

        sort `nlf';
        tempname y_hat u_y;
	
				// define parameters for local polinomials depending on specified options
				if (!missing("`bs_grid'")) local at_n "at(`bs_xgrid')"; // bootstrap 
				if (!missing("`grid'"))    local at_n "n(`grid')";	      
	   
        if (!missing("`bs_grid'") | !missing("`grid'")) {;          
		   		tempvar y_grd x_coord;
	        locpoly1 `y' `nlf', `at_n' generate(`x_coord' `y_grd' ) nodraw `options';
	        m_ipolate `x_coord' `y_grd' `nlf' `y_hat';
	        drop `y_grd' `x_coord';
	      };
	      else {;
	        locpoly1 `y' `nlf', at(`nlf') generate(`y_hat') nodraw `options';
        };

        generate `u_y' = `y'-`y_hat'; drop `y_hat';

        foreach var of varlist `x' {;
	        tempvar `var'_hat u_`var';
          if("`: sortedby'"!="`nlf'") sort `nlf'; 
        	if (!missing("`bs_grid'") | !missing("`grid'")) {;                 
		   			tempvar `var'_grd x_`var'_c;                 
		   			locpoly1 `var' `nlf', `at_n' generate(`x_`var'_c' ``var'_grd' ) nodraw `options';
            m_ipolate `x_`var'_c' ``var'_grd'  `nlf' ``var'_hat';
            drop ``var'_grd' `x_`var'_c';
          };               
        	else {;
          	locpoly1 `var' `nlf', at(`nlf') generate(``var'_hat') nodraw `options';
        	};

          generate `u_`var'' = `var'-``var'_hat'; drop ``var'_hat';
          local xstr `xstr' `u_`var'';
        }; // end foreach     

        regress `u_y' `xstr', noconstant; estimates store reg1;
        
        if (!missing("`generate'")) {;
            tempvar  x_hat_b resid;
            tempname BETTAS;
            matrix `BETTAS' = e(b); matrix colname `BETTAS' = `x'; // here we use x's not u_x's !!!
            matrix score `x_hat_b' = `BETTAS';
            generate `resid'  = `y'-`x_hat_b';
               
	    	if("`: sortedby'"!="`nlf'") sort `nlf';   

            if (!missing("`bs_grid'") | !missing("`grid'")) {;

               tempvar r_grd x_r_c fd;
               locpoly1 `resid' `nlf', `at_n' generate(`x_r_c' `r_grd' ) slope(`fd') nodraw `options'; 
               m_ipolate `x_r_c' `r_grd' `nlf' `generate';

               tempname LIV;                                            // create matrix for LIV
               mkmat `x_r_c' `r_grd' `fd', matrix(`LIV') nomissing;
               matrix colname `LIV' = coord nfl nfl_slop;

               if(!missing("`slope'")) m_ipolate `x_r_c' `fd' `nlf' `slope';    	
	    			};	    
            else {;
               if (!missing("`slope'")) local slp slope(`slope');
               locpoly1 `resid' `nlf', at(`nlf') generate(`generate') `slp' nodraw `options';
            }; // end_if

            label variable `generate' "Estimated nonlinear function of `nlf'";
            if (!missing("`slope'")) label variable `slope' "Estimated FD of nonlinear `nlf'";

            tempfile tfile;
            keep `id' `generate' `slope';
            sort `id'; save `tfile', replace;
            restore;
            sort `id'; merge `id' using `tfile'; drop _merge; sort `id_org';
        }; // end if
        else restore;

        estimates restore reg1; // restore estimates from regression of u_y on u_x1,...u_xn
        }; // end quietly


        /********** Saved Results ******************************************/
        tempname COEFF; matrix `COEFF' = e(b); matrix colname `COEFF' = `x';
        tempname VCM  ; matrix `VCM'   = e(V); matrix colname `VCM'   = `x';
                                               matrix rowname `VCM'   = `x';
        local nobs = e(N);    local dof  = e(df_r); local F   = e(F);   local r2   = e(r2);
        local rmse = e(rmse); local mss  = e(mss);  local rss = e(rss); local r2_a = e(r2_a);
        local ll   = e(ll);   local model= e(model); local estat_cmd = e(estat_cmd);
        local cmd  = e(cmd);  local predict = e(predict);
        matrix b = `COEFF'; matrix V = `VCM';
        ereturn post b V, depname(`y') obs(`nobs') dof(`dof') esample(`touse');
        
	capture confirm matrix `LIV'; if (!_rc) ereturn matrix LIV =`LIV';

        ereturn scalar df_m   = `nobs'-`dof';
        ereturn scalar F      = `F';
        ereturn scalar r2     = `r2';
        ereturn scalar rmse   = `rmse';
        ereturn scalar mss    = `mss';
        ereturn scalar rss    = `rss';
        ereturn scalar r2_a   = `r2_a';
        ereturn scalar ll     = `ll';
        if (missing("`degree'")) ereturn scalar degree  = 1;
        else                     ereturn scalar degree  = `degree';

        ereturn local title     "Partial linear regression: Robinson";
        ereturn local depvar    `y';
        ereturn local model     `model';
        ereturn local estat_cmd `estat_cmd';
        ereturn local cmd       `cmd';
        if (!missing("`generate'")) ereturn local nlf `generate';
        else                        ereturn local nlf ;

        /************  OUTPUT TABLE ****************************************/
        display;
        display in green "      Source {c |}       SS       df       MS {col 55} Number of obs ="
                in yellow %8.0f e(N);
        display in green "{hline 13}{c +}{hline 30} {col 55} F(" e(df_m) ", " e(df_r) ")" _col(70) "="
                        _col(72) in yellow %7.0g e(F);
        display in green "       Model {c |}"
                in yellow _col(15) %12.0g e(mss) %6.0f e(df_m) _col(34) %11.0g e(mss)/e(df_m)
                in green "{col 55} Prob > f      ="
                in yellow %8.4f Ftail(e(df_m),e(df_r),e(F));
        display in green "    Residual {c |}"
                in yellow _col(15) %12.0g e(rss) %6.0f e(df_r) _col(34) %11.0g e(rss)/e(df_r)
                in green "{col 55} R-squared     =" in yellow %8.4f e(r2);
        display in green "{hline 13}{c +}{hline 30} {col 55} Adj R-squared ="
                in yellow %8.4f e(r2_a);
        display in green "       Total {c |}"
                in yellow %12.3f e(mss)+e(rss) %6.0f e(df_r)+e(df_m) _col(34) %11.0g (e(mss)+e(rss))/(e(df_r)+e(df_m))
                in green " {col 55} Root MSE      ="
                in yellow %8.4f e(rmse) _newline;

        ereturn display, level(`level'); // output table of coefficients
        display in green "Note: Standard errors are not adjusted for the presence of estimated regressors";

end; // program plreg_rob

/*******************************************************************/
/*******************************************************************/

capture program drop m_ipolate;
program define m_ipolate, rclass;
        args ipol_x ipol_y x yhat degree;
        generate double `yhat' = .;

        mata: ipolate("`ipol_x'","`ipol_y'","`x'","`yhat'",`degree');
end; // program m_ipolate

#delim cr

mata:

void function ipolate(string scalar x,
                      string scalar y,
                      string scalar z,
                      string scalar zz)
{
st_view(mx,.,x)
st_view(my,.,y)
st_view(mz,.,z)

st_store(.,zz,mm_ipolate(mx,my,mz,1))
}
end


